/*
 * arch/mips/vr4181a/common/ccu.c
 *
 * Clock Controll Unit routines for the NEC VR4181A.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2002-2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/types.h>

#include <asm/time.h>
#include <asm/vr4181a/vr4181a.h>

#define CLKSPEED	0xb000
#define DIV(x)		(((x) & 0x7000) >> 12)
#define CLKSP(x)	((x) & 0x001f)
#define CLKDIVCTRL	0xb002
#define PCICLKDIV	0x0300
#define ECUSYSCLKDIV	0x0030
#define PCLKDIV	0x0003
#define CMUCLKMSK0	0xb010
#define MSKSIU2_18M	0x0400
#define MSKSIU1_18M	0x0200
#define MSKSIU0_18M	0x0100
#define MSKCSUPCLK	0x0040
#define MSKAIUPCLK	0x0020
#define MSKPIUPCLK	0x0010
#define MSKADUPCLK	0x0008
#define MSKADU18M	0x0002
#define CMUCLKMSK1	0xb012
#define MSKI2S18M	0x4000
#define MSKUSBF48M	0x2000
#define MSKUSBH48M	0x1000
#define MSKI2CCLK	0x0100
#define MSKAC97CLK	0x0010
#define MSKPWM2CLK	0x0004
#define CMUCLKMSK2	0xc108
#define MSKPWM01CLK	0x0001
#define CMUCLKMSK3	0x0490
#define MSKAC97PCI	0x0020
#define MSKUSBFPCI	0x0010
#define MSKUSBHPCI	0x0008
#define MSKLCUCLK	0x0004
#define MSKIOPCICLK	0x0002
#define MSKDMACLK	0x0001

static u32 cmuclkmsk[4];

static inline void
write_cmuclkmsk(u16 reg, u32 mask)
{
	switch (reg) {
	case 0:
		vr4181a_writew((u16) mask, CMUCLKMSK0);
		break;
	case 1:
		vr4181a_writew((u16) mask, CMUCLKMSK1);
		break;
	case 2:
		vr4181a_writew((u16) mask, CMUCLKMSK2);
		break;
	case 3:
		vr4181a_writel(mask, CMUCLKMSK3);
		break;
	}

	vr4181a_write_fixed;
}

void
vr4181a_clock_supply(unsigned short clock)
{
	u32 mask;
	u16 regno;

	switch (clock) {
	case ADU18M_CLOCK:
		regno = 0;
		mask = MSKADU18M;
		break;
	case ADU_CLOCK:
		regno = 0;
		mask = MSKADUPCLK;
		break;
	case PIU_CLOCK:
		regno = 0;
		mask = MSKPIUPCLK;
		break;
	case AIU_CLOCK:
		regno = 0;
		mask = MSKAIUPCLK;
		break;
	case CSU_CLOCK:
		regno = 0;
		mask = MSKCSUPCLK;
		break;
	case SIU0_CLOCK:
		regno = 0;
		mask = MSKSIU0_18M;
		break;
	case SIU1_CLOCK:
		regno = 0;
		mask = MSKSIU1_18M;
		break;
	case SIU2_CLOCK:
		regno = 0;
		mask = MSKSIU2_18M;
		break;
	case PWM2_CLOCK:
		regno = 1;
		mask = MSKPWM2CLK;
		break;
	case AC97_CLOCK:
		regno = 1;
		mask = MSKAC97CLK;
		break;
	case I2C_CLOCK:
		regno = 1;
		mask = MSKI2CCLK;
		break;
	case USB_HOST_CLOCK:
		regno = 1;
		mask = MSKUSBH48M;
		break;
	case USB_FUNCTION_CLOCK:
		regno = 1;
		mask = MSKUSBF48M;
		break;
	case I2S_CLOCK:
		regno = 1;
		mask = MSKI2S18M;
		break;
	case PWM01_CLOCK:
		regno = 2;
		mask = MSKPWM01CLK;
		break;
	case DMA_CLOCK:
		regno = 3;
		mask = MSKDMACLK;
		break;
	case PCI_BRIDGE_CLOCK:
		regno = 3;
		mask = MSKIOPCICLK;
		break;
	case LCD_CLOCK:
		regno = 3;
		mask = MSKLCUCLK;
		break;
	case USB_HOST_PCI_CLOCK:
		regno = 3;
		mask = MSKUSBHPCI;
		break;
	case USB_FUNCTION_PCI_CLOCK:
		regno = 3;
		mask = MSKUSBFPCI;
		break;
	case AC97_PCI_CLOCK:
		regno = 3;
		mask = MSKAC97PCI;
		break;
	default:
		regno = 0;
		mask = 0;
		break;
	}

	cmuclkmsk[regno] |= mask;
	write_cmuclkmsk(regno, cmuclkmsk[regno]);
}

void
vr4181a_clock_mask(unsigned short clock)
{
	u32 mask;
	u16 regno;

	switch (clock) {
	case ADU18M_CLOCK:
		regno = 0;
		mask = MSKADU18M;
		break;
	case ADU_CLOCK:
		regno = 0;
		mask = MSKADUPCLK;
		break;
	case PIU_CLOCK:
		regno = 0;
		mask = MSKPIUPCLK;
		break;
	case AIU_CLOCK:
		regno = 0;
		mask = MSKAIUPCLK;
		break;
	case CSU_CLOCK:
		regno = 0;
		mask = MSKCSUPCLK;
		break;
	case SIU0_CLOCK:
		regno = 0;
		mask = MSKSIU0_18M;
		break;
	case SIU1_CLOCK:
		regno = 0;
		mask = MSKSIU1_18M;
		break;
	case SIU2_CLOCK:
		regno = 0;
		mask = MSKSIU2_18M;
		break;
	case PWM2_CLOCK:
		regno = 1;
		mask = MSKPWM2CLK;
		break;
	case AC97_CLOCK:
		regno = 1;
		mask = MSKAC97CLK;
		break;
	case I2C_CLOCK:
		regno = 1;
		mask = MSKI2CCLK;
		break;
	case USB_HOST_CLOCK:
		regno = 1;
		mask = MSKUSBH48M;
		break;
	case USB_FUNCTION_CLOCK:
		regno = 1;
		mask = MSKUSBF48M;
		break;
	case I2S_CLOCK:
		regno = 1;
		mask = MSKI2S18M;
		break;
	case PWM01_CLOCK:
		regno = 2;
		mask = MSKPWM01CLK;
		break;
	case DMA_CLOCK:
		regno = 3;
		mask = MSKDMACLK;
		break;
	case PCI_BRIDGE_CLOCK:
		regno = 3;
		mask = MSKIOPCICLK;
		break;
	case LCD_CLOCK:
		regno = 3;
		mask = MSKLCUCLK;
		break;
	case USB_HOST_PCI_CLOCK:
		regno = 3;
		mask = MSKUSBHPCI;
		break;
	case USB_FUNCTION_PCI_CLOCK:
		regno = 3;
		mask = MSKUSBFPCI;
		break;
	case AC97_PCI_CLOCK:
		regno = 3;
		mask = MSKAC97PCI;
		break;
	default:
		regno = 0;
		mask = 0;
		break;
	}

	cmuclkmsk[regno] &= ~mask;
	write_cmuclkmsk(regno, cmuclkmsk[regno]);
}

static inline unsigned long
get_aclock_frequency(void)
{
	u16 clkspeed, clksp;

	clkspeed = vr4181a_readw(CLKSPEED);
	clksp = CLKSP(clkspeed);

	switch (clksp) {
	case 8:
		printk(KERN_ERR "CCU: AClock set up Illegal value\n");
		return 147500000;	/* 147.5MHz */
	case 9:
		return 131100000;	/* 131.1MHz */
	case 10:
		return 118000000;	/* 118.0MHz */
	case 12:
		return 98300000;	/* 98.3MHz */
	case 13:
		return 90700000;	/* 90.7MHz */
	case 14:
		return 84100000;	/* 84.1MHz */
	case 15:
		return 78600000;	/* 78.6MHz */
	case 16:
		return 73700000;	/* 73.7MHz */
	}

	return 0;
}

static inline unsigned long
get_tclock_frequency(unsigned long aclock)
{
	u16 clkspeed, div;

	clkspeed = vr4181a_readw(CLKSPEED);
	div = DIV(clkspeed);

	switch (div) {
	case 1:
		return aclock;
	case 2:
		return aclock / 2;
	case 3:
		return aclock / 3;
	case 4:
		return aclock / 4;
	}

	return 0;
}

static inline unsigned long
set_pciclock_frequency(unsigned long aclock)
{
	unsigned long clock;
	u16 clkdivctrl;

	clkdivctrl = vr4181a_readw(CLKDIVCTRL) & ~PCICLKDIV;

	if (aclock > 132000000) {	/* more than 33MHz x 4 */
		printk(KERN_ERR "CCU: PCI Clock is more than 33MHz\n");
		clock = aclock / 4;
		clkdivctrl |= 0x0300;
	} else if (aclock > 99000000) {	/* more than 33MHz x 3 */
		clock = aclock / 4;
		clkdivctrl |= 0x0300;
	} else if (aclock > 66000000) {	/* more than 33MHz x 2 */
		clock = aclock / 3;
		clkdivctrl |= 0x0200;
	} else if (aclock > 33000000) {	/* more than 33MHz x 1 */
		printk(KERN_ERR "CCU: PCI Clock divide set up Illegal value\n");
		clock = aclock / 2;
		clkdivctrl |= 0x0100;
	} else {
		printk(KERN_ERR "CCU: PCI Clock divide set up Illegal value\n");
		clock = aclock;
		clkdivctrl |= 0x0000;
	}

	vr4181a_writew(clkdivctrl, CLKDIVCTRL);
	vr4181a_write_fixed;

	return clock;
}

static inline unsigned long
set_ecu_sysclock_frequency(unsigned long tclock, unsigned long ecu_sysclock_max)
{
	unsigned long clock;
	u16 clkdivctrl;

	clkdivctrl = vr4181a_readw(CLKDIVCTRL) & ~ECUSYSCLKDIV;
	if (ecu_sysclock_max > tclock) {
		clock = tclock;
		clkdivctrl |= 0x0000;
	} else if (ecu_sysclock_max > tclock / 2) {
		clock = tclock / 2;
		clkdivctrl |= 0x0010;
	} else if (ecu_sysclock_max > tclock / 4) {
		clock = tclock / 4;
		clkdivctrl |= 0x0020;
	} else {
		printk(KERN_ERR
		       "CCU: ECU_SysClock divide set up Illegal value\n");
		clock = tclock / 8;
		clkdivctrl |= 0x0030;
	}

	vr4181a_writew(clkdivctrl, CLKDIVCTRL);
	vr4181a_write_fixed;

	return clock;
}

static inline unsigned long
set_pclock_frequency(unsigned long tclock)
{
	unsigned long clock;
	u16 clkdivctrl;

	clkdivctrl = vr4181a_readw(CLKDIVCTRL) & ~PCLKDIV;

	if (tclock > 132000000) {
		printk(KERN_ERR "CCU: TClock divide set up Illegal value\n");
		clock = tclock / 8;
		clkdivctrl |= 0x0003;
	} else if (tclock > 74000000) {
		clock = tclock / 4;
		clkdivctrl |= 0x0002;
	} else if (tclock > 37000000) {
		clock = tclock / 2;
		clkdivctrl |= 0x0001;
	} else {
		clock = tclock;
		clkdivctrl |= 0x0000;
	}

	vr4181a_writew(clkdivctrl, CLKDIVCTRL);
	vr4181a_write_fixed;

	return clock;
}

static unsigned long tclock, ecu_sysclock;

void __init
vr4181a_ccu_init(unsigned long ecu_sysclock_max)
{
	unsigned long aclock, pciclock, ecu_sysclock, pclock;

	aclock = get_aclock_frequency();

	tclock = get_tclock_frequency(aclock);
	pciclock = set_pciclock_frequency(aclock);

	ecu_sysclock = set_ecu_sysclock_frequency(tclock, ecu_sysclock_max);
	pclock = set_pclock_frequency(tclock);

	mips_counter_frequency = tclock / 4;

	cmuclkmsk[0] = vr4181a_readw(CMUCLKMSK0);
	cmuclkmsk[1] = vr4181a_readw(CMUCLKMSK1);
	cmuclkmsk[2] = vr4181a_readw(CMUCLKMSK2);
	cmuclkmsk[3] = vr4181a_readl(CMUCLKMSK3);

	printk(KERN_INFO "CPU core clock: %ldHz\n", aclock);
	printk(KERN_INFO "Internal System Bus clock: %ldHz\n", tclock);
	printk(KERN_INFO "Internal PCI Bus clock: %ldHz\n", pciclock);
	printk(KERN_INFO "ECU Interface clock: %ldHz\n", ecu_sysclock);
	printk(KERN_INFO "Internal ISA Bus clock: %ldHz\n", pclock);
}

unsigned long
vr4181a_get_tclock(void)
{
	return tclock;
}

unsigned long
vr4181a_get_ecu_sysclock(void)
{
	return ecu_sysclock;
}
