/*
 * arch/mips/vr4181a/common/bcu.c
 *
 * Bus Control Unit routines for the NEC VR4181A.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2002-2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/types.h>

#include <asm/vr4181a/vr4181a.h>

/*
 * HBU
 */
#define SDRAM		0x0000
#define SFLASH		0x0008
#define PCS0		0x0010
#define PCS1		0x0018
#define PCS2		0x0020
#define PCS3		0x0028
#define PCS4		0x0030
#define ISAW		0x0038
#define PCIW0		0x0060
#define PCIW1		0x0068
#define INTCS		0x0070
#define ROMCS		0x0078
#define CPUSTAT		0x0480
#define TBUSCTRL	0x0488

#define WIDTH_32BIT	0x0080
#define WIDTH_16BIT	0x0040
#define WIDTH_8BIT	0x0000

#define VISPCI		0x0020

#define MASK_2M		0x000f
#define MASK_4M		0x000e
#define MASK_8M		0x000d
#define MASK_16M	0x000c
#define MASK_32M	0x000b
#define MASK_64M	0x000a
#define MASK_128M	0x0009
#define MASK_256M	0x0008
#define MASK_512M	0x0007
#define MASK_1024M	0x0006
#define MASK_2048M	0x0005

/*
 * EXIBU
 */
#define EXIBUCFG	0x0100
#define PCS0TIM		0x0110
#define PCS1TIM		0x0118
#define PCS2TIM		0x0120
#define PCS3TIM		0x0128
#define PCS4TIM		0x0130
#define ISATIM		0x0138
#define EXBERRADR	0x0150
#define EXBERRCS	0x0160
#define EXBBTMODE	0x0170
#define ROMCSTIM	0x0178

unsigned long vr4181a_internal_registers_base = -1;

EXPORT_SYMBOL(vr4181a_internal_registers_base);

static u32 __init
hbu_register_value(struct vr4181a_address_window *win)
{
	u32 value;

	value = win->base_address & 0xffe00000;

	switch (win->data_width) {
	case DATA_WIDTH_8BIT:
		value |= WIDTH_8BIT;
		break;
	case DATA_WIDTH_16BIT:
		value |= WIDTH_16BIT;
		break;
	case DATA_WIDTH_32BIT:
		value |= WIDTH_32BIT;
		break;
	}

	if (win->pci_master_access == PCI_MASTER_ACCESS_ENABLE)
		value |= VISPCI;

	switch (win->size) {
	case 0x200000:
		value |= MASK_2M;
		break;
	case 0x400000:
		value |= MASK_4M;
		break;
	case 0x800000:
		value |= MASK_8M;
		break;
	case 0x1000000:
		value |= MASK_16M;
		break;
	case 0x2000000:
		value |= MASK_32M;
		break;
	case 0x4000000:
		value |= MASK_64M;
		break;
	case 0x8000000:
		value |= MASK_128M;
		break;
	case 0x10000000:
		value |= MASK_256M;
		break;
	case 0x20000000:
		value |= MASK_512M;
		break;
	case 0x40000000:
		value |= MASK_1024M;
		break;
	case 0x80000000:
		value |= MASK_2048M;
		break;
	}

	return value;
}

void __init
vr4181a_hbu_init(struct vr4181a_address_maps *maps)
{
	u32 value;

	value = hbu_register_value(&maps->sync_flash_memory);
	vr4181a_writel(value, SFLASH);

	value = hbu_register_value(&maps->pcs0);
	vr4181a_writel(value, PCS0);

	value = hbu_register_value(&maps->pcs1);
	vr4181a_writel(value, PCS1);

	value = hbu_register_value(&maps->pcs2);
	vr4181a_writel(value, PCS2);

	value = hbu_register_value(&maps->pcs3);
	vr4181a_writel(value, PCS3);

	value = hbu_register_value(&maps->pcs4);
	vr4181a_writel(value, PCS4);

	vr4181a_ecu_set_window(&maps->isa_bus);
	value = hbu_register_value(&maps->isa_bus);
	vr4181a_writel(value, ISAW);

	value = hbu_register_value(&maps->pci_bus0);
	vr4181a_writel(value, PCIW0);

	value = hbu_register_value(&maps->pci_bus1);
	vr4181a_writel(value, PCIW1);

	value = hbu_register_value(&maps->rom);
	vr4181a_writel(value, ROMCS);
}

static inline unsigned long
set_lcock_frequency(unsigned long tclock, unsigned long lclock_max)
{
	unsigned long clock;
	u32 exibucfg;

	exibucfg = vr4181a_readl(EXIBUCFG);
	exibucfg &= ~0x00000030;

	if (lclock_max > tclock) {
		printk(KERN_ERR "EXIBU: LClock divide set up Illegal value\n");
		clock = tclock;
		exibucfg |= 0x00000030;
	} else if (lclock_max > tclock / 2) {
		clock = tclock / 2;
		exibucfg |= 0x00000010;
	} else if (lclock_max > tclock / 3) {
		clock = tclock / 3;
		exibucfg |= 0x00000020;
	} else {
		clock = tclock / 4;
		exibucfg |= 0x00000000;
	}

	vr4181a_writel(exibucfg, EXIBUCFG);

	return clock;
}

static u32 __init
exibu_register_value(struct vr4181a_access_config *config)
{
	u32 value = 0;

	if (config->config_active == 0)
		return value;

	switch (config->io_signal_setup_time) {
	case 2:
		value |= 0x40000000;
		break;
	case 4:
		value |= 0x80000000;
		break;
	case 8:
		value |= 0xc0000000;
		break;
	}

	switch (config->page_access_length) {
	case 4:
		value |= 0x00000000;
		break;
	case 8:
		value |= 0x10000000;
		break;
	case 16:
		value |= 0x20000000;
		break;
	case 32:
		value |= 0x30000000;
		break;
	}

	switch (config->control_signal_active_level) {
	case ACTIVE_LOW:
		value |= 0x00000000;
		break;
	case ACTIVE_HIGH:
		value |= 0x08000000;
		break;
	}

	switch (config->chip_select_active_level) {
	case ACTIVE_LOW:
		value |= 0x00000000;
		break;
	case ACTIVE_HIGH:
		value |= 0x04000000;
		break;
	}

	switch (config->control_signal_deassert_timing) {
	case 0:
		value |= 0x00000000;
		break;
	case 1:
		value |= 0x01000000;
		break;
	case 2:
		value |= 0x02000000;
		break;
	case 3:
		value |= 0x03000000;
		break;
	}

	if (config->ready_mode == READY_USE) {
		switch (config->iordy_sync_times) {
		case LATCH_1_TIME:
			value |= 0x00000000;
			break;
		case LATCH_2_TIMES:
			value |= 0x00800000;
			break;
		}
	}

	switch (config->ready_mode) {
	case READY_UNUSE:
		value |= 0x00000000;
		break;
	case READY_USE:
		value |= 0x00400000;
		break;
	}

	switch (config->bus_idel_time) {
	case 0:
		value |= 0x00000000;
		break;
	case 1:
		value |= 0x00080000;
		break;
	case 2:
		value |= 0x00100000;
		break;
	case 3:
		value |= 0x00180000;
		break;
	case 4:
		value |= 0x00200000;
		break;
	case 5:
		value |= 0x00280000;
		break;
	case 6:
		value |= 0x00300000;
		break;
	case 7:
		value |= 0x00380000;
		break;
	}

	if (config->ready_mode == READY_USE) {
		switch (config->control_signal_width_min) {
		case 2:
			value |= 0x00000000;
			break;
		case 4:
			value |= 0x00020000;
			break;
		case 8:
			value |= 0x00040000;
			break;
		case 16:
			value |= 0x00060000;
			break;
		}
	}

	switch (config->control_signal_hold_time) {
	case 0:
		value |= 0x00000000;
		break;
	case 1:
		value |= 0x00008000;
		break;
	case 2:
		value |= 0x00010000;
		break;
	case 3:
		value |= 0x00018000;
		break;
	}

	if (config->ready_mode == READY_UNUSE) {
		value |=
		    (u32) (config->control_signal_assert_time_page_access -
			   1) << 9;
		value |= (u32) (config->control_signal_assert_time - 1) << 3;
	} else {
		value |=
		    (u32) ((config->ready_signal_timeout_counter -
			    1) / 256) << 3;
	}

	switch (config->control_signal_setup_time) {
	case 0:
		value |= 0x00000000;
		break;
	case 1:
		value |= 0x00000002;
		break;
	case 2:
		value |= 0x00000004;
		break;
	case 3:
		value |= 0x00000006;
		break;
	}

	return value;
}

void __init
vr4181a_exibu_init(struct vr4181a_access_setup *setup, unsigned long lclock_max)
{
	unsigned long lclock;
	u32 value;

	lclock = set_lcock_frequency(vr4181a_get_tclock(), lclock_max);

	printk(KERN_INFO "External Bus clock: %ldHz\n", lclock);

	value = exibu_register_value(&setup->pcs0);
	vr4181a_writel(value, PCS0TIM);

	value = exibu_register_value(&setup->pcs1);
	vr4181a_writel(value, PCS1TIM);

	value = exibu_register_value(&setup->pcs2);
	vr4181a_writel(value, PCS2TIM);

	value = exibu_register_value(&setup->pcs3);
	vr4181a_writel(value, PCS3TIM);

	value = exibu_register_value(&setup->pcs4);
	vr4181a_writel(value, PCS4TIM);

	value = exibu_register_value(&setup->isa_bus);
	vr4181a_writel(value, ISATIM);

	value = exibu_register_value(&setup->rom);
	vr4181a_writel(value, ROMCSTIM);
}
