/*
 * arch/mips/tx4938/toshiba_rbtx4938/pci_ops.c
 *
 *     Define the pci_ops for the Toshiba rbtx4938
 *
 * Author: source@mvista.com
 *
 * Copyright 2001-2002 MontaVista Software Inc.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation 
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under the
 * terms of the GNU General Public License version 2. This program is
 * licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/config.h>

#include <asm/addrspace.h>
#include <asm/pci_channel.h>
#include <asm/tx4938/rbtx4938.h>
#include <asm/debug.h>

#ifdef CONFIG_PCI

/* initialize in setup */
struct resource pci_io_resource = {
	"pci IO space",
	0,
	0,
	IORESOURCE_IO
};

/* initialize in setup */
struct resource pci_mem_resource = {
	"pci memory space",
	0,
	0,
	IORESOURCE_MEM
};

struct resource tx4938_pcic1_pci_io_resource = {
       	"PCI1 IO",
       	0,
       	0,
       	IORESOURCE_IO
};
struct resource tx4938_pcic1_pci_mem_resource = {
       	"PCI1 mem",
       	0,
       	0,
       	IORESOURCE_MEM
};

extern struct pci_ops tx4938_pci_ops;

struct pci_channel mips_pci_channels[] = {
	/* h/w only supports devices 0x00 to 0x14 */
	{&tx4938_pci_ops, &pci_io_resource, &pci_mem_resource,
	 PCI_DEVFN(0x00, 0), PCI_DEVFN(0x14, 7)},
	/* h/w only supports devices 0x00 to 0x14 */
	{&tx4938_pci_ops, &tx4938_pcic1_pci_io_resource, &tx4938_pcic1_pci_mem_resource,
	 PCI_DEVFN(0x00, 0), PCI_DEVFN(0x14, 7)},
	{(struct pci_ops *) NULL, (struct resource *) NULL,
	 (struct resource *) NULL}
};

unsigned int pcibios_assign_all_busses(void)
{
	return 1;
}

extern struct tx4938_pcic_reg *pcicptrs[4];
static inline struct tx4938_pcic_reg *pci_dev_to_pcicptr(struct pci_dev *dev)
{
	struct pci_channel *channel = (struct pci_channel *)dev->sysdata;
		return pcicptrs[channel - &mips_pci_channels[0]];
}  

static int
mkaddr(unsigned char bus, unsigned char dev_fn, unsigned char where,
       int *flagsp,struct tx4938_pcic_reg *pcicptr)  
{
	if (bus > 0) {
		/* Type 1 configuration */
		pcicptr->g2pcfgadrs = ((bus & 0xff) << 0x10) |
		    ((dev_fn & 0xff) << 0x08) | (where & 0xfc) | 1;
	} else {
		if (dev_fn >= PCI_DEVFN(TX4938_PCIC_MAX_DEVNU, 0))
			return -1;

		/* Type 0 configuration */
		pcicptr->g2pcfgadrs = ((bus & 0xff) << 0x10) |
		    ((dev_fn & 0xff) << 0x08) | (where & 0xfc);
	}
	/* clear M_ABORT and Disable M_ABORT Int. */
	pcicptr->pcistatus =
	    (pcicptr->pcistatus & 0x0000ffff) |
	    (PCI_STATUS_REC_MASTER_ABORT << 16);
	pcicptr->pcimask &= ~PCI_STATUS_REC_MASTER_ABORT;
	return 0;
}

static int check_abort(int flags, struct tx4938_pcic_reg *pcicptr)
{
	int code = PCIBIOS_SUCCESSFUL;
	/* wait write cycle completion before checking error status */
	while (pcicptr->pcicstatus & TX4938_PCIC_PCICSTATUS_IWB)
				;         
	if (pcicptr->pcistatus & (PCI_STATUS_REC_MASTER_ABORT << 16)) {
		pcicptr->pcistatus =
		    (pcicptr->
		     pcistatus & 0x0000ffff) | (PCI_STATUS_REC_MASTER_ABORT
						<< 16);
		pcicptr->pcimask |= PCI_STATUS_REC_MASTER_ABORT;
		code = PCIBIOS_DEVICE_NOT_FOUND;
	}
	return code;
}

/*
 * We can't address 8 and 16 bit words directly.  Instead we have to
 * read/write a 32bit word and mask/modify the data we actually want.
 */
static int tx4938_pcibios_read_config_byte(struct pci_dev *dev,
					   int where, unsigned char *val)
{
	int flags, retval;
	unsigned char bus, func_num;
	struct tx4938_pcic_reg *pcicptr = pci_dev_to_pcicptr(dev); 

	db_assert((where & 3) == 0);
	db_assert(where < (1 << 8));

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags, pcicptr))
		return -1;
#ifdef __BIG_ENDIAN
	*val =
	    *(volatile u8 *) ((ulong) & pcicptr->
			      g2pcfgdata | ((where & 3) ^ 3));
#else
	*val =
	    *(volatile u8 *) ((ulong) & pcicptr->
			      g2pcfgdata | (where & 3));
#endif
	retval = check_abort(flags, pcicptr);
	if (retval == PCIBIOS_DEVICE_NOT_FOUND)
		*val = 0xff;
//printk("CFG R1 0x%02x 0x%02x 0x%08x\n", dev->devfn, where, *val );
	return retval;
}

static int tx4938_pcibios_read_config_word(struct pci_dev *dev,
					   int where, unsigned short *val)
{
	int flags, retval;
	unsigned char bus, func_num;
	struct tx4938_pcic_reg *pcicptr = pci_dev_to_pcicptr(dev); 

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;


	db_assert((where & 3) == 0);
	db_assert(where < (1 << 8));

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags, pcicptr))
		return -1;
#ifdef __BIG_ENDIAN
	*val =
	    *(volatile u16 *) ((ulong) & pcicptr->
			       g2pcfgdata | ((where & 3) ^ 2));
#else
	*val =
	    *(volatile u16 *) ((ulong) & pcicptr->
			       g2pcfgdata | (where & 3));
#endif
	retval = check_abort(flags, pcicptr);
	if (retval == PCIBIOS_DEVICE_NOT_FOUND)
		*val = 0xffff;
//printk("CFG R2 0x%02x 0x%02x 0x%08x\n", dev->devfn, where, *val );
	return retval;
}

static int tx4938_pcibios_read_config_dword(struct pci_dev *dev,
					    int where, unsigned int *val)
{
	int flags, retval;
	unsigned char bus, func_num;
	struct tx4938_pcic_reg *pcicptr = pci_dev_to_pcicptr(dev); 

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	db_assert((where & 3) == 0);
	db_assert(where < (1 << 8));

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags, pcicptr))
		return -1;
	*val = pcicptr->g2pcfgdata;
	retval = check_abort(flags, pcicptr);
	if (retval == PCIBIOS_DEVICE_NOT_FOUND)
		*val = 0xffffffff;

//printk("CFG R4 0x%02x 0x%02x 0x%08x\n", dev->devfn, where, *val );
	return retval;
}

static int tx4938_pcibios_write_config_byte(struct pci_dev *dev,
					    int where, unsigned char val)
{
	int flags;
	unsigned char bus, func_num;
	struct tx4938_pcic_reg *pcicptr = pci_dev_to_pcicptr(dev); 

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags, pcicptr))
		return -1;
#ifdef __BIG_ENDIAN
	*(volatile u8 *) ((ulong) & pcicptr->
			  g2pcfgdata | ((where & 3) ^ 3)) = val;
#else
	*(volatile u8 *) ((ulong) & pcicptr->
			  g2pcfgdata | (where & 3)) = val;
#endif
//printk("CFG W1 0x%02x 0x%02x 0x%08x\n", dev->devfn, where, val );
	return check_abort(flags, pcicptr);
}

static int tx4938_pcibios_write_config_word(struct pci_dev *dev,
					    int where, unsigned short val)
{
	int flags;
	unsigned char bus, func_num;
	struct tx4938_pcic_reg *pcicptr = pci_dev_to_pcicptr(dev); 

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags, pcicptr))
		return -1;
#ifdef __BIG_ENDIAN
	*(volatile u16 *) ((ulong) & pcicptr->
			   g2pcfgdata | ((where & 3) ^ 2)) = val;
#else
	*(volatile u16 *) ((ulong) & pcicptr->
			   g2pcfgdata | (where & 3)) = val;
#endif
//printk("CFG W2 0x%02x 0x%02x 0x%08x\n", dev->devfn, where, val );
	return check_abort(flags, pcicptr);
}

static int tx4938_pcibios_write_config_dword(struct pci_dev *dev,
					     int where, unsigned int val)
{
	int flags;
	unsigned char bus, func_num;
	struct tx4938_pcic_reg *pcicptr = pci_dev_to_pcicptr(dev); 

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags, pcicptr))
		return -1;
	pcicptr->g2pcfgdata = val;
//printk("CFG W4 0x%02x 0x%02x 0x%08x\n", dev->devfn, where, val );
	return check_abort(flags, pcicptr);
}

struct pci_ops tx4938_pci_ops = {
	tx4938_pcibios_read_config_byte,
	tx4938_pcibios_read_config_word,
	tx4938_pcibios_read_config_dword,
	tx4938_pcibios_write_config_byte,
	tx4938_pcibios_write_config_word,
	tx4938_pcibios_write_config_dword
};
#endif				/* CONFIG_PCI */
