/*
 * linux/arch/mips/tx4938/toshiba_rbtx4938/irq.c
 *
 * Toshiba RBTX4938 specific interrupt handlers
 *
 * Author: source@mvista.com
 *
 * Copyright 2001-2002 MontaVista Software Inc.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation 
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under the
 * terms of the GNU General Public License version 2. This program is
 * licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

/*
IRQ  Device  

16   TX4938-CP0/00 Software 0
17   TX4938-CP0/01 Software 1
18   TX4938-CP0/02 Cascade TX4938-CP0
19   TX4938-CP0/03 Multiplexed -- do not use 
20   TX4938-CP0/04 Multiplexed -- do not use 
21   TX4938-CP0/05 Multiplexed -- do not use 
22   TX4938-CP0/06 Multiplexed -- do not use 
23   TX4938-CP0/07 CPU TIMER

24   TX4938-PIC/00 
25   TX4938-PIC/01 
26   TX4938-PIC/02 Cascade RBTX4938-IOC
27   TX4938-PIC/03 RBTX4938 RTL-8019AS Ethernet 
28   TX4938-PIC/04 
29   TX4938-PIC/05 TX4938 ETH1
30   TX4938-PIC/06 TX4938 ETH0
31   TX4938-PIC/07 
32   TX4938-PIC/08 TX4938 SIO 0
33   TX4938-PIC/09 TX4938 SIO 1
34   TX4938-PIC/10 TX4938 DMA0
35   TX4938-PIC/11 TX4938 DMA1
36   TX4938-PIC/12 TX4938 DMA2
37   TX4938-PIC/13 TX4938 DMA3
38   TX4938-PIC/14
39   TX4938-PIC/15
40   TX4938-PIC/16 TX4938 PCIC
41   TX4938-PIC/17 TX4938 TMR0
42   TX4938-PIC/18 TX4938 TMR1
43   TX4938-PIC/19 TX4938 TMR2
44   TX4938-PIC/20
45   TX4938-PIC/21
46   TX4938-PIC/22 TX4938 PCIERR
47   TX4938-PIC/23
48   TX4938-PIC/24
49   TX4938-PIC/25
50   TX4938-PIC/26
51   TX4938-PIC/27
52   TX4938-PIC/28
53   TX4938-PIC/29
54   TX4938-PIC/30
55   TX4938-PIC/31 TX4938 SPI

56 RBTX4938-IOC/00 PCI-D
57 RBTX4938-IOC/01 PCI-C
58 RBTX4938-IOC/02 PCI-B
59 RBTX4938-IOC/03 PCI-A
60 RBTX4938-IOC/04 RTC
61 RBTX4938-IOC/05 ATA
62 RBTX4938-IOC/06 MODEM
63 RBTX4938-IOC/07 SWINT
*/

#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/swap.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/timex.h>
#include <asm/bootinfo.h>
#include <asm/page.h>
#include <asm/bootinfo.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/pci.h>
#include <asm/processor.h>
#include <asm/ptrace.h>
#include <asm/reboot.h>
#include <asm/time.h>
#include <linux/version.h>
#include <linux/bootmem.h>
#include <linux/blk.h>
#include <asm/tx4938/rbtx4938.h>

#undef TOSHIBA_RBTX4938_IRQ_DEBUG

#ifdef TOSHIBA_RBTX4938_IRQ_DEBUG
#define TOSHIBA_RBTX4938_IRQ_NONE        0x00000000

#define TOSHIBA_RBTX4938_IRQ_INFO          ( 1 <<  0 )
#define TOSHIBA_RBTX4938_IRQ_WARN          ( 1 <<  1 )
#define TOSHIBA_RBTX4938_IRQ_EROR          ( 1 <<  2 )

#define TOSHIBA_RBTX4938_IRQ_IOC_INIT      ( 1 << 10 )
#define TOSHIBA_RBTX4938_IRQ_IOC_STARTUP   ( 1 << 11 )
#define TOSHIBA_RBTX4938_IRQ_IOC_SHUTDOWN  ( 1 << 12 )
#define TOSHIBA_RBTX4938_IRQ_IOC_ENABLE    ( 1 << 13 )
#define TOSHIBA_RBTX4938_IRQ_IOC_DISABLE   ( 1 << 14 )
#define TOSHIBA_RBTX4938_IRQ_IOC_MASK      ( 1 << 15 )
#define TOSHIBA_RBTX4938_IRQ_IOC_ENDIRQ    ( 1 << 16 )

#define TOSHIBA_RBTX4938_IRQ_ISA_INIT      ( 1 << 20 )
#define TOSHIBA_RBTX4938_IRQ_ISA_STARTUP   ( 1 << 21 )
#define TOSHIBA_RBTX4938_IRQ_ISA_SHUTDOWN  ( 1 << 22 )
#define TOSHIBA_RBTX4938_IRQ_ISA_ENABLE    ( 1 << 23 )
#define TOSHIBA_RBTX4938_IRQ_ISA_DISABLE   ( 1 << 24 )
#define TOSHIBA_RBTX4938_IRQ_ISA_MASK      ( 1 << 25 )
#define TOSHIBA_RBTX4938_IRQ_ISA_ENDIRQ    ( 1 << 26 )

#define TOSHIBA_RBTX4938_SETUP_ALL         0xffffffff
#endif

#ifdef TOSHIBA_RBTX4938_IRQ_DEBUG
static const u32 toshiba_rbtx4938_irq_debug_flag = (TOSHIBA_RBTX4938_IRQ_NONE
						    | TOSHIBA_RBTX4938_IRQ_INFO
						    | TOSHIBA_RBTX4938_IRQ_WARN
						    | TOSHIBA_RBTX4938_IRQ_EROR
                                                 | TOSHIBA_RBTX4938_IRQ_IOC_INIT
                                                 | TOSHIBA_RBTX4938_IRQ_IOC_STARTUP
                                                 | TOSHIBA_RBTX4938_IRQ_IOC_SHUTDOWN
                                                 | TOSHIBA_RBTX4938_IRQ_IOC_ENABLE
                                                 | TOSHIBA_RBTX4938_IRQ_IOC_DISABLE
                                                 | TOSHIBA_RBTX4938_IRQ_IOC_MASK
                                                 | TOSHIBA_RBTX4938_IRQ_IOC_ENDIRQ
                                                 | TOSHIBA_RBTX4938_IRQ_ISA_INIT
                                                 | TOSHIBA_RBTX4938_IRQ_ISA_STARTUP
                                                 | TOSHIBA_RBTX4938_IRQ_ISA_SHUTDOWN
                                                 | TOSHIBA_RBTX4938_IRQ_ISA_ENABLE
                                                 | TOSHIBA_RBTX4938_IRQ_ISA_DISABLE
                                                 | TOSHIBA_RBTX4938_IRQ_ISA_MASK
                                                 | TOSHIBA_RBTX4938_IRQ_ISA_ENDIRQ
    );
#endif

#ifdef TOSHIBA_RBTX4938_IRQ_DEBUG
#define TOSHIBA_RBTX4938_IRQ_DPRINTK(flag,str...) \
        if ( (toshiba_rbtx4938_irq_debug_flag) & (flag) ) \
        { \
           char tmp[100]; \
           sprintf( tmp, str ); \
           printk( "%s(%s:%u)::%s", __FUNCTION__, __FILE__, __LINE__, tmp ); \
        }
#else
#define TOSHIBA_RBTX4938_IRQ_DPRINTK(flag,str...)
#endif


#define do_asm_sync()      \
	__asm__ __volatile__(   \
		".set push\n\t" \
		".set mips2\n\t"\
		"sync\n\t"      \
		".set pop\n\t" )

static unsigned int toshiba_rbtx4938_irq_ioc_startup(unsigned int irq);
static void toshiba_rbtx4938_irq_ioc_shutdown(unsigned int irq);
static void toshiba_rbtx4938_irq_ioc_enable(unsigned int irq);
static void toshiba_rbtx4938_irq_ioc_disable(unsigned int irq);
static void toshiba_rbtx4938_irq_ioc_mask_and_ack(unsigned int irq);
static void toshiba_rbtx4938_irq_ioc_end(unsigned int irq);

static spinlock_t toshiba_rbtx4938_ioc_lock = SPIN_LOCK_UNLOCKED;

#define TOSHIBA_RBTX4938_IOC_NAME "RBTX4938-IOC"
static struct hw_interrupt_type toshiba_rbtx4938_irq_ioc_type = {
	typename:TOSHIBA_RBTX4938_IOC_NAME,
	startup:toshiba_rbtx4938_irq_ioc_startup,
	shutdown:toshiba_rbtx4938_irq_ioc_shutdown,
	enable:toshiba_rbtx4938_irq_ioc_enable,
	disable:toshiba_rbtx4938_irq_ioc_disable,
	ack:toshiba_rbtx4938_irq_ioc_mask_and_ack,
	end:toshiba_rbtx4938_irq_ioc_end,
	set_affinity:NULL
};

#define TOSHIBA_RBTX4938_IOC_INTR_ENAB 0xb7f02000
#define TOSHIBA_RBTX4938_IOC_INTR_STAT 0xb7f0200a

u8
last_bit2num(u8 num)
{
	u8 i = ((sizeof(num)*8)-1);

	do
	{
		if ( num & (1<<i) )
		{
		  break;
		}
	} while ( --i );

	return( i );
}

int
toshiba_rbtx4938_irq_nested(int sw_irq)
{
	u8 level3;

	level3 = reg_rd08(TOSHIBA_RBTX4938_IOC_INTR_STAT) & 0xff;
	if (level3) {
                /* must use last_bit2num so onboard ATA has priority */
		sw_irq = TOSHIBA_RBTX4938_IRQ_IOC_BEG + last_bit2num(level3);
	}

	wbflush();
	return (sw_irq);
}

#define TOSHIBA_RBTX4938_PIC_ACTION(s) { no_action, 0, 0, s, NULL, NULL }
static struct irqaction toshiba_rbtx4938_irq_ioc_action =
TOSHIBA_RBTX4938_PIC_ACTION(TOSHIBA_RBTX4938_IOC_NAME);


/**********************************************************************************/
/* Functions for ioc                                                              */
/**********************************************************************************/

void
dump_irqs(char *s)
{
#if 0
	return;
#else
	int i;
	for (i = 0; i < 64; i++) {
		printk("%s: i=%02d s=0x%02x a=0x%08x d=%1d h=0x%08x n=%s\n",
		       s, i, irq_desc[i].status, (unsigned int)irq_desc[i].action,
		       irq_desc[i].depth, (unsigned int)irq_desc[i].handler,
		       (irq_desc[i].handler ? irq_desc[i].handler->
			typename : "null"));
	}
	printk("%s: 0x%08x = 0x%02x\n", s, 0xb7f02000, TX4938_RD08(0xb7f02000));
	printk("%s: 0x%08x = 0x%02x\n", s, 0xb7f02002, TX4938_RD08(0xb7f02002));
	printk("%s: 0x%08x = 0x%02x\n", s, 0xb7f02004, TX4938_RD08(0xb7f02004));
	printk("%s: 0x%08x = 0x%02x\n", s, 0xb7f02006, TX4938_RD08(0xb7f02006));
	printk("%s: 0x%08x = 0x%02x\n", s, 0xb7f03000, TX4938_RD08(0xb7f03000));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff610, TX4938_RD32(0xff1ff610));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff614, TX4938_RD32(0xff1ff614));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff618, TX4938_RD32(0xff1ff618));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff61c, TX4938_RD32(0xff1ff61c));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff620, TX4938_RD32(0xff1ff620));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff624, TX4938_RD32(0xff1ff624));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff628, TX4938_RD32(0xff1ff628));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff62c, TX4938_RD32(0xff1ff62c));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff640, TX4938_RD32(0xff1ff640));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff680, TX4938_RD32(0xff1ff680));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff6a0, TX4938_RD32(0xff1ff6a0));
#endif
}

static void __init
toshiba_rbtx4938_irq_ioc_init(void)
{
	int i;

	TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_IOC_INIT,
				     "beg=%d end=%d\n",
				     TOSHIBA_RBTX4938_IRQ_IOC_BEG,
				     TOSHIBA_RBTX4938_IRQ_IOC_END);

	for (i = TOSHIBA_RBTX4938_IRQ_IOC_BEG;
	     i <= TOSHIBA_RBTX4938_IRQ_IOC_END; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = 0;
		irq_desc[i].depth = 3;
		irq_desc[i].handler = &toshiba_rbtx4938_irq_ioc_type;
	}

	setup_irq(RBTX4938_IRQ_IOCINT,
		  &toshiba_rbtx4938_irq_ioc_action);

	return;
}

static unsigned int
toshiba_rbtx4938_irq_ioc_startup(unsigned int irq)
{
	TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_IOC_STARTUP,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4938_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4938_IRQ_IOC_END) {
		TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	toshiba_rbtx4938_irq_ioc_enable(irq);

	return (0);
}

static void
toshiba_rbtx4938_irq_ioc_shutdown(unsigned int irq)
{
	TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_IOC_SHUTDOWN,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4938_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4938_IRQ_IOC_END) {
		TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	toshiba_rbtx4938_irq_ioc_disable(irq);

	return;
}

static void
toshiba_rbtx4938_irq_ioc_enable(unsigned int irq)
{
	unsigned long flags;
	volatile unsigned char v;

	TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_IOC_ENABLE,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4938_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4938_IRQ_IOC_END) {
		TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	spin_lock_irqsave(&toshiba_rbtx4938_ioc_lock, flags);

	v = TX4938_RD08(TOSHIBA_RBTX4938_IOC_INTR_ENAB);
	v |= (1 << (irq - TOSHIBA_RBTX4938_IRQ_IOC_BEG));
	TX4938_WR08(TOSHIBA_RBTX4938_IOC_INTR_ENAB, v);

	__asm__ __volatile__
		("sync\n\t");

	TX4938_RD08(TOSHIBA_RBTX4938_IOC_INTR_ENAB);

	spin_unlock_irqrestore(&toshiba_rbtx4938_ioc_lock, flags);

	return;
}

static void
toshiba_rbtx4938_irq_ioc_disable(unsigned int irq)
{
	unsigned long flags;
	volatile unsigned char v;

	TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_IOC_DISABLE,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4938_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4938_IRQ_IOC_END) {
		TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	spin_lock_irqsave(&toshiba_rbtx4938_ioc_lock, flags);

	v = TX4938_RD08(TOSHIBA_RBTX4938_IOC_INTR_ENAB);
	v &= ~(1 << (irq - TOSHIBA_RBTX4938_IRQ_IOC_BEG));
	TX4938_WR08(TOSHIBA_RBTX4938_IOC_INTR_ENAB, v);

	__asm__ __volatile__
		("sync\n\t");

	TX4938_RD08(TOSHIBA_RBTX4938_IOC_INTR_ENAB);

	spin_unlock_irqrestore(&toshiba_rbtx4938_ioc_lock, flags);


	return;
}

static void
toshiba_rbtx4938_irq_ioc_mask_and_ack(unsigned int irq)
{
	TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_IOC_MASK, "irq=%d\n",
				     irq);

	if (irq < TOSHIBA_RBTX4938_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4938_IRQ_IOC_END) {
		TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	toshiba_rbtx4938_irq_ioc_disable(irq);

	return;
}

static void
toshiba_rbtx4938_irq_ioc_end(unsigned int irq)
{
	TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_IOC_ENDIRQ,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4938_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4938_IRQ_IOC_END) {
		TOSHIBA_RBTX4938_IRQ_DPRINTK(TOSHIBA_RBTX4938_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS))) {
		toshiba_rbtx4938_irq_ioc_enable(irq);
	}

	return;
}

extern void __init txx9_spi_irqinit(int irc_irq);
void __init
init_IRQ(void)
{
	extern void tx4938_irq_init(void);

	cli();

	/* Now, interrupt control disabled, */
	/* all IRC interrupts are masked, */
	/* all IRC interrupt mode are Low Active. */

	/* mask all IOC interrupts */
	*rbtx4938_imask_ptr = 0;

	/* clear SoftInt interrupts */
	*rbtx4938_softint_ptr = 0;
	tx4938_irq_init();
	toshiba_rbtx4938_irq_ioc_init();
	/* Onboard 10M Ether: High Active */
	TX4938_WR(TX4938_MKA(TX4938_IRC_IRDM0), 0x00000040);

	if (tx4938_ccfgptr->pcfg & TX4938_PCFG_SPI_SEL) {
		txx9_spi_irqinit(RBTX4938_IRQ_IRC_SPI);
        }

#ifdef CONFIG_PCI
	{
                extern void toshiba_tx4938_setup_pci_irq(void);
                toshiba_tx4938_setup_pci_irq();
	}
#endif

	wbflush();

	//dump_irqs( "init_IRQ" );

	return;
}
