/*
 * linux/arch/mps/tx4938/common/irq.c
 *
 * Common tx4938 irq handler
 *
 * Author: source@mvista.com
 *
 * Copyright 2001-2002 MontaVista Software Inc.
 *
 * Copyright (C) 2000-2001 Toshiba Corporation 
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under the
 * terms of the GNU General Public License version 2. This program is
 * licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel_stat.h>
#include <linux/module.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/timex.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/irq.h>
#include <asm/bitops.h>
#include <asm/bootinfo.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/mipsregs.h>
#include <asm/system.h>
#include <asm/tx4938/rbtx4938.h>

/**********************************************************************************/
/* DEBUG                                                                          */
/**********************************************************************************/
#undef TX4938_IRQ_CHECK_CP0
#undef TX4938_IRQ_CHECK_PIC

#undef TX4938_IRQ_DEBUG

#ifdef TX4938_IRQ_DEBUG
#define TX4938_IRQ_NONE        0x00000000

#define TX4938_IRQ_INFO        ( 1 <<  0 )
#define TX4938_IRQ_WARN        ( 1 <<  1 )
#define TX4938_IRQ_EROR        ( 1 <<  2 )

#define TX4938_IRQ_INIT        ( 1 <<  5 )
#define TX4938_IRQ_NEST1       ( 1 <<  6 )
#define TX4938_IRQ_NEST2       ( 1 <<  7 )
#define TX4938_IRQ_NEST3       ( 1 <<  8 )
#define TX4938_IRQ_NEST4       ( 1 <<  9 )

#define TX4938_IRQ_CP0_INIT     ( 1 << 10 )
#define TX4938_IRQ_CP0_STARTUP  ( 1 << 11 )
#define TX4938_IRQ_CP0_SHUTDOWN ( 1 << 12 )
#define TX4938_IRQ_CP0_ENABLE   ( 1 << 13 )
#define TX4938_IRQ_CP0_DISABLE  ( 1 << 14 )
#define TX4938_IRQ_CP0_MASK     ( 1 << 15 )
#define TX4938_IRQ_CP0_ENDIRQ   ( 1 << 16 )

#define TX4938_IRQ_PIC_INIT     ( 1 << 20 )
#define TX4938_IRQ_PIC_STARTUP  ( 1 << 21 )
#define TX4938_IRQ_PIC_SHUTDOWN ( 1 << 22 )
#define TX4938_IRQ_PIC_ENABLE   ( 1 << 23 )
#define TX4938_IRQ_PIC_DISABLE  ( 1 << 24 )
#define TX4938_IRQ_PIC_MASK     ( 1 << 25 )
#define TX4938_IRQ_PIC_ENDIRQ   ( 1 << 26 )

#define TX4938_IRQ_ALL         0xffffffff
#endif

#ifdef TX4938_IRQ_DEBUG
static const u32 tx4938_irq_debug_flag = (TX4938_IRQ_NONE
					  | TX4938_IRQ_INFO
					  | TX4938_IRQ_WARN | TX4938_IRQ_EROR
					  | TX4938_IRQ_CP0_INIT
//					  | TX4938_IRQ_CP0_STARTUP
//					  | TX4938_IRQ_CP0_SHUTDOWN
//					  | TX4938_IRQ_CP0_ENABLE
//					  | TX4938_IRQ_CP0_DISABLE
//					  | TX4938_IRQ_CP0_MASK
//					  | TX4938_IRQ_CP0_ENDIRQ
					  | TX4938_IRQ_PIC_INIT
//					  | TX4938_IRQ_PIC_STARTUP
//					  | TX4938_IRQ_PIC_SHUTDOWN
//					  | TX4938_IRQ_PIC_ENABLE
//					  | TX4938_IRQ_PIC_DISABLE
//					  | TX4938_IRQ_PIC_MASK
//					  | TX4938_IRQ_PIC_ENDIRQ
					  | TX4938_IRQ_INIT
					  | TX4938_IRQ_NEST1
					  | TX4938_IRQ_NEST2
					  | TX4938_IRQ_NEST3
					  | TX4938_IRQ_NEST4
    );
#endif

#ifdef TX4938_IRQ_DEBUG
#define TX4938_IRQ_DPRINTK(flag,str...) \
        if ( (tx4938_irq_debug_flag) & (flag) ) \
        { \
           char tmp[100]; \
           sprintf( tmp, str ); \
           printk( "%s(%s:%u)::%s", __FUNCTION__, __FILE__, __LINE__, tmp ); \
        }
#else
#define TX4938_IRQ_DPRINTK(flag,str...)
#endif

/**********************************************************************************/
/* Forwad definitions for all pic's                                               */
/**********************************************************************************/

static unsigned int tx4938_irq_cp0_startup(unsigned int irq);
static void tx4938_irq_cp0_shutdown(unsigned int irq);
static void tx4938_irq_cp0_enable(unsigned int irq);
static void tx4938_irq_cp0_disable(unsigned int irq);
static void tx4938_irq_cp0_mask_and_ack(unsigned int irq);
static void tx4938_irq_cp0_end(unsigned int irq);

static unsigned int tx4938_irq_pic_startup(unsigned int irq);
static void tx4938_irq_pic_shutdown(unsigned int irq);
static void tx4938_irq_pic_enable(unsigned int irq);
static void tx4938_irq_pic_disable(unsigned int irq);
static void tx4938_irq_pic_mask_and_ack(unsigned int irq);
static void tx4938_irq_pic_end(unsigned int irq);

/**********************************************************************************/
/* Kernel structs for all pic's                                                   */
/**********************************************************************************/

static spinlock_t tx4938_cp0_lock = SPIN_LOCK_UNLOCKED;
static spinlock_t tx4938_pic_lock = SPIN_LOCK_UNLOCKED;

#define TX4938_CP0_NAME "TX4938-CP0"
static struct hw_interrupt_type tx4938_irq_cp0_type = {
	typename:TX4938_CP0_NAME,
	startup:tx4938_irq_cp0_startup,
	shutdown:tx4938_irq_cp0_shutdown,
	enable:tx4938_irq_cp0_enable,
	disable:tx4938_irq_cp0_disable,
	ack:tx4938_irq_cp0_mask_and_ack,
	end:tx4938_irq_cp0_end,
	set_affinity:NULL
};

#define TX4938_PIC_NAME "TX4938-PIC"
static struct hw_interrupt_type tx4938_irq_pic_type = {
	typename:TX4938_PIC_NAME,
	startup:tx4938_irq_pic_startup,
	shutdown:tx4938_irq_pic_shutdown,
	enable:tx4938_irq_pic_enable,
	disable:tx4938_irq_pic_disable,
	ack:tx4938_irq_pic_mask_and_ack,
	end:tx4938_irq_pic_end,
	set_affinity:NULL
};

#define TX4938_PIC_ACTION(s) { no_action, 0, 0, s, NULL, NULL }
static struct irqaction tx4938_irq_pic_action =
TX4938_PIC_ACTION(TX4938_PIC_NAME);

/**********************************************************************************/
/* Functions for cp0                                                              */
/**********************************************************************************/

#define CCP0_STATUS 12
#define CCP0_CAUSE 13  

#define tx4938_irq_cp0_mask(irq) ( 1 << ( irq-TX4938_IRQ_CP0_BEG+8 ) )

static void
tx4938_irq_cp0_modify(unsigned cp0_reg, unsigned clr_bits, unsigned set_bits)
{
	unsigned long val = 0;

	switch (cp0_reg) {
	case CCP0_STATUS:{
			val =  read_c0_status();  
			break;
		}
	case CCP0_CAUSE:{
			val = read_c0_cause();
			break;
		}
	}

	val &= (~clr_bits);
	val |= (set_bits);

	switch (cp0_reg) {
	case CCP0_STATUS:{
			write_c0_status(val);
			break;
		}
	case CCP0_CAUSE:{
			write_c0_cause(val);
			break;
		}
	}

	return;
}

static void __init
tx4938_irq_cp0_init(void)
{
	int i;

	TX4938_IRQ_DPRINTK(TX4938_IRQ_CP0_INIT, "beg=%d end=%d\n",
			   TX4938_IRQ_CP0_BEG, TX4938_IRQ_CP0_END);

	for (i = TX4938_IRQ_CP0_BEG; i <= TX4938_IRQ_CP0_END; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = 0;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &tx4938_irq_cp0_type;
	}

	return;
}

static unsigned int
tx4938_irq_cp0_startup(unsigned int irq)
{
	TX4938_IRQ_DPRINTK(TX4938_IRQ_CP0_STARTUP, "irq=%d \n", irq);

#ifdef TX4938_IRQ_CHECK_CP0
	{
		if (irq < TX4938_IRQ_CP0_BEG || irq > TX4938_IRQ_CP0_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	tx4938_irq_cp0_enable(irq);

	return (0);
}

static void
tx4938_irq_cp0_shutdown(unsigned int irq)
{
	TX4938_IRQ_DPRINTK(TX4938_IRQ_CP0_SHUTDOWN, "irq=%d \n", irq);

#ifdef TX4938_IRQ_CHECK_CP0
	{
		if (irq < TX4938_IRQ_CP0_BEG || irq > TX4938_IRQ_CP0_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	tx4938_irq_cp0_disable(irq);

	return;
}

static void
tx4938_irq_cp0_enable(unsigned int irq)
{
	unsigned long flags;

	TX4938_IRQ_DPRINTK(TX4938_IRQ_CP0_ENABLE, "irq=%d \n", irq);

#ifdef TX4938_IRQ_CHECK_CP0
	{
		if (irq < TX4938_IRQ_CP0_BEG || irq > TX4938_IRQ_CP0_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	spin_lock_irqsave(&tx4938_cp0_lock, flags);

	tx4938_irq_cp0_modify(CCP0_STATUS, 0, tx4938_irq_cp0_mask(irq));

	spin_unlock_irqrestore(&tx4938_cp0_lock, flags);

	return;
}

static void
tx4938_irq_cp0_disable(unsigned int irq)
{
	unsigned long flags;

	TX4938_IRQ_DPRINTK(TX4938_IRQ_CP0_DISABLE, "irq=%d \n", irq);

#ifdef TX4938_IRQ_CHECK_CP0
	{
		if (irq < TX4938_IRQ_CP0_BEG || irq > TX4938_IRQ_CP0_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	spin_lock_irqsave(&tx4938_cp0_lock, flags);

	tx4938_irq_cp0_modify(CCP0_STATUS, tx4938_irq_cp0_mask(irq), 0);

	spin_unlock_irqrestore(&tx4938_cp0_lock, flags);

	return;
}

static void
tx4938_irq_cp0_mask_and_ack(unsigned int irq)
{
	TX4938_IRQ_DPRINTK(TX4938_IRQ_CP0_MASK, "irq=%d \n", irq);

#ifdef TX4938_IRQ_CHECK_CP0
	{
		if (irq < TX4938_IRQ_CP0_BEG || irq > TX4938_IRQ_CP0_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	tx4938_irq_cp0_disable(irq);

	return;
}

static void
tx4938_irq_cp0_end(unsigned int irq)
{
	TX4938_IRQ_DPRINTK(TX4938_IRQ_CP0_ENDIRQ, "irq=%d \n", irq);

#ifdef TX4938_IRQ_CHECK_CP0
	{
		if (irq < TX4938_IRQ_CP0_BEG || irq > TX4938_IRQ_CP0_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS))) {
		tx4938_irq_cp0_enable(irq);
	}

	return;
}

/**********************************************************************************/
/* Functions for pic                                                              */
/**********************************************************************************/

u32
tx4938_irq_pic_addr(int irq)
{
	/* MVMCP -- need to formulize this */
	irq -= TX4938_IRQ_PIC_BEG;

	switch (irq) {
	case 17:
	case 16:
	case 1:
	case 0:{
			return (TX4938_MKA(TX4938_IRC_IRLVL0));
		}
	case 19:
	case 18:
	case 3:
	case 2:{
			return (TX4938_MKA(TX4938_IRC_IRLVL1));
		}
	case 21:
	case 20:
	case 5:
	case 4:{
			return (TX4938_MKA(TX4938_IRC_IRLVL2));
		}
	case 23:
	case 22:
	case 7:
	case 6:{
			return (TX4938_MKA(TX4938_IRC_IRLVL3));
		}
	case 25:
	case 24:
	case 9:
	case 8:{
			return (TX4938_MKA(TX4938_IRC_IRLVL4));
		}
	case 27:
	case 26:
	case 11:
	case 10:{
			return (TX4938_MKA(TX4938_IRC_IRLVL5));
		}
	case 29:
	case 28:
	case 13:
	case 12:{
			return (TX4938_MKA(TX4938_IRC_IRLVL6));
		}
	case 31:
	case 30:
	case 15:
	case 14:{
			return (TX4938_MKA(TX4938_IRC_IRLVL7));
		}
	}

	return (0);
}

u32
tx4938_irq_pic_mask(int irq)
{
	/* MVMCP -- need to formulize this */
	irq -= TX4938_IRQ_PIC_BEG;

	switch (irq) {
	case 31:
	case 29:
	case 27:
	case 25:
	case 23:
	case 21:
	case 19:
	case 17:{
			return (0x07000000);
		}
	case 30:
	case 28:
	case 26:
	case 24:
	case 22:
	case 20:
	case 18:
	case 16:{
			return (0x00070000);
		}
	case 15:
	case 13:
	case 11:
	case 9:
	case 7:
	case 5:
	case 3:
	case 1:{
			return (0x00000700);
		}
	case 14:
	case 12:
	case 10:
	case 8:
	case 6:
	case 4:
	case 2:
	case 0:{
			return (0x00000007);
		}
	}
	return (0x00000000);
}

static void
tx4938_irq_pic_modify(unsigned pic_reg, unsigned clr_bits, unsigned set_bits)
{
	unsigned long val = 0;

	val = TX4938_RD(pic_reg);
	val &= (~clr_bits);
	val |= (set_bits);
	TX4938_WR(pic_reg, val);

	__asm__ __volatile__
		("sync\n\t");
	TX4938_RD(pic_reg);
	
	return;
}

static void __init
tx4938_irq_pic_init(void)
{
	unsigned long flags;
	int i;

	TX4938_IRQ_DPRINTK(TX4938_IRQ_PIC_INIT, "beg=%d end=%d\n",
			   TX4938_IRQ_PIC_BEG, TX4938_IRQ_PIC_END);

	for (i = TX4938_IRQ_PIC_BEG; i <= TX4938_IRQ_PIC_END; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = 0;
		irq_desc[i].depth = 2;
		irq_desc[i].handler = &tx4938_irq_pic_type;
	}

	setup_irq(TX4938_IRQ_NEST_PIC_ON_CP0, &tx4938_irq_pic_action);

	spin_lock_irqsave(&tx4938_pic_lock, flags);

	TX4938_WR(0xff1ff640, 0x6);	/* irq level mask -- only accept hightest */
	TX4938_WR(0xff1ff600, TX4938_RD(0xff1ff600) | 0x1);	/* irq enable */

	spin_unlock_irqrestore(&tx4938_pic_lock, flags);

	return;
}

static unsigned int
tx4938_irq_pic_startup(unsigned int irq)
{
	TX4938_IRQ_DPRINTK(TX4938_IRQ_PIC_STARTUP, "irq=%d\n", irq);

#ifdef TX4938_IRQ_CHECK_PIC
	{
		if (irq < TX4938_IRQ_PIC_BEG || irq > TX4938_IRQ_PIC_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	tx4938_irq_pic_enable(irq);

	return (0);
}

static void
tx4938_irq_pic_shutdown(unsigned int irq)
{
	TX4938_IRQ_DPRINTK(TX4938_IRQ_PIC_SHUTDOWN, "irq=%d\n", irq);

#ifdef TX4938_IRQ_CHECK_PIC
	{
		if (irq < TX4938_IRQ_PIC_BEG || irq > TX4938_IRQ_PIC_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	tx4938_irq_pic_disable(irq);

	return;
}

static void
tx4938_irq_pic_enable(unsigned int irq)
{
	unsigned long flags;

	TX4938_IRQ_DPRINTK(TX4938_IRQ_PIC_ENABLE, "irq=%d\n", irq);

#ifdef TX4938_IRQ_CHECK_PIC
	{
		if (irq < TX4938_IRQ_PIC_BEG || irq > TX4938_IRQ_PIC_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	spin_lock_irqsave(&tx4938_pic_lock, flags);

	tx4938_irq_pic_modify(tx4938_irq_pic_addr(irq), 0,
			      tx4938_irq_pic_mask(irq));

	spin_unlock_irqrestore(&tx4938_pic_lock, flags);

	return;
}

static void
tx4938_irq_pic_disable(unsigned int irq)
{
	unsigned long flags;

	TX4938_IRQ_DPRINTK(TX4938_IRQ_PIC_DISABLE, "irq=%d\n", irq);

#ifdef TX4938_IRQ_CHECK_PIC
	{
		if (irq < TX4938_IRQ_PIC_BEG || irq > TX4938_IRQ_PIC_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	spin_lock_irqsave(&tx4938_pic_lock, flags);

	tx4938_irq_pic_modify(tx4938_irq_pic_addr(irq),
			      tx4938_irq_pic_mask(irq), 0);

	spin_unlock_irqrestore(&tx4938_pic_lock, flags);

	return;
}

static void
tx4938_irq_pic_mask_and_ack(unsigned int irq)
{
	TX4938_IRQ_DPRINTK(TX4938_IRQ_PIC_MASK, "irq=%d\n", irq);

#ifdef TX4938_IRQ_CHECK_PIC
	{
		if (irq < TX4938_IRQ_PIC_BEG || irq > TX4938_IRQ_PIC_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	tx4938_irq_pic_disable(irq);

	return;
}

static void
tx4938_irq_pic_end(unsigned int irq)
{
	TX4938_IRQ_DPRINTK(TX4938_IRQ_PIC_ENDIRQ, "irq=%d\n", irq);

#ifdef TX4938_IRQ_CHECK_PIC
	{
		if (irq < TX4938_IRQ_PIC_BEG || irq > TX4938_IRQ_PIC_END) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_EROR,
					   "bad irq=%d \n", irq);
			panic("\n");
		}
	}
#endif

	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS))) {
		tx4938_irq_pic_enable(irq);
	}

	return;
}

/**********************************************************************************/
/* Main init functions                                                            */
/**********************************************************************************/

void __init
tx4938_irq_init(void)
{
	extern asmlinkage void tx4938_irq_handler(void);

	TX4938_IRQ_DPRINTK(TX4938_IRQ_INIT, "-\n");

	TX4938_IRQ_DPRINTK(TX4938_IRQ_INIT, "=Calling tx4938_irq_cp0_init()\n");
	tx4938_irq_cp0_init();

	TX4938_IRQ_DPRINTK(TX4938_IRQ_INIT, "=Calling tx4938_irq_pic_init()\n");
	tx4938_irq_pic_init();

	TX4938_IRQ_DPRINTK(TX4938_IRQ_INIT,
			   "=Calling set_except_vector(tx4938_irq_handler)\n");
	set_except_vector(0, tx4938_irq_handler);

	TX4938_IRQ_DPRINTK(TX4938_IRQ_INIT, "+\n");

	return;
}

int
tx4938_irq_nested(void)
{
	int sw_irq = 0;
	u32 level2;

	TX4938_IRQ_DPRINTK(TX4938_IRQ_NEST1, "-\n");

	level2 = TX4938_RD(0xff1ff6a0);
	TX4938_IRQ_DPRINTK(TX4938_IRQ_NEST2, "=level2a=0x%x\n", level2);

	if ((level2 & 0x10000) == 0) {
		level2 &= 0x1f;
		TX4938_IRQ_DPRINTK(TX4938_IRQ_NEST3, "=level2b=0x%x\n", level2);

		sw_irq = TX4938_IRQ_PIC_BEG + level2;
		TX4938_IRQ_DPRINTK(TX4938_IRQ_NEST3, "=sw_irq=%d\n", sw_irq);

		if (sw_irq == 26) {
			TX4938_IRQ_DPRINTK(TX4938_IRQ_NEST4, "=irq-%d\n",
					   sw_irq);

#ifdef CONFIG_TOSHIBA_RBTX4938
			{
				extern int toshiba_rbtx4938_irq_nested(int sw_irq);
				sw_irq = toshiba_rbtx4938_irq_nested(sw_irq);
			}
#endif

			TX4938_IRQ_DPRINTK(TX4938_IRQ_NEST4, "=irq+%d\n",
					   sw_irq);
		}
	}

	TX4938_IRQ_DPRINTK(TX4938_IRQ_NEST2, "=sw_irq=%d\n", sw_irq);

	TX4938_IRQ_DPRINTK(TX4938_IRQ_NEST1, "+\n");

	return (sw_irq);
}
void print_intr_pending_bits(void)
{
	unsigned int status;

	printk("interrupt pending register %x\n",*(unsigned int *)0xff1ff680);
	status = *(unsigned int *)0xff1ff680;
	if (status & (1 << 25))
		printk("ACLCPME interrupt\n");
	if (status & (1 << 24))
		printk("ACLC interrupt\n");
	if (status & (1 << 23))
		printk("PCIPMC interrupt\n");
	if (status & (1 << 22))
		printk("PCIERR interrupt\n");
	if (status & (1 << 21))
		printk("RESERVED interrupt\n");
	if (status & (1 << 20))
		printk("RESERVED interrupt\n");
	if (status & (1 << 19))
		printk("TMR 2 interrupt\n");
	if (status & (1 << 18))
		printk("TMR 1 interrupt\n");
	if (status & (1 << 17))
		printk("TMR 0 interrupt\n");
	if (status & (1 << 16))
		printk("PCIC interrupt\n");
	if (status & (1 << 15))
		printk("PDMAC interrupt\n");
	if (status & (1 << 14))
		printk("IRC interrupt\n");
	if (status & (1 << 13))
		printk("DMA [3] interrupt\n");
	if (status & (1 << 12))
		printk("DMA [2] interrupt\n");
	if (status & (1 << 11))
		printk("DMA [1] interrupt\n");
	if (status & (1 << 10))
		printk("DMA [0] interrupt\n");
	if (status & (1 << 9))
		printk("SIO  1 interrupt\n");
	if (status & (1 << 8))
		printk("SIO  0 interrupt\n");
	if (status & (1 << 7))
		printk("INT 5  interrupt\n");
	if (status & (1 << 6))
		printk("INT 4 interrupt\n");
	if (status & (1 << 5))
		printk("INT 3 interrupt\n");
	if (status & (1 << 4))
		printk("INT 2 interrupt\n");
	if (status & (1 << 3))
		printk("INT 1 interrupt\n");
	if (status & (1 << 2))
		printk("INT 0 interrupt\n");
	if (status & (1 << 1))
		printk("Write Timeout Error interrupt\n");
	if (status & (1 << 0))
		printk("ECC Error interrupt\n");
	printk("interrupt current status register %x\n",*(unsigned int *)0xff1ff6a0);
	status = *(unsigned int *)0xff1ff6a0;
	if (status & (1 << 16))
		printk("Interrupt request has NOT been generated\n");
	else
		printk("Interrupt request has been generated\n");
	printk("Interrupt level is %x\n",(status & 0x700) >> 8);
	switch (status & 0x1f) {
		case 0: printk("ECC Error\n");break;
		case 1: printk("Write Timeout Error\n");break;
		case 2: printk("INT 0\n");break;
		case 3: printk("INT 1\n");break;
		case 4: printk("INT 2\n");break;
		case 5: printk("INT 3\n");break;
		case 6: printk("INT 4\n");break;
		case 7: printk("INT 5\n");break;
		case 8: printk("SIO 0\n");break;
		case 9: printk("SIO 1\n");break;
		case 10: printk("DMA 0\n");break;
		case 11: printk("DMA 1\n");break;
		case 12: printk("DMA 2\n");break;
		case 13: printk("DMA 3\n");break;
		case 14: printk("IRC\n");break;
		case 15: printk("PDMAC\n");break;
		case 16: printk("PCIC\n");break;
		case 17: printk("TMR 0\n");break;
		case 18: printk("TMR 1\n");break;
		case 19: printk("TMR 2\n");break;
		case 20: printk("RESERVED\n");break;
		case 21: printk("RESERVED\n");break;
		case 22: printk("PCIERR\n");break;
		case 23: printk("PCIPMC\n");break;
		default: printk("RESERVED\n");break;
	}
}
void mips_spurious_interrupt(struct pt_regs *regs)
{
#if 0
	return;
#else
	unsigned long status, cause;

	printk("got spurious interrupt\n");
	status = read_c0_status();
	cause = read_c0_cause();
	printk("interrupt current status register %x\n",*(unsigned int *)0xff1ff6a0);
	printk("interrupt pending register %x\n",*(unsigned int *)0xff1ff680);
        print_intr_pending_bits();
	printk("status %lx cause %lx\n", status, cause);
	printk("epc %lx badvaddr %lx \n", regs->cp0_epc, regs->cp0_badvaddr);
//	while(1);
#endif
}
