/*
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * IDE routines for typical pc-like standard configurations.
 *
 * Copyright (C) 1998, 1999, 2001 by Ralf Baechle
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/sched.h>
#include <linux/ide.h>
#include <linux/ioport.h>
#include <linux/hdreg.h>
#include <asm/addrspace.h>
#include <asm/ptrace.h>
#include <asm/hdreg.h>
#include <asm/tx4925/tx4925.h>
#include <asm/tx4925/toshiba_rbtx4925.h>

#ifdef CONFIG_PCMCIA_RBTX4925

extern void toshiba_rbtx4925_irq_pcmcia_enable(unsigned int irq);
extern void toshiba_rbtx4925_irq_pcmcia_disable(unsigned int irq);

static ide_ioreg_t pcmcia_io_base = 0;

static ide_ioreg_t
get_pcmcia_io_base(void)
{
	int cnt;
	u32 pcm;

	for (cnt = 0; cnt < 8; cnt++) {
		pcm = (tx4925_ebuscptr->ch[cnt].cr & 0x07000008);
		if (!(pcm & 8)) {
			continue;
		}
		pcm >>= 24;
		if (pcm == 2) {
			return (ide_ioreg_t) tx4925_ebuscptr->ch[cnt].bar;
		}
	}
	return 0;
}
#endif

static int
rbtx4925_ide_default_irq(ide_ioreg_t base)
{
#ifdef CONFIG_PCMCIA_RBTX4925
	if (pcmcia_io_base && (base == pcmcia_io_base)) {
		return (RBTX4925_PCMCIA_IRQ + L1121_RDY);
	}
#endif
	return 0;
}

static ide_ioreg_t
rbtx4925_ide_default_io_base(int index)
{
#ifdef CONFIG_PCMCIA_RBTX4925
	if (index == 0) {
		if (!pcmcia_io_base) {
			pcmcia_io_base = get_pcmcia_io_base() 
				- RBTX4925_ISA_IO_OFFSET;
		}
		return pcmcia_io_base;
	}
#endif
	return 0;
}

static void
rbtx4925_ide_init_hwif_ports(hw_regs_t * hw, ide_ioreg_t data_port,
			     ide_ioreg_t ctrl_port, int *irq)
{
	ide_ioreg_t reg = data_port;
	int i;
	for (i = IDE_DATA_OFFSET; i <= IDE_STATUS_OFFSET; i++) {
		hw->io_ports[i] = reg;
		reg += 1;
	}
	if (ctrl_port) {
		hw->io_ports[IDE_CONTROL_OFFSET] = ctrl_port;
	} else {
		hw->io_ports[IDE_CONTROL_OFFSET]
		    = hw->io_ports[IDE_DATA_OFFSET] + 0x206;
	}
	if (irq != NULL)
		*irq = 0;
	hw->io_ports[IDE_IRQ_OFFSET] = 0;
}

static int
rbtx4925_ide_request_irq(unsigned int irq,
			 void (*handler) (int, void *, struct pt_regs *),
			 unsigned long flags, const char *device, void *dev_id)
{
	int ret;

	ret = request_irq(irq, handler, flags, device, dev_id);
#ifdef CONFIG_PCMCIA_RBTX4925
	toshiba_rbtx4925_irq_pcmcia_enable(irq);
#endif
	return ret;
}

static void
rbtx4925_ide_free_irq(unsigned int irq, void *dev_id)
{
#ifdef CONFIG_PCMCIA_RBTX4925
	toshiba_rbtx4925_irq_pcmcia_disable(irq);
#endif
	free_irq(irq, dev_id);
}

static int
rbtx4925_ide_check_region(ide_ioreg_t from, unsigned int extent)
{
	return check_region(from, extent);
}

static void
rbtx4925_ide_request_region(ide_ioreg_t from, unsigned int extent,
			    const char *name)
{
	request_region(from, extent, name);
}

static void
rbtx4925_ide_release_region(ide_ioreg_t from, unsigned int extent)
{
	release_region(from, extent);
}

struct ide_ops rbtx4925_ide_ops = {
	&rbtx4925_ide_default_irq,
	&rbtx4925_ide_default_io_base,
	&rbtx4925_ide_init_hwif_ports,
	&rbtx4925_ide_request_irq,
	&rbtx4925_ide_free_irq,
	&rbtx4925_ide_check_region,
	&rbtx4925_ide_request_region,
	&rbtx4925_ide_release_region
};
