/*
 * linux/arch/mips/tx4925/toshiba_rbtx4925/pci_ops.c
 *
 * Toshiba rbtx4925 pci routines
 *
 * Author: MontaVista Software, Inc.
 *         source@mvista.com
 *
 * Copyright 2001-2002 MontaVista Software Inc.
 *
 * Copyright (C) 1996, 1997, 2001  Ralf Baechle 
 * Copyright (C) 2000 RidgeRun, Inc.
 * Author: RidgeRun, Inc.
 *   glonnon@ridgerun.com, skranz@ridgerun.com, stevej@ridgerun.com
 *
 * Copyright 2001 MontaVista Software Inc.
 * Author: jsun@mvista.com or jsun@junsun.net
 *
 * Copyright 2002 MontaVista Software Inc.
 * Author: Michael Pruznick, michael_pruznick@mvista.com
 *
 * Copyright (C) 2000-2001 Toshiba Corporation 
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/types.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/config.h>

#include <asm/addrspace.h>
#include <asm/pci_channel.h>
#include <asm/tx4925/tx4925.h>
#include <asm/tx4925/tx4925_pci.h>
#include <asm/tx4925/toshiba_rbtx4925.h>
#include <asm/debug.h>

/* initialize in setup */
struct resource pci_io_resource = {
	"pci IO space",
	0,
	0,
	IORESOURCE_IO
};

/* initialize in setup */
struct resource pci_mem_resource = {
	"pci memory space",
	0,
	0,
	IORESOURCE_MEM
};

extern struct pci_ops tx4925_pci_ops;

struct pci_channel mips_pci_channels[] = {
	/* h/w only supports devices 0x00 to 0x14 */
	{&tx4925_pci_ops, &pci_io_resource, &pci_mem_resource,
	 PCI_DEVFN(0x00, 0), PCI_DEVFN(TX4925_PCIC_MAX_DEVNU, 7)},
	{NULL, NULL, NULL, 0, 0}
};

unsigned int
pcibios_assign_all_busses(void)
{
	return 1;
}

static int
mkaddr(unsigned char bus, unsigned char dev_fn, unsigned char where,
       int *flagsp)
{
	if (bus > 0) {
		/* Type 1 configuration */
		tx4925_pcicptr->g2pcfgadrs = ((bus & 0xff) << 0x10) |
		    ((dev_fn & 0xff) << 0x08) | (where & 0xfc) | 1;
	} else {
		if (dev_fn > PCI_DEVFN(TX4925_PCIC_MAX_DEVNU, 0))
			return -1;

		/* Type 0 configuration */
		tx4925_pcicptr->g2pcfgadrs = ((bus & 0xff) << 0x10) |
		    ((dev_fn & 0xff) << 0x08) | (where & 0xfc);
	}
	/* clear M_ABORT and Disable M_ABORT Int. */
	tx4925_pcicptr->pcistatus =
	    (tx4925_pcicptr->pcistatus & 0x0000ffff) |
	    (PCI_STATUS_REC_MASTER_ABORT << 16);
	tx4925_pcicptr->pcimask &= ~PCI_STATUS_REC_MASTER_ABORT;
	tx4925_pcicptr->g2pmask &= ~(1 << 6);
	return 0;
}

static int
check_abort(int flags)
{
	int code = PCIBIOS_SUCCESSFUL;
	if (tx4925_pcicptr->pcistatus & (PCI_STATUS_REC_MASTER_ABORT << 16)) {
		tx4925_pcicptr->pcistatus =
		    (tx4925_pcicptr->pcistatus & 0x0000ffff) |
		    (PCI_STATUS_REC_MASTER_ABORT << 16);
		tx4925_pcicptr->g2pstatus = (1 << 6);
		code = PCIBIOS_DEVICE_NOT_FOUND;
	}
	tx4925_pcicptr->pcimask |= PCI_STATUS_REC_MASTER_ABORT;
	tx4925_pcicptr->g2pmask |= (1 << 6);
	return code;
}

/*
 * We can't address 8 and 16 bit words directly.  Instead we have to
 * read/write a 32bit word and mask/modify the data we actually want.
 */
static int
tx4925_pcibios_read_config_byte(struct pci_dev *dev,
				int where, unsigned char *val)
{
	int flags, retval;
	unsigned char bus, func_num;

	db_assert((where & 3) == 0);
	db_assert(where < (1 << 8));

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags))
		return -1;
#ifdef __BIG_ENDIAN
	*val =
	    *(volatile u8 *) ((ulong) & tx4925_pcicptr->
			      g2pcfgdata | ((where & 3) ^ 3));
#else
	*val =
	    *(volatile u8 *) ((ulong) & tx4925_pcicptr->
			      g2pcfgdata | (where & 3));
#endif
	retval = check_abort(flags);
	if (retval == PCIBIOS_DEVICE_NOT_FOUND)
		*val = 0xff;
	return retval;
}

static int
tx4925_pcibios_read_config_word(struct pci_dev *dev,
				int where, unsigned short *val)
{
	int flags, retval;
	unsigned char bus, func_num;

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	db_assert((where & 3) == 0);
	db_assert(where < (1 << 8));

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags))
		return -1;
#ifdef __BIG_ENDIAN
	*val =
	    *(volatile u16 *) ((ulong) & tx4925_pcicptr->
			       g2pcfgdata | ((where & 3) ^ 2));
#else
	*val =
	    *(volatile u16 *) ((ulong) & tx4925_pcicptr->
			       g2pcfgdata | (where & 3));
#endif
	retval = check_abort(flags);
	if (retval == PCIBIOS_DEVICE_NOT_FOUND)
		*val = 0xffff;
	return retval;
}

static int
tx4925_pcibios_read_config_dword(struct pci_dev *dev,
				 int where, unsigned int *val)
{
	int flags, retval;
	unsigned char bus, func_num;

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	db_assert((where & 3) == 0);
	db_assert(where < (1 << 8));

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags))
		return -1;
	*val = tx4925_pcicptr->g2pcfgdata;
	retval = check_abort(flags);
	if (retval == PCIBIOS_DEVICE_NOT_FOUND)
		*val = 0xffffffff;

	return retval;
}

static int
tx4925_pcibios_write_config_byte(struct pci_dev *dev,
				 int where, unsigned char val)
{
	int flags;
	unsigned char bus, func_num;

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags))
		return -1;
#ifdef __BIG_ENDIAN
	*(volatile u8 *) ((ulong) & tx4925_pcicptr->
			  g2pcfgdata | ((where & 3) ^ 3)) = val;
#else
	*(volatile u8 *) ((ulong) & tx4925_pcicptr->g2pcfgdata | (where & 3)) =
	    val;
#endif
	return check_abort(flags);
}

static int
tx4925_pcibios_write_config_word(struct pci_dev *dev,
				 int where, unsigned short val)
{
	int flags;
	unsigned char bus, func_num;

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags))
		return -1;
#ifdef __BIG_ENDIAN
	*(volatile u16 *) ((ulong) & tx4925_pcicptr->
			   g2pcfgdata | ((where & 3) ^ 2)) = val;
#else
	*(volatile u16 *) ((ulong) & tx4925_pcicptr->g2pcfgdata | (where & 3)) =
	    val;
#endif
	return check_abort(flags);
}

static int
tx4925_pcibios_write_config_dword(struct pci_dev *dev,
				  int where, unsigned int val)
{
	int flags;
	unsigned char bus, func_num;

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	/* check if the bus is top-level */
	if (dev->bus->parent != NULL) {
		bus = dev->bus->number;
		db_assert(bus != 0);
	} else {
		bus = 0;
	}

	func_num = PCI_FUNC(dev->devfn);
	if (mkaddr(bus, dev->devfn, where, &flags))
		return -1;
	tx4925_pcicptr->g2pcfgdata = val;
	return check_abort(flags);
}

struct pci_ops tx4925_pci_ops = {
	tx4925_pcibios_read_config_byte,
	tx4925_pcibios_read_config_word,
	tx4925_pcibios_read_config_dword,
	tx4925_pcibios_write_config_byte,
	tx4925_pcibios_write_config_word,
	tx4925_pcibios_write_config_dword
};
