/*
 * linux/arch/mips/tx4925/toshiba_rbtx4925/irq.c
 *
 * Toshiba RBTX4925 specific interrupt handlers
 *
 * Author: MontaVista Software, Inc.  source@mvista.com
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

/*
IRQ  Device  

16   TX4925-CP0/00 Software 0
17   TX4925-CP0/01 Software 1
18   TX4925-CP0/02 Cascade TX4925-CP0
19   TX4925-CP0/03 Multiplexed -- do not use 
20   TX4925-CP0/04 Multiplexed -- do not use 
21   TX4925-CP0/05 Multiplexed -- do not use 
22   TX4925-CP0/06 Multiplexed -- do not use 
23   TX4925-CP0/07 CPU TIMER

24   TX4925-PIC/00 
25   TX4925-PIC/01
26   TX4925-PIC/02 Cascade RBTX4925-IOC
27   TX4925-PIC/03 RBTX4925 RTL-8019AS Ethernet
28   TX4925-PIC/04
29   TX4925-PIC/05
30   TX4925-PIC/06
31   TX4925-PIC/07
32   TX4925-PIC/08
33   TX4925-PIC/09
34   TX4925-PIC/10
35   TX4925-PIC/11
36   TX4925-PIC/12 TX4925 SerialIO Channel 0
37   TX4925-PIC/13 TX4925 SerialIO Channel 1
38   TX4925-PIC/14
39   TX4925-PIC/15
40   TX4925-PIC/16
41   TX4925-PIC/17
42   TX4925-PIC/18
43   TX4925-PIC/19
44   TX4925-PIC/20
45   TX4925-PIC/21
46   TX4925-PIC/22
47   TX4925-PIC/23
48   TX4925-PIC/24
49   TX4925-PIC/25
50   TX4925-PIC/26
51   TX4925-PIC/27
52   TX4925-PIC/28
53   TX4925-PIC/29
54   TX4925-PIC/30
55   TX4925-PIC/31

56 RBTX4925-IOC/00 PCI-D
57 RBTX4925-IOC/01 PCI-C
58 RBTX4925-IOC/02 PCI-B
59 RBTX4925-IOC/03 PCI-A
60 RBTX4925-IOC/04
61 RBTX4925-IOC/05
62 RBTX4925-IOC/06
63 RBTX4925-IOC/07

64 RBTX4925-PCMCIA/00 PCMCIA(L1121) P_RDY
65 RBTX4925-PCMCIA/01 PCMCIA(L1121) P_BVD[2]
66 RBTX4925-PCMCIA/02 PCMCIA(L1121) P_BVD[1]
67 RBTX4925-PCMCIA/03 PCMCIA(L1121) P_VS[2]
68 RBTX4925-PCMCIA/04 PCMCIA(L1121) P_VS[1]
69 RBTX4925-PCMCIA/05 PCMCIA(L1121) P_nIOIS16
70 RBTX4925-PCMCIA/06 PCMCIA(L1121) P_nCD[2]
71 RBTX4925-PCMCIA/07 PCMCIA(L1121) P_nCD[1]
*/

#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/swap.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/timex.h>
#include <asm/bootinfo.h>
#include <asm/page.h>
#include <asm/bootinfo.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/pci.h>
#include <asm/processor.h>
#include <asm/ptrace.h>
#include <asm/reboot.h>
#include <asm/time.h>
#include <linux/version.h>
#include <linux/bootmem.h>
#include <linux/blk.h>
#include <asm/tx4925/toshiba_rbtx4925.h>

#undef TOSHIBA_RBTX4925_IRQ_DEBUG

#ifdef TOSHIBA_RBTX4925_IRQ_DEBUG
#define TOSHIBA_RBTX4925_IRQ_NONE        0x00000000

#define TOSHIBA_RBTX4925_IRQ_INFO          ( 1 <<  0 )
#define TOSHIBA_RBTX4925_IRQ_WARN          ( 1 <<  1 )
#define TOSHIBA_RBTX4925_IRQ_EROR          ( 1 <<  2 )

#define TOSHIBA_RBTX4925_IRQ_IOC_INIT      ( 1 << 10 )
#define TOSHIBA_RBTX4925_IRQ_IOC_STARTUP   ( 1 << 11 )
#define TOSHIBA_RBTX4925_IRQ_IOC_SHUTDOWN  ( 1 << 12 )
#define TOSHIBA_RBTX4925_IRQ_IOC_ENABLE    ( 1 << 13 )
#define TOSHIBA_RBTX4925_IRQ_IOC_DISABLE   ( 1 << 14 )
#define TOSHIBA_RBTX4925_IRQ_IOC_MASK      ( 1 << 15 )
#define TOSHIBA_RBTX4925_IRQ_IOC_ENDIRQ    ( 1 << 16 )

#define TOSHIBA_RBTX4925_IRQ_ISA_INIT      ( 1 << 20 )
#define TOSHIBA_RBTX4925_IRQ_ISA_STARTUP   ( 1 << 21 )
#define TOSHIBA_RBTX4925_IRQ_ISA_SHUTDOWN  ( 1 << 22 )
#define TOSHIBA_RBTX4925_IRQ_ISA_ENABLE    ( 1 << 23 )
#define TOSHIBA_RBTX4925_IRQ_ISA_DISABLE   ( 1 << 24 )
#define TOSHIBA_RBTX4925_IRQ_ISA_MASK      ( 1 << 25 )
#define TOSHIBA_RBTX4925_IRQ_ISA_ENDIRQ    ( 1 << 26 )

#define TOSHIBA_RBTX4925_SETUP_ALL         0xffffffff
#endif

#ifdef TOSHIBA_RBTX4925_IRQ_DEBUG
static const u32 toshiba_rbtx4925_irq_debug_flag = (TOSHIBA_RBTX4925_IRQ_NONE
						    | TOSHIBA_RBTX4925_IRQ_INFO
						    | TOSHIBA_RBTX4925_IRQ_WARN
						    | TOSHIBA_RBTX4925_IRQ_EROR
//                                                 | TOSHIBA_RBTX4925_IRQ_IOC_INIT
//                                                 | TOSHIBA_RBTX4925_IRQ_IOC_STARTUP
//                                                 | TOSHIBA_RBTX4925_IRQ_IOC_SHUTDOWN
//                                                 | TOSHIBA_RBTX4925_IRQ_IOC_ENABLE
//                                                 | TOSHIBA_RBTX4925_IRQ_IOC_DISABLE
//                                                 | TOSHIBA_RBTX4925_IRQ_IOC_MASK
//                                                 | TOSHIBA_RBTX4925_IRQ_IOC_ENDIRQ
//                                                 | TOSHIBA_RBTX4925_IRQ_ISA_INIT
//                                                 | TOSHIBA_RBTX4925_IRQ_ISA_STARTUP
//                                                 | TOSHIBA_RBTX4925_IRQ_ISA_SHUTDOWN
//                                                 | TOSHIBA_RBTX4925_IRQ_ISA_ENABLE
//                                                 | TOSHIBA_RBTX4925_IRQ_ISA_DISABLE
//                                                 | TOSHIBA_RBTX4925_IRQ_ISA_MASK
//                                                 | TOSHIBA_RBTX4925_IRQ_ISA_ENDIRQ
    );
#endif

#ifdef TOSHIBA_RBTX4925_IRQ_DEBUG
#define TOSHIBA_RBTX4925_IRQ_DPRINTK(flag,str...) \
        if ( (toshiba_rbtx4925_irq_debug_flag) & (flag) ) \
        { \
           char tmp[100]; \
           sprintf( tmp, str ); \
           printk( "%s(%s:%u)::%s", __FUNCTION__, __FILE__, __LINE__, tmp ); \
        }
#else
#define TOSHIBA_RBTX4925_IRQ_DPRINTK(flag,str...)
#endif


#define do_asm_sync()      \
	__asm__ __volatile__(   \
		".set push\n\t" \
		".set mips2\n\t"\
		"sync\n\t"      \
		".set pop\n\t" )

#define TOSHIBA_RBTX4925_IRQ_ISA_BEG MI8259_IRQ_ISA_BEG
#define TOSHIBA_RBTX4925_IRQ_ISA_END MI8259_IRQ_ISA_END
#define TOSHIBA_RBTX4925_IRQ_ISA_MID ((TOSHIBA_RBTX4925_IRQ_ISA_BEG+TOSHIBA_RBTX4925_IRQ_ISA_END+1)/2)

/* mvmcp -- todo these are defaults from tx4927 might not work for tx4925 */
#define TOSHIBA_RBTX4925_IRQ_NEST_IOC_ON_PIC TX4925_IRQ_NEST_EXT_ON_PIC
#define TOSHIBA_RBTX4925_IRQ_NEST_ISA_ON_IOC (TOSHIBA_RBTX4925_IRQ_IOC_BEG+2)
#define TOSHIBA_RBTX4925_IRQ_NEST_ISA_ON_ISA (TOSHIBA_RBTX4925_IRQ_ISA_BEG+2)


#define TOSHIBA_RBTX4925_IRQ_PCMCIA_BEG  (TOSHIBA_RBTX4925_IRQ_IOC_END+1)     /* 64 */
#define TOSHIBA_RBTX4925_IRQ_PCMCIA_END  (TOSHIBA_RBTX4925_IRQ_PCMCIA_BEG+8)  /* 71 */
#define TOSHIBA_RBTX4925_IRQ_NEST_PCMCIA_ON_PIC (TOSHIBA_RBTX4925_IRQ_NEST_IOC_ON_PIC)


static unsigned int toshiba_rbtx4925_irq_ioc_startup(unsigned int irq);
static void toshiba_rbtx4925_irq_ioc_shutdown(unsigned int irq);
static void toshiba_rbtx4925_irq_ioc_enable(unsigned int irq);
static void toshiba_rbtx4925_irq_ioc_disable(unsigned int irq);
static void toshiba_rbtx4925_irq_ioc_mask_and_ack(unsigned int irq);
static void toshiba_rbtx4925_irq_ioc_end(unsigned int irq);

#ifdef CONFIG_PCMCIA_RBTX4925
void toshiba_rbtx4925_irq_pcmcia_enable(unsigned int irq);
void toshiba_rbtx4925_irq_pcmcia_disable(unsigned int irq);

static unsigned int toshiba_rbtx4925_irq_pcmcia_startup(unsigned int irq);
static void toshiba_rbtx4925_irq_pcmcia_shutdown(unsigned int irq);
static void toshiba_rbtx4925_irq_pcmcia_ack(unsigned int irq);
static void toshiba_rbtx4925_irq_pcmcia_end(unsigned int irq);
#endif

static spinlock_t toshiba_rbtx4925_ioc_lock = SPIN_LOCK_UNLOCKED;

#define TOSHIBA_RBTX4925_IOC_NAME "RBTX4925-IOC"
static struct hw_interrupt_type toshiba_rbtx4925_irq_ioc_type = {
	typename:TOSHIBA_RBTX4925_IOC_NAME,
	startup:toshiba_rbtx4925_irq_ioc_startup,
	shutdown:toshiba_rbtx4925_irq_ioc_shutdown,
	enable:toshiba_rbtx4925_irq_ioc_enable,
	disable:toshiba_rbtx4925_irq_ioc_disable,
	ack:toshiba_rbtx4925_irq_ioc_mask_and_ack,
	end:toshiba_rbtx4925_irq_ioc_end,
	set_affinity:NULL
};

#ifdef CONFIG_PCMCIA_RBTX4925
#define TOSHIBA_RBTX4925_PCMCIA_NAME "RBTX4925-PCMCIA"
static struct hw_interrupt_type toshiba_rbtx4925_irq_pcmcia_type = {
	typename:TOSHIBA_RBTX4925_PCMCIA_NAME,
	startup:toshiba_rbtx4925_irq_pcmcia_startup,
	shutdown:toshiba_rbtx4925_irq_pcmcia_shutdown,
	enable:toshiba_rbtx4925_irq_pcmcia_enable,
	disable:toshiba_rbtx4925_irq_pcmcia_disable,
	ack:toshiba_rbtx4925_irq_pcmcia_ack,
	end:toshiba_rbtx4925_irq_pcmcia_end,
	set_affinity:NULL,
};
#define TOSHIBA_RBTX4925_PCMCIA_INTR_STAT       RBTX4925_IOC_REG_ADDR(PCMCIA_INT_STAT2)
#endif				/* CONFIG_PCMCIA_RBTX4925 */


#define TOSHIBA_RBTX4925_IOC_INTR_ENAB 0xbb002000
#define TOSHIBA_RBTX4925_IOC_INTR_STAT 0xbb002006

u32
bit2num(u32 num)
{
	u32 i;

	for (i = 0; i < (sizeof (num) * 8); i++) {
		if (num & (1 << i)) {
			return (i);
		}
	}
	return (0);
}

int
toshiba_rbtx4925_irq_nested(int sw_irq)
{
	u32 level3;

	level3 = reg_rd08(TOSHIBA_RBTX4925_IOC_INTR_STAT) & 0xff;
	if (level3) {
		sw_irq = TOSHIBA_RBTX4925_IRQ_IOC_BEG + bit2num(level3);
		if (sw_irq != TOSHIBA_RBTX4925_IRQ_NEST_ISA_ON_IOC) {
			goto RETURN;
		}
	}
#ifdef CONFIG_PCMCIA_RBTX4925
	{
		level3 = reg_rd08(TOSHIBA_RBTX4925_PCMCIA_INTR_STAT) & 0x1;
		if (level3) {
			u32 level4;
			level4 = L1121_inb(L1121_ISR);
			if (level4) {
				sw_irq =
				    TOSHIBA_RBTX4925_IRQ_PCMCIA_BEG +
				    bit2num(level4);
				goto RETURN;
			}
		}
	}
#endif

      RETURN:
	return (sw_irq);
}

#define TOSHIBA_RBTX4925_PIC_ACTION(s) { no_action, 0, 0, s, NULL, NULL }
static struct irqaction toshiba_rbtx4925_irq_ioc_action =
TOSHIBA_RBTX4925_PIC_ACTION(TOSHIBA_RBTX4925_IOC_NAME);
#ifdef CONFIG_PCMCIA_RBTX4925
static struct irqaction toshiba_rbtx4925_irq_pcmcia_action =
TOSHIBA_RBTX4925_PIC_ACTION(TOSHIBA_RBTX4925_PCMCIA_NAME);
#endif


/**********************************************************************************/
/* Functions for ioc                                                              */
/**********************************************************************************/

void
dump_irqs(char *s)
{
#if 0
	return;
#else
	int i;
	for (i = 0; i < 64; i++) {
		printk("%s: i=%02d s=0x%02x a=0x%08x d=%1d h=0x%08x n=%s\n",
		       s, i, irq_desc[i].status, (unsigned int)irq_desc[i].action,
		       irq_desc[i].depth, (unsigned int)irq_desc[i].handler,
		       (irq_desc[i].handler ? irq_desc[i].handler->
			typename : "null"));
	}
	printk("%s: 0x%08x = 0x%02x\n", s, 0xbb002000, TX4925_RD08(0xbb002000));
	printk("%s: 0x%08x = 0x%02x\n", s, 0xbb002002, TX4925_RD08(0xbb002002));
	printk("%s: 0x%08x = 0x%02x\n", s, 0xbb002004, TX4925_RD08(0xbb002004));
	printk("%s: 0x%08x = 0x%02x\n", s, 0xbb002006, TX4925_RD08(0xbb002006));
	printk("%s: 0x%08x = 0x%02x\n", s, 0xbb003000, TX4925_RD08(0xbb003000));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff610, TX4925_RD32(0xff1ff610));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff614, TX4925_RD32(0xff1ff614));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff618, TX4925_RD32(0xff1ff618));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff61c, TX4925_RD32(0xff1ff61c));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff620, TX4925_RD32(0xff1ff620));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff624, TX4925_RD32(0xff1ff624));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff628, TX4925_RD32(0xff1ff628));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff62c, TX4925_RD32(0xff1ff62c));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff640, TX4925_RD32(0xff1ff640));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff680, TX4925_RD32(0xff1ff680));
	printk("%s: 0x%08x = 0x%08x\n", s, 0xff1ff6a0, TX4925_RD32(0xff1ff6a0));
#endif
}

static void __init
toshiba_rbtx4925_irq_ioc_init(void)
{
	int i;

	TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_IOC_INIT,
				     "beg=%d end=%d\n",
				     TOSHIBA_RBTX4925_IRQ_IOC_BEG,
				     TOSHIBA_RBTX4925_IRQ_IOC_END);

	for (i = TOSHIBA_RBTX4925_IRQ_IOC_BEG;
	     i <= TOSHIBA_RBTX4925_IRQ_IOC_END; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = 0;
		irq_desc[i].depth = 3;
		irq_desc[i].handler = &toshiba_rbtx4925_irq_ioc_type;
	}

	setup_irq(TOSHIBA_RBTX4925_IRQ_NEST_IOC_ON_PIC,
		  &toshiba_rbtx4925_irq_ioc_action);

	return;
}

static unsigned int
toshiba_rbtx4925_irq_ioc_startup(unsigned int irq)
{
	TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_IOC_STARTUP,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4925_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4925_IRQ_IOC_END) {
		TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	toshiba_rbtx4925_irq_ioc_enable(irq);

	return (0);
}

static void
toshiba_rbtx4925_irq_ioc_shutdown(unsigned int irq)
{
	TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_IOC_SHUTDOWN,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4925_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4925_IRQ_IOC_END) {
		TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	toshiba_rbtx4925_irq_ioc_disable(irq);

	return;
}

static void
toshiba_rbtx4925_irq_ioc_enable(unsigned int irq)
{
	unsigned long flags;
	volatile unsigned char v;

	TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_IOC_ENABLE,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4925_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4925_IRQ_IOC_END) {
		TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	spin_lock_irqsave(&toshiba_rbtx4925_ioc_lock, flags);

	v = TX4925_RD08(TOSHIBA_RBTX4925_IOC_INTR_ENAB);
	v |= (1 << (irq - TOSHIBA_RBTX4925_IRQ_IOC_BEG));
	TOSHIBA_RBTX4925_WR08(TOSHIBA_RBTX4925_IOC_INTR_ENAB, v);

	spin_unlock_irqrestore(&toshiba_rbtx4925_ioc_lock, flags);

	return;
}

static void
toshiba_rbtx4925_irq_ioc_disable(unsigned int irq)
{
	unsigned long flags;
	volatile unsigned char v;

	TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_IOC_DISABLE,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4925_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4925_IRQ_IOC_END) {
		TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	spin_lock_irqsave(&toshiba_rbtx4925_ioc_lock, flags);

	v = TX4925_RD08(TOSHIBA_RBTX4925_IOC_INTR_ENAB);
	v &= ~(1 << (irq - TOSHIBA_RBTX4925_IRQ_IOC_BEG));
	TOSHIBA_RBTX4925_WR08(TOSHIBA_RBTX4925_IOC_INTR_ENAB, v);

	spin_unlock_irqrestore(&toshiba_rbtx4925_ioc_lock, flags);


	return;
}

static void
toshiba_rbtx4925_irq_ioc_mask_and_ack(unsigned int irq)
{
	TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_IOC_MASK, "irq=%d\n",
				     irq);

	if (irq < TOSHIBA_RBTX4925_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4925_IRQ_IOC_END) {
		TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	toshiba_rbtx4925_irq_ioc_disable(irq);

	return;
}

static void
toshiba_rbtx4925_irq_ioc_end(unsigned int irq)
{
	TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_IOC_ENDIRQ,
				     "irq=%d\n", irq);

	if (irq < TOSHIBA_RBTX4925_IRQ_IOC_BEG
	    || irq > TOSHIBA_RBTX4925_IRQ_IOC_END) {
		TOSHIBA_RBTX4925_IRQ_DPRINTK(TOSHIBA_RBTX4925_IRQ_EROR,
					     "bad irq=%d\n", irq);
		panic("\n");
	}

	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS))) {
		toshiba_rbtx4925_irq_ioc_enable(irq);
	}

	return;
}

#ifdef CONFIG_PCMCIA_RBTX4925
/**********************************************************************************/
/* Functions for pcmcia                                                           */
/**********************************************************************************/
static void __init
toshiba_rbtx4925_irq_pcmcia_init(void)
{
	int i;

	for (i = TOSHIBA_RBTX4925_IRQ_PCMCIA_BEG;
	     i <= TOSHIBA_RBTX4925_IRQ_PCMCIA_END; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = 0;
		irq_desc[i].depth = 3;
		irq_desc[i].handler = &toshiba_rbtx4925_irq_pcmcia_type;
	}

	setup_irq(TOSHIBA_RBTX4925_IRQ_NEST_PCMCIA_ON_PIC,
		  &toshiba_rbtx4925_irq_pcmcia_action);

	return;
}

void
toshiba_rbtx4925_irq_pcmcia_enable(unsigned int irq)
{
	int irq_nr = irq - TOSHIBA_RBTX4925_IRQ_PCMCIA_BEG;
	L1121_outb(L1121_inb(L1121_IER) | (1 << irq_nr), L1121_IER);
	do_asm_sync();
}

void
toshiba_rbtx4925_irq_pcmcia_disable(unsigned int irq)
{
	int irq_nr = irq - TOSHIBA_RBTX4925_IRQ_PCMCIA_BEG;
	L1121_outb(L1121_inb(L1121_IER) & ~(1 << irq_nr), L1121_IER);
	do_asm_sync();
}

static unsigned int
toshiba_rbtx4925_irq_pcmcia_startup(unsigned int irq)
{
	int irq_nr = irq - TOSHIBA_RBTX4925_IRQ_PCMCIA_BEG;
	/* setup edge sense based on current state */
	L1121_outb((L1121_inb(L1121_ESNR) & ~(1 << irq_nr)) |
		   (L1121_inb(L1121_SR) & (1 << irq_nr)), L1121_ESNR);
	toshiba_rbtx4925_irq_pcmcia_enable(irq);
	return 0;
}

static void
toshiba_rbtx4925_irq_pcmcia_shutdown(unsigned int irq)
{
	toshiba_rbtx4925_irq_pcmcia_disable(irq);
}

static void
toshiba_rbtx4925_irq_pcmcia_ack(unsigned int irq)
{
	int irq_nr = irq - TOSHIBA_RBTX4925_IRQ_PCMCIA_BEG;
	/* setup edge sense based on current state */
	L1121_outb((L1121_inb(L1121_ESNR) & ~(1 << irq_nr)) |
		   (L1121_inb(L1121_SR) & (1 << irq_nr)), L1121_ESNR);
	/* clear edge interrupt */
	L1121_outb(1 << irq_nr, L1121_ECLR);
	toshiba_rbtx4925_irq_pcmcia_disable(irq);
}

static void
toshiba_rbtx4925_irq_pcmcia_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		toshiba_rbtx4925_irq_pcmcia_enable(irq);
}
#endif				/* CONFIG_PCMCIA_RBTX4925 */

void __init
init_IRQ(void)
{
	extern void tx4925_irq_init(void);

	cli();

	tx4925_irq_init();
	toshiba_rbtx4925_irq_ioc_init();


#ifdef CONFIG_PCMCIA_RBTX4925
	toshiba_rbtx4925_irq_pcmcia_init();
#endif

#ifdef CONFIG_PCI
	{
		extern void toshiba_rbtx4925_pci_irq_init(void);
		toshiba_rbtx4925_pci_irq_init();
	}
#endif

	wbflush();

	//dump_irqs( "init_IRQ" );

	return;
}
