/*
 * Copyright (C) 2000, 2001 Broadcom Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * These are routines to set up and handle interrupts from the
 * sb1250 general purpose timer 0.  We're using the timer as a
 * system clock, so we set it up to run at 100 Hz.  On every
 * interrupt, we update our idea of what the time of day is,
 * then call do_timer() in the architecture-independent kernel
 * code to do general bookkeeping (e.g. update jiffies, run
 * bottom halves, etc.)
 */
#include <linux/config.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
#include <linux/kernel_stat.h>

#include <asm/irq.h>
#include <asm/ptrace.h>
#include <asm/addrspace.h>
#include <asm/time.h>
#include <asm/div64.h>
#include <asm/debug.h>

#include <asm/sibyte/sb1250.h>
#include <asm/sibyte/sb1250_regs.h>
#include <asm/sibyte/sb1250_int.h>
#include <asm/sibyte/sb1250_scd.h>
#include <asm/sibyte/64bit.h>


#define IMR_IP2_VAL	K_INT_MAP_I0
#define IMR_IP3_VAL	K_INT_MAP_I1
#define IMR_IP4_VAL	K_INT_MAP_I2

extern rwlock_t xtime_lock;
extern unsigned int zbbus_mhz;		/* freq of zb-bus counter */

extern int sb1250_steal_irq(int irq);

#define	zbcycle2usec_shift      32
#define	USECS_PER_JIFFY        (1000000/HZ)

static unsigned scaled_usec_per_zbcycle;
static unsigned long last_jiffies;
static u64 last_zbcount;

/*
 * return (u32)( ((u64)x * (u64)y) >> shift ), where 0 < shift <= 32
 */
static inline int scaled_mult(int x, int y, int shift)
{
	int hi;
	unsigned lo;

	db_assert(shift <= 32);
	db_assert(shift > 0);

	__asm__("mult\t%2,%3\n\t"
	        "mfhi\t%0\n\t"
	        "mflo\t%1"
	        :"=r" (hi), "=r" (lo)
	        :"r" (x), "r" (y));

	if (shift == 32)
		return hi;
	else
		return (hi << (32 - shift)) | (lo >> shift);
}

void sb1250_time_init(void)
{
	u64 temp;
	int cpu = smp_processor_id();
	int irq = K_INT_TIMER_0+cpu;

	/* Only have 4 general purpose timers */
	if (cpu > 3) {
		BUG();
	}

	if (!cpu) {
		/* Use our own gettimeoffset() routine */
		do_gettimeoffset = sb1250_gettimeoffset;
	}

	sb1250_mask_irq(cpu, irq);

	/* Map the timer interrupt to ip[4] of this cpu */
	out64(IMR_IP4_VAL, KSEG1 + A_IMR_REGISTER(cpu, R_IMR_INTERRUPT_MAP_BASE)
	      + (irq<<3));

	/* the general purpose timer ticks at 1 Mhz independent if the rest of the system */
	/* Disable the timer and set up the count */
	out64(0, KSEG1 + A_SCD_TIMER_REGISTER(cpu, R_SCD_TIMER_CFG));
	out64(
#ifndef CONFIG_SIMULATION
		1000000/HZ
#else
		50000/HZ
#endif
		, KSEG1 + A_SCD_TIMER_REGISTER(cpu, R_SCD_TIMER_INIT));

	/* Set the timer running */
	out64(M_SCD_TIMER_ENABLE|M_SCD_TIMER_MODE_CONTINUOUS,
	      KSEG1 + A_SCD_TIMER_REGISTER(cpu, R_SCD_TIMER_CFG));

	sb1250_unmask_irq(cpu, irq);
	sb1250_steal_irq(irq);
	/*
	 * This interrupt is "special" in that it doesn't use the request_irq
	 * way to hook the irq line.  The timer interrupt is initialized early
	 * enough to make this a major pain, and it's also firing enough to
	 * warrant a bit of special case code.  sb1250_timer_interrupt is
	 * called directly from irq_handler.S when IP[4] is set during an
	 * interrupt
	 */

	/* 
	 * scaled_usec_per_zbcycle = usec_per_zbcycle << zbcycle2usec_shift
	 */ 
	temp = 1000000ULL << zbcycle2usec_shift;
	do_div(temp, zbbus_mhz * 1000000);
	scaled_usec_per_zbcycle = temp;
}

static inline u64
read_zbcount(void)
{
	return in64(KSEG1 + A_SCD_ZBBUS_CYCLE_COUNT);
}

void sb1250_timer_interrupt(struct pt_regs *regs)
{
	int cpu = smp_processor_id();
	int irq = K_INT_TIMER_0+cpu;

	/* Reset the timer */
	out64(M_SCD_TIMER_ENABLE|M_SCD_TIMER_MODE_CONTINUOUS,
	      KSEG1 + A_SCD_TIMER_REGISTER(cpu, R_SCD_TIMER_CFG));

	/*
	 * CPU 0 handles the global timer interrupt job as well
	 * as local profiling and process accouting
	 */
	if (cpu == 0) {
		write_lock(&xtime_lock);
		last_jiffies = jiffies+1;
		last_zbcount = read_zbcount();
		write_unlock(&xtime_lock);
		ll_timer_interrupt(irq, regs);
		/* ll_timer_interrupt will invoke ll_local_timer_interrupt */
	} else {
		/* every CPU should do profiling and process accouting */
		ll_local_timer_interrupt(irq, regs);
	}
}

/*
 * So hard to get smp gettimeoffset right!
 * In order to get monotonic increasing offset, we now use
 * zb bus counter, and also check the "yard stick" we stake whenever
 * a jiffie timer interrupt happens (see above).
 */
unsigned long sb1250_gettimeoffset(void)
{
	u64 count;
	unsigned ret;
	unsigned count_diff;

	/* we have xtime lock, irq disabled */
	count = read_zbcount();
	count_diff = count - last_zbcount;
	ret = scaled_mult(count_diff,
			scaled_usec_per_zbcycle,
			zbcycle2usec_shift);
	if (last_jiffies > jiffies) {
		/*
		 * we are in a narrow window where timer interrupt
		 * has happened but jiffies has not been increased yet
		 * The offset should be about 1 jiffie.  We just return
		 * the maximum intra-jiffie offset here.
		 */
		db_assert(last_jiffies == jiffies + 1);
		ret = USECS_PER_JIFFY-1;
	} else {
		db_assert(last_jiffies == jiffies);
	}

	if (ret >= USECS_PER_JIFFY)
		ret = USECS_PER_JIFFY-1;

	return ret;
}
