/*
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 1999,2000 MIPS Technologies, Inc.  All rights reserved.
 *
 * ########################################################################
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ########################################################################
 *
 * Setting up the clock on the MIPS boards.
 *
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel_stat.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/timex.h>
#include <asm/mipsregs.h>
#include <asm/ptrace.h>
#include <asm/debug.h>
#include <asm/time.h>
#include <asm/rc32438/rc32438.h>
#include <asm/rc32438/tim.h>
#include <asm/rc32438/int.h>

static unsigned long r4k_offset; /* Amount to incr compare reg each time */
extern unsigned int mips_counter_frequency;
extern unsigned int idt_cpu_freq;
/* how many counter cycles in a jiffy */
static unsigned long cycles_per_jiffy=0;

/* 
 * Figure out the r4k offset, the amount to increment the compare
 * register for each time tick. There is no RTC available.
 *
 * The RC32438 counts at half the CPU *core* speed.
 */
static unsigned long __init cal_r4koff(void)
{
	mips_counter_frequency = idt_cpu_freq * 1000000 / 2;
	cycles_per_jiffy = mips_counter_frequency / HZ;
	return (mips_counter_frequency / HZ);
}


/*
 * high-level timer interrupt service routines.  This function
 * is set as irqaction->handler and is invoked through do_IRQ.
 */
static void rc32438_timer_interrupt(int irq, void *dev_id,
				    struct pt_regs *regs)
{
	unsigned int count, tmp;

	/* ack timer interrupt, and set next interrupt */
	count = timer->tim[0].count;
	timer->tim[0].compare = (count + cycles_per_jiffy);
	tmp = timer->tim[0].ctc;
	tmp = tmp & 0xfffffffd;
	timer->tim[0].ctc =  tmp;

	/*
	 * call the generic timer interrupt handling
	 */
	timer_interrupt(irq, dev_id, regs);
}


void __init rc32438_timer_setup(struct irqaction *irq)
{
	/*
	 * replace NEW_TIME_C's CP0 counter based handler
	 * with our custom handler.
	 */
	irq->handler = rc32438_timer_interrupt;
	setup_irq(8, irq);

	timer->tim[0].compare = r4k_offset;
 	timer->tim[0].ctc = 0x1;
}


void __init rc32438_time_init(void)
{
        unsigned int est_freq, flags;

	__save_and_cli(flags);

	printk("calculating r4koff... ");
	r4k_offset = cal_r4koff();
	printk("%08lx(%d)\n", r4k_offset, (int) r4k_offset);

	est_freq = 2*r4k_offset*HZ;	
	est_freq += 5000;    /* round */
	est_freq -= est_freq%10000;
	printk("CPU frequency %d.%02d MHz\n", est_freq/1000000, 
	       (est_freq%1000000)*100/1000000);

	__restore_flags(flags);
}



