/*
 * serial_gdb.c - polling driver for 16550 UART in IDT RC32334
 *
 * Copyright (C) 2000 by Lineo, Inc.
 * Written by Quinn Jensen (jensenq@lineo.com)
 *
 */
#include <linux/serial_reg.h>

/* set remote gdb baud rate at 115200 */

#define GDB_BAUD 115200

extern unsigned int idt_cpu_freq;

#define DIVISOR ((((idt_cpu_freq * 1000 * 1000) / 2) / 16) / GDB_BAUD)

/* turn this on to watch the debug protocol echoed on the console port */
#undef DEBUG_REMOTE_DEBUG

#ifdef __MIPSEB__
#define CONS_PORT 0xb8050003
#define GDB_PORT  0xb8050023
#else
#define CONS_PORT 0xb8050000
#define GDB_PORT  0xb8050020
#endif
           
volatile unsigned char *ports[2] = {
	(volatile unsigned char *)CONS_PORT,
	(volatile unsigned char *)GDB_PORT
};


void reset_gdb_port(void);
void cons_putc(char c);
int port_getc(int port);
void port_putc(int port, char c);

int cons_getc(void)
{
	return port_getc(0);
}

void cons_putc(char c)
{
	port_putc(0, c);
}

void cons_puts(char *s)
{
	while(*s) {
		if(*s == '\n') cons_putc('\r');
		cons_putc(*s);
		s++;
	}
}

void cons_do_putn(int n)
{
	if(n) {
		cons_do_putn(n / 10);
		cons_putc(n % 10 + '0');
	}
}

void cons_putn(int n)
{
	if(n < 0) {
		cons_putc('-');
		n = -n;
	}

	if (n == 0) {
		cons_putc('0');
	} else {
		cons_do_putn(n);
	}
}

#ifdef DEBUG_REMOTE_DEBUG
static enum {HUH, SENDING, GETTING} state;

static void sent(int c)
{
	switch(state) {
	case HUH:
	case GETTING:
		cons_puts("\nSNT ");
		state = SENDING;
		/* fall through */
	case SENDING:
		cons_putc(c);
		break;
	}       
}

static void got(int c)
{
	switch(state) {
	case HUH:
	case SENDING:
		cons_puts("\nGOT ");
		state = GETTING;
		/* fall through */
	case GETTING:
		cons_putc(c);
		break;
	}       
}
#endif /* DEBUG_REMOTE */

static int first = 1;

int getDebugChar(void)
{
	int c;

	if(first) reset_gdb_port();

	c = port_getc(1);

#ifdef DEBUG_REMOTE_DEBUG
	got(c);
#endif

	return c;
}

int port_getc(int p)
{
	volatile unsigned char *port = ports[p];
	int c;

	while((*(port + UART_LSR * 4) & UART_LSR_DR) == 0) {
		continue;
	}       	

	c = *(port + UART_RX * 4);

	return c;
}

void putDebugChar(char c)
{
	if(first) reset_gdb_port();

#ifdef DEBUG_REMOTE_DEBUG
	sent(c);
#endif
	idt_disp_char(3, '%');
	port_putc(1, c);
}

#define OK_TO_XMT (UART_LSR_TEMT | UART_LSR_THRE)

void port_putc(int p, char c)
{
	volatile unsigned char *port = ports[p];
	volatile unsigned char *lsr = port + UART_LSR * 4;

	while((*lsr & OK_TO_XMT) != OK_TO_XMT) {
		continue;
	}

	*(port + UART_TX * 4) = c;
}

void reset_gdb_port(void)
{
	volatile unsigned char *port = ports[1];
	unsigned int divisor = DIVISOR;
	
	first = 0;
#if 1	
	cons_puts("reset_gdb_port: initializing remote debug serial port (internal UART 1, ");
	cons_putn(GDB_BAUD);
	cons_puts("baud, MHz=");
	cons_putn(idt_cpu_freq);
	cons_puts(", divisor=");
	cons_putn(divisor);
	cons_puts(")\n");

	/* reset the port */
	*(port + UART_CSR * 4) = 0;

	/* clear and enable the FIFOs */
	*(port + UART_FCR * 4) = UART_FCR_ENABLE_FIFO | UART_FCR_CLEAR_RCVR | 
		UART_FCR_CLEAR_XMIT | UART_FCR_TRIGGER_14;

	/* set the baud rate */
	*(port + UART_LCR * 4) = UART_LCR_DLAB;		/* enable DLL, DLM registers */
	*(port + UART_DLL * 4) = divisor;
	*(port + UART_DLM * 4) = divisor >> 8;

	/* set the line control stuff and disable DLL, DLM regs */

	*(port + UART_LCR * 4) = 0 | 	/* 1 stop bit */
		UART_LCR_WLEN8;		/* 8 bit word length */
	
	/* leave interrupts off */
	*(port + UART_IER * 4) = 0;

	/* the modem controls don't leave the chip on this port, so leave them alone */
	*(port + UART_MCR * 4) = 0;
#endif

}
