/*
 * BRIEF MODULE DESCRIPTION
 *      DS1501 (Dallas Semiconductor) Real Time Clock Support
 *
 * Author: Steve Longerbeam <stevel@mvista.com, or source@mvista.com>
 *
 * 2002,2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/types.h>
#include <linux/time.h>
#include <asm/time.h>
#include <asm/addrspace.h>
#include <asm/debug.h>
#include <asm/rc32438/rc32438.h>
#include <asm/rc32438/ds1553rtc.h>

#undef BCD_TO_BIN
#define BCD_TO_BIN(val) (((val)&15) + ((val)>>4)*10)

#undef BIN_TO_BCD
#define BIN_TO_BCD(val) ((((val)/10)<<4) + (val)%10)

#define EPOCH 2000

static unsigned long
rtc_ds1553_get_time(void)
{	
	u32 century, year, month, day, hour, minute, second;

	/* read time data */
	writeb(readb(&rtc->control) | TDC_ENA_READ, &rtc->control);
	century  = BCD_TO_BIN(readb(&rtc->control) & 0x3f);
	year = BCD_TO_BIN(readb(&rtc->year)) + (century * 100);
	month = BCD_TO_BIN(readb(&rtc->month) & 0x1f);
	day = BCD_TO_BIN(readb(&rtc->date) & 0x3f);
	hour = BCD_TO_BIN(readb(&rtc->hours) & 0x3f);	/* 24 hour format */
	minute = BCD_TO_BIN(readb(&rtc->mins) & 0x7f);
	second = BCD_TO_BIN(readb(&rtc->secs) & 0x7f);
	writeb(readb(&rtc->control) & TDC_DIS_READ, &rtc->control);

	return mktime(year, month, day, hour, minute, second);
}

static int 
rtc_ds1553_set_time(unsigned long t)
{
	struct rtc_time tm;
	u8 temp, ctrl;
	u8 year, month, day, hour, minute, second;

	/* freeze external registers */
	ctrl = readb(&rtc->control);
	writeb(ctrl | TDC_ENA_WRITE, &rtc->control);

	/* convert */
	to_tm(t, &tm);

	/* check each field one by one */

	year = BIN_TO_BCD(tm.tm_year - EPOCH);
	if (year != readb(&rtc->year))
		writeb(year, &rtc->year);

	temp = readb(&rtc->month);
	month = BIN_TO_BCD(tm.tm_mon + 1); /* tm_mon starts from 0 to 11 */
	if (month != (temp & 0x1f))
		writeb((month & 0x1f) | (temp & ~0x1f), &rtc->month);

	temp = readb(&rtc->date);
	day = BIN_TO_BCD(tm.tm_mday);
	if (day != (temp & 0x3f))
		writeb(day, &rtc->date);

	temp = readb(&rtc->hours);
	hour = BIN_TO_BCD(tm.tm_hour) & 0x3f;	/* 24 hour format */
	if (hour != (temp & 0x3f))
		writeb(hour, &rtc->hours);

	temp = readb(&rtc->mins);
	minute = BIN_TO_BCD(tm.tm_min);
	if (minute != (temp & 0x7f))
		writeb(minute, &rtc->mins);

	temp = readb(&rtc->secs);
	second = BIN_TO_BCD(tm.tm_sec);
	if (second != (temp & 0x7f))
		writeb(second, &rtc->secs);
	
	writeb(ctrl & TDC_DIS_WRITE, &rtc->control);
	return 0;
}

void
rtc_ds1553_init(void)
{
	u8 stop;

	writeb(readb(&rtc->control) | TDC_ENA_READ, &rtc->control);
	stop = readb(&rtc->secs);
	writeb(readb(&rtc->control) & TDC_DIS_READ, &rtc->control);

	if (stop & TDS_STOP) {
		/* start clock */
		writeb(readb(&rtc->control) | TDC_ENA_WRITE, &rtc->control);
		writeb(stop | TDS_STOP, &rtc->secs);
		writeb(readb(&rtc->control) & TDC_DIS_WRITE, &rtc->control);
		/* wait 2s till oscillator stabilizes */
		mdelay(2000);
	}
	
	/* set the function pointers */
	rtc_get_time = rtc_ds1553_get_time;
	rtc_set_time = rtc_ds1553_set_time;
}

