/*
 * prom.c - PROM console driver for IDT79S334 board
 *
 * Copyright (C) 2000 by Lineo, Inc.
 * Written by Quinn Jensen (jensenq@lineo.com)
 *
 */
#include <linux/config.h>
#include <linux/kdev_t.h>
#include <linux/major.h>
#include <linux/console.h>
#include <asm/bootinfo.h>
#include <asm/page.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/bootmem.h>
#include <linux/serial.h>
#include <linux/serialP.h>
#include <asm/serial.h>
#include <linux/ioport.h>

char arcs_cmdline[CL_SIZE];

/* Kernel Boot parameters */
static unsigned char bootparm[]="ip=192.168.1.3:192.168.1.1::netmask::eth0";

#ifndef CONFIG_BOOT_FROM_PROM

#define NVRAM_BASE_ADDR   0xac080000   

#define NVRAM_ENVSTART   32
#define NVRAM_ENVSIZE   4            

#define WAIT(x) { volatile int i=0; while (++i < (x)) ; }

unsigned char nvramRead( int offs )
{
   unsigned char* nvramDataPointer=(unsigned char *)(NVRAM_BASE_ADDR + offs);

   return ((unsigned char)(*nvramDataPointer)) ;
}
#endif /* CONFIG_BOOT_FROM_PROM */
  
#define PROM_ENTRY(x)   (0xbfc00000+((x)*8))

extern void cons_putc(char c);
extern void cons_puts(char *s);

void (*prom_puts)(char *str) = cons_puts;
void (*prom_putchar)(char c) = cons_putc;
int (*prom_getchar)(void) = (int (*)(void))PROM_ENTRY(11);

char ether_cmdline[32]; /* For Ethernet MAC address storage */
unsigned int idt_cpu_freq = IDT_CLOCK_MULT * IDT_BUS_FREQ;

#ifdef CONFIG_KGDB
int remote_debug;
#endif

/* 
 * register serial ports at run time when we can compute baud_base
 */

#ifdef __MIPSEB__
#define IDT_BASE 0xb8050003
#else
#define IDT_BASE 0xb8050000
#endif

struct serial_struct idt_serial[2] = {
	{flags: STD_COM_FLAGS, iomem_base: (u8 *)(IDT_BASE), 
		iomem_reg_shift: 2, io_type: SERIAL_IO_MEM},
	{flags: STD_COM_FLAGS, iomem_base: (u8 *)(IDT_BASE + 0x20),
		iomem_reg_shift: 2, io_type: SERIAL_IO_MEM},
};
#ifndef CONFIG_BOOT_FROM_PROM
void prom_setup_cmdline(void)
{

    int byteCount = 0;
    int envCount = NVRAM_ENVSTART;

    char *cmdp = arcs_cmdline;
    char *ethp = ether_cmdline;

    int clen = CL_SIZE - 1;

    int esize = (int) ( (nvramRead(NVRAM_ENVSIZE)<< 8) + 
			nvramRead(NVRAM_ENVSIZE + 1) );

    if(esize < 0 || esize > 512) return;	/* bad format */
    
    *cmdp = '\0';
    *ethp = '\0';

	/* copy relevant env vars from the nvram
	 *
	 * prom environment variables of the form "bootparm?=stuff"
	 * are combined in the order they appear in the nvram to
	 * to form the boot command line.  e.g.
	 *
	 *	bootaddr=1.2.3.4
	 *	bootparm1=root=/dev/nfs
	 *	bootparm2=nfsroot=/home/nfsroot,nolock
	 *
	 * becomes
	 *
	 *	root=/dev/nfs nfsroot=/home/nfsroot,nolock
	 */

    for(;;){

	/* first byte is length */
	unsigned int elen =
	    (unsigned int) (nvramRead(envCount) & 0xFF );
    
	char tmp_env[256], *env = tmp_env;
	
	if(elen == 0 || elen > esize) break;	/* bad format */
    
	for(byteCount=0; byteCount<elen; byteCount++)
	    env[byteCount] = nvramRead(envCount + byteCount);
	env[byteCount] = '\0';
	
	envCount += elen;
	esize -= elen;
    
	env++;
	elen--;

	if(strncmp(env, "bootparm", 8) == 0) {
	    unsigned int vlen;

	    /* skip beyond the '=' and trim to fit */
	    env += 10;
	    vlen = elen - 10;
	    if(clen < vlen + 2) vlen = clen;

	    /* copy it */

	    memcpy(cmdp, env, vlen);
	    cmdp += vlen; clen -= vlen;

	    /* add a blank and null terminate */
	    
	    *cmdp++ = ' '; clen--;
	    *cmdp = '\0';
	} 
	else if(strncmp(env,"ethaddress",10) == 0) {
#if 0
	    extern unsigned char Ether_Addr[6]; /* drivers/net/bancore.c */
	    unsigned int vlen;
	    unsigned int i =0;
	    unsigned int sn_enaddr[6];
	    char *p;

	    /* skip beyond the '=' and trim to fit */
	    env += 11;
	    vlen = elen - 11;
	    /* copy it */

	    memcpy(ethp, env, vlen);
	    ethp += vlen;
	    *ethp = '\0';

	    p = ether_cmdline;

	    for (i = 0; i < 6; i++){
		sn_enaddr[i] = simple_strtoul (p, &p, 0);
		if (i != 5) {
		    if (*p != '-' && *p != ':')
			{
			    break;
			}
		    p++;
		}
	    }
	    if (i != 6 || *p) {
		printk("Ethernet address %s is not valid. Using defaults.\n",
		       ether_cmdline);
		continue;
	    }
	    else
		for(i=0;i<6;i++)

		    Ether_Addr[i] = sn_enaddr[i];

	    printk("Kernel is compiled without ethernet support.\n");
#endif
	}

	else if(strncmp(env, "MHZ=", 4) == 0) {
	    idt_cpu_freq = simple_strtoul(env + 4, 0, 10);
	    printk("CPU Clock at %d MHz (from MHZ environment variable)\n", 
		   idt_cpu_freq);
	}
	
    
	/* all done? */
	
	if(clen <= 0) break;
	if(esize <= 0) break;
    }
    printk("\n");
}
#endif
void prom_console_write(struct console *con, const char *s, unsigned count)
{
	int i;

	for(i = 0; i < count; i++, s++) {
		if(*s == '\n') prom_putchar('\r');
		prom_putchar(*s);
	}
}

kdev_t prom_console_device(struct console *con)
{
	return MKDEV(TTY_MAJOR, 64 + con->index);
}

#if 0
int prom_console_wait_key(struct console *con)
{
	return prom_getchar();
}
#endif

int prom_console_setup(struct console *con, char *options)
{
	return 0;
}

struct console prom_console_driver = {
	name: "ttyS",
	write: prom_console_write,		/* write */
	read: NULL,				/* read */
	device: prom_console_device,		/* device */
//	prom_console_wait_key,		/* wait_key */
	unblank: NULL,				/* unblank */
	setup: prom_console_setup,   		/* setup */
	flags: CON_PRINTBUFFER,
	index: -1,
	cflag: 0,
	next: NULL
};

extern unsigned long mips_machgroup;
extern unsigned long mips_machtype;

#define PFN_UP(x)	(((x) + PAGE_SIZE-1) >> PAGE_SHIFT)
#define PFN_ALIGN(x)	(((unsigned long)(x) + (PAGE_SIZE - 1)) & PAGE_MASK)

/* IDT 79EB438 memory map -- we really should be auto sizing it */

#define RAM_FIRST       0x80000400  /* Leave room for interrupt vectors */
#if 0
#define RAM_SIZE        (64*1024*1024 - 0x0400)
#endif
#define RAM_SIZE        64*1024*1024
#define RAM_END         (0x80000000 + RAM_SIZE)     
struct resource rc32438_res_ram = {
	"RAM",
	0,
	RAM_SIZE,
	IORESOURCE_MEM
};
extern void _ftext, _end;

int __init prom_init(int argc, char **argv, char **envp)
{
	/* turn on the console */

#define CONFIG_IDT_PROM_CONSOLE
#ifdef CONFIG_IDT_PROM_CONSOLE

	register_console(&prom_console_driver);

#endif

	/* set up command line */

#ifdef CONFIG_BOOT_FROM_PROM
	sprintf(arcs_cmdline,"%s\n",bootparm);
#else
	prom_setup_cmdline();
#endif

	/* set our arch type */

	mips_machgroup = MACH_GROUP_IDT;

	/*
	 * give all RAM to boot allocator,
	 * except where the kernel was loaded
	 */

  printk("prom.c:RAM_FIRST %x _ftext %x _end %x\n",
	 PHYSADDR((unsigned long)RAM_FIRST),
	 PHYSADDR((unsigned long)&_ftext),
	 PHYSADDR((unsigned long)&_end));

	add_memory_region(0,
			  rc32438_res_ram.end - rc32438_res_ram.start,
			  BOOT_MEM_RAM);
#ifdef CONFIG_KGDB
	if(strstr(arcs_cmdline, "kgdb=on")) {
		remote_debug = 1;
	}
#endif

	return 0;
}

void prom_free_prom_memory(void)
{
	printk("stubbed prom_free_prom_memory()\n");
}








