/*
 * BRIEF MODULE DESCRIPTION
 *	RC32355 interrupt routines.
 *
 * Author: Steve Longerbeam <stevel@mvista.com, or source@mvista.com>
 *
 * 2002 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel_stat.h>
#include <linux/module.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/timex.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/delay.h>

#include <asm/bitops.h>
#include <asm/bootinfo.h>
#include <asm/io.h>
#include <asm/mipsregs.h>
#include <asm/system.h>
#include <asm/rc32438/rc32438.h>
#include <asm/rc32438/gpio.h>
#undef DEBUG_IRQ
#ifdef DEBUG_IRQ
/* note: prints function name for you */
#define DPRINTK(fmt, args...) printk("%s: " fmt, __FUNCTION__ , ## args)
#else
#define DPRINTK(fmt, args...)
#endif

#ifdef CONFIG_KGDB
extern void breakpoint(void);
#endif

extern asmlinkage void idtIRQ(void);
extern void set_debug_traps(void);
extern irq_cpustat_t irq_stat [NR_CPUS];
unsigned int local_bh_count[NR_CPUS];
unsigned int local_irq_count[NR_CPUS];
extern ll_timer_interrupt(int irq, struct pt_regs *regs);
static unsigned int startup_irq(unsigned int irq);
static void end_irq(unsigned int irq_nr);
static void mask_and_ack_irq(unsigned int irq_nr);
static void rc32438_enable_irq(unsigned int irq_nr);
static void rc32438_disable_irq(unsigned int irq_nr);
void s438_pci_dispatch(int irq, void *dev_id, struct pt_regs *regs);
extern unsigned int do_IRQ(int irq, struct pt_regs *regs);
extern void __init init_generic_irq(void);
extern void idt_disp_str(char *s);

#define RC32438_NR_IRQS  (GROUP4_IRQ_BASE + 64)

#ifdef CONFIG_PM
extern void counter0_irq(int irq, void *dev_id, struct pt_regs *regs);
#endif

typedef struct {
	int irq_base;   /* Base IRQ # of this interrupt group */
	int num_irqs;   /* Number of IRQs in this group */
	u32 mask;       /* mask of valid bits in pending/mask registers */
} intr_group_t;

static const intr_group_t intr_group[NUM_INTR_GROUPS] = {
	{ GROUP0_IRQ_BASE,  15, 0x0000ffff },
	{ GROUP1_IRQ_BASE,  13, 0x00003fff },
	{ GROUP2_IRQ_BASE,   3, 0x00000003 },
	{ GROUP3_IRQ_BASE,  18, 0x0007ffff },
	{ GROUP4_IRQ_BASE,  32, 0xffffffff }
};

#define READ_PEND(g) \
       rc32438_readl(IC_GROUP0_PEND + (g)*IC_GROUP_OFFSET)
#define READ_MASK(g) \
       rc32438_readl(IC_GROUP0_MASK + (g)*IC_GROUP_OFFSET)
#define WRITE_MASK(g,val) \
       rc32438_writel((val), IC_GROUP0_MASK + (g)*IC_GROUP_OFFSET)

static inline int irq_to_group(unsigned int irq_nr)
{
	int i;
	for (i=NUM_INTR_GROUPS-1; i >= 0; i--) {
		if (irq_nr >= intr_group[i].irq_base)
			break;
	}

	return i;
}

static inline int ip_to_irq(int ipnum)
{
	if (ipnum <= 6 && ipnum >= 2)
		return intr_group[ipnum-2].irq_base;
	else
		return ipnum;
}

static inline int irq_to_ip(int irq)
{
	if (irq < GROUP0_IRQ_BASE) {
		return irq;
	} else {
		return irq_to_group(irq) + 2;
	}
}

static inline int group_to_ip(int group)
{
	return group + 2;
}

static inline int ip_to_group(int ipnum)
{
	return ipnum - 2;
}

static inline void enable_local_irq(unsigned int irq_nr)
{
	int ipnum = irq_to_ip(irq_nr);
	clear_c0_cause(1 << (ipnum + 8));
	set_c0_status(1 << (ipnum + 8));
}

static inline void disable_local_irq(unsigned int irq_nr)
{
	int ipnum = irq_to_ip(irq_nr);
	clear_c0_status(1 << (ipnum + 8));
}

static inline void ack_local_irq(unsigned int irq_nr)
{
	int ipnum = irq_to_ip(irq_nr);
	clear_c0_cause(1 << (ipnum + 8));
}

static void enable_exp_irq(unsigned int irq_nr, int group)
{
	const intr_group_t* g = &intr_group[group];
	u32 mask, intr_bit;

	// calc interrupt bit within group
	intr_bit = (1 << (irq_nr - g->irq_base)) & g->mask;
	if (!intr_bit)
		return;
	
	DPRINTK("irq%d (group %d, mask %d)\n",
		irq_nr, group, intr_bit);
	
	// first enable the IP mapped to this IRQ
	enable_local_irq(irq_nr);

	// unmask intr within group
	mask = READ_MASK(group) & g->mask;

	WRITE_MASK(group, mask & ~intr_bit);
}

static void disable_exp_irq(unsigned int irq_nr, int group)
{
	const intr_group_t* g = &intr_group[group];
	u32 mask, intr_bit;
	
	// calc interrupt bit within group
	intr_bit = (1 << (irq_nr - g->irq_base)) & g->mask;
	if (!intr_bit)
		return;
	
	DPRINTK("irq%d (group %d, mask %d)\n",
		irq_nr, group, intr_bit);
	
	// mask intr within group
	mask = READ_MASK(group) & g->mask;
	mask |= intr_bit;

	WRITE_MASK(group, mask);
	
	/*
	  if there are no more interrupts enabled in this
	  group, disable corresponding IP
	*/
	if (mask == g->mask)
		disable_local_irq(irq_nr);
}


static void rc32438_enable_irq(unsigned int irq_nr)
{
	unsigned long flags;
	save_and_cli(flags);

	if (irq_nr < GROUP0_IRQ_BASE)
		enable_local_irq(irq_nr);
	else {
		int group = irq_to_group(irq_nr);

		enable_exp_irq(irq_nr, group);
	}

	restore_flags(flags);
}


static void rc32438_disable_irq(unsigned int irq_nr)
{
	unsigned long flags;
	save_and_cli(flags);

	if (irq_nr < GROUP0_IRQ_BASE)
		disable_local_irq(irq_nr);
	else {
		int group = irq_to_group(irq_nr);
		disable_exp_irq(irq_nr, group);
	}
	
	restore_flags(flags);
}


void rc32438_ack_irq(unsigned int irq_nr)
{
	gpio->gpioistat = 0xf7ffffff;
	ack_local_irq(irq_nr);
	
}

static unsigned int startup_irq(unsigned int irq_nr)
{

	rc32438_enable_irq(irq_nr);
	return 0; 
}


static void shutdown_irq(unsigned int irq_nr)
{
	rc32438_disable_irq(irq_nr);
	return;
}


static void mask_and_ack_irq(unsigned int irq_nr)
{
	rc32438_disable_irq(irq_nr);

}

static void end_irq(unsigned int irq_nr)
{
        int group;
	if (!(irq_desc[irq_nr].status & (IRQ_DISABLED|IRQ_INPROGRESS))) {
		if (irq_nr < GROUP0_IRQ_BASE) {
			ack_local_irq(irq_nr);
			enable_local_irq(irq_nr);
		} else {

                        if (irq_nr == 84)
			    gpio->gpioistat = 0x0;

			ack_local_irq(irq_nr);

			group = irq_to_group(irq_nr);
			enable_exp_irq(irq_nr, group);
                        
		}
	} else {
		printk("warning: end_irq %d did not enable (%x)\n", 
		       irq_nr, irq_desc[irq_nr].status);
	}
}

static struct hw_interrupt_type rc32438_irq_type = {
	"RC32438",
	startup_irq,
	shutdown_irq,
	rc32438_enable_irq,
	rc32438_disable_irq,
	mask_and_ack_irq,
	end_irq,
	NULL
};

void __init init_IRQ(void)
{
	int i;
	unsigned long cp0_status;

	cp0_status = read_c0_status();

	memset(irq_desc, 0, sizeof(irq_desc));
	set_except_vector(0, idtIRQ);
	
	init_generic_irq();

	for (i = 0; i < RC32438_NR_IRQS; i++) {
                irq_desc[i].status = IRQ_DISABLED;
                irq_desc[i].action = NULL;
                irq_desc[i].depth = 1;
		irq_desc[i].handler = &rc32438_irq_type;
	}

#ifdef CONFIG_KGDB
	/* If local serial I/O used for debug port, enter kgdb at once */
	cons_puts("Waiting for kgdb to connect...");
	set_debug_traps();
	breakpoint();
#endif

}


/*
 * Interrupts are nested. Even if an interrupt handler is registered
 * as "fast", we might get another interrupt before we return from
 * *_dispatch().
 */

/* Dispatch to expanded interrupts */
static void int01234_dispatch(struct pt_regs *regs, int ipnum)
{
	int group, intr;
	const intr_group_t* g;
	u32 pend;

	group = ip_to_group(ipnum);
	g = &intr_group[group];

	pend = READ_PEND(group) & g->mask;
	pend &= ~READ_MASK(group); // only unmasked interrupts
	if (!pend){
		return; // no interrupts pending in this group
        }
	intr = 31 - rc32438_clz(pend);
#ifdef DEBUG_IRQ
	idtprintf("%02d%02d", group, intr);
#endif

	 do_IRQ(g->irq_base + intr, regs);
}

static void mips_spurious_interrupt(struct pt_regs *regs)
{
#if 1
        return;
#else
        unsigned long status, cause;

        printk("got spurious interrupt\n");
        status = read_c0_status();
        cause = read_c0_cause();
        printk("status %x cause %x\n", status, cause);
        printk("epc %x badvaddr %x \n", regs->cp0_epc, regs->cp0_badvaddr);
//      while(1);
#endif
}

/* Main Interrupt dispatcher */
void rc32438_irqdispatch(unsigned long cp0_cause, struct pt_regs *regs)
{
	unsigned long ip;
	int ipnum;
       	cp0_cause = read_c0_cause();
	ip = (cp0_cause >> 8) & 0xff;
	
	if (!ip) {
		mips_spurious_interrupt(regs);
		return;
	}
	
	ipnum = 31 - rc32438_clz(ip);

	if (ipnum >= 3 && ipnum <= 6) {
		int01234_dispatch(regs, ipnum);
	} else {
		int irq = ip_to_irq(ipnum);
		do_IRQ(irq, regs);
	}
}

void s438_pci_dispatch(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned char bit;

again:
	for(irq = 111, bit = 1 << 7; irq >= 104; irq--, bit >>= 1) {
		unsigned char int_status = ~ *(unsigned char *)0xac0c0000;
		if(int_status & bit) {
			if(bit == 1 && (int_status & 0xF) != 1) goto again;
			do_IRQ(irq, regs);
		}
	}

	
}











