/*
 * setup.c - boot time setup code for IDT 79S355/79EB355 (Banyan) board
 *
 * Copyright (C) 2000 by Lineo, Inc.
 * Written by Quinn Jensen (jensenq@lineo.com)
 * Ammended by Gray Girling (GGirling@uk.research.att.com) from IDT 79S334
 * (can still be used with 79S334 board)
 *
 */
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/irq.h>
#include <linux/console.h>
#include <asm/bootinfo.h>
#include <asm/io.h>
#include <linux/ioport.h>
#include <asm/mipsregs.h>
#include <asm/pgtable.h>
#include <linux/mc146818rtc.h>	/* for rtc_ops, we fake the RTC */
#include <asm/time.h>
#include <asm/reboot.h>
#include <asm/traps.h>
#include <asm/addrspace.h>	/* for KSEG1ADDR() */
#include <asm/rc32434/rc32434.h>
#include <asm/rc32434/pci.h>
#include <asm/rc32434/pcikorina.h>
#include <asm/rc32434/integ.h>
#include <asm/rc32434/rst.h>

#define WORKAROUND_434_BUSERR

#ifdef CONFIG_BLK_DEV_IDE
extern struct ide_ops std_ide_ops;
extern struct ide_ops *ide_ops;
#endif

extern char *__init prom_getcmdline(void);
extern void rc32434_time_init(void);
extern void rc32434_timer_setup(struct irqaction *irq);

void (*__wbflush) (void);
void idt_disp_str(char *s);
#ifdef CONFIG_PCI
extern void __init rc32434_pcibridge_init(void);
#endif
struct resource rc32434_res_pci_mem1;
struct resource rc32434_res_pci_mem2;
struct resource rc32434_res_pci_mem3;

#define PCI_MEM1_START PCI_ADDR_START
#define PCI_MEM1_END   PCI_ADDR_START + CPUTOPCI_MEM_WIN - 1
#define PCI_MEM2_START PCI_ADDR_START + CPUTOPCI_MEM_WIN
#define PCI_MEM2_END   PCI_ADDR_START + ( 2* CPUTOPCI_MEM_WIN)  - 1
#define PCI_IO1_START   PCI_ADDR_START + (2 * CPUTOPCI_MEM_WIN)
#define PCI_IO1_END     PCI_ADDR_START + (2* CPUTOPCI_MEM_WIN) + CPUTOPCI_IO_WIN -1
#define PCI_IO2_START   PCI_ADDR_START + (2 * CPUTOPCI_MEM_WIN) + CPUTOPCI_IO_WIN
#define PCI_IO2_END     PCI_ADDR_START + (2* CPUTOPCI_MEM_WIN) + (2 * CPUTOPCI_IO_WIN) -1

#define epldMask ((volatile unsigned char *)0xB900000d)

struct resource rc32434_res_pci_mem1 = {
	"PCI Mem1",
	0x50000000,
	0x5FFFFFFF,
	IORESOURCE_MEM,
	&rc32434_res_pci_mem1,
	NULL,
	&rc32434_res_pci_mem2
};
struct resource rc32434_res_pci_mem2 = {
	"PCI Mem2",
	0x60000000,
	0x6FFFFFFF,
	IORESOURCE_MEM,
	&rc32434_res_pci_mem1,
	NULL,
	NULL
};
struct resource rc32434_res_pci_mem3 = {
	"PCI Mem3",
	0x18C00000,
	0x18FFFFFF,
	IORESOURCE_MEM,
	&rc32434_res_pci_mem1,
	NULL,
	NULL
};

struct resource rc32434_res_pci_io1 = {
	"PCI I/O1",
	0x18800000,
	0x188FFFFF,
	IORESOURCE_IO
};

#define DIG_CLEAR ((volatile unsigned char *)0xAC000000)
#define DIG0 ((volatile unsigned char *)0xAC040003)
#define DIG1 ((volatile unsigned char *)0xAC040002)
#define DIG2 ((volatile unsigned char *)0xAC040001)
#define DIG3 ((volatile unsigned char *)0xAC040000)

void
idt_disp_char(int i, char c)
{
	switch (i) {
	case 0:
		*DIG0 = c;
		break;
	case 1:
		*DIG1 = c;
		break;
	case 2:
		*DIG2 = c;
		break;
	case 3:
		*DIG3 = c;
		break;
	default:
		*DIG0 = '?';
		break;
	}
}

void
idt_disp_str(char *s)
{
	if (s == 0) {
		char c = *DIG_CLEAR;
	} else {
		int i;
		for (i = 0; i < 4; i++) {
			if (s[i])
				idt_disp_char(i, s[i]);
		}
	}
}

static void
idt_machine_restart(char *command)
{
	printk("idt_machine_restart: command=%s\n", command);

	/* just jump to the reset vector */

	((void (*)(void)) KSEG1ADDR(0x1FC00000u)) ();
}

static void
idt_machine_halt(void)
{
	printk("idt_machine_halt:  halted\n");
	for (;;)
		continue;
}

static void
idt_machine_power_off(void)
{
	printk
	    ("idt_machine_power_off:  It is now safe to turn off the power\n");
	for (;;)
		continue;
}

static void
rc32434_wbflush(void)
{
	__asm__ volatile ("sync");
}

int
rc32434_be_handler(struct pt_regs *regs, int is_fixup)
{
	INTEG_t integ = (INTEG_t) INTEG_VirtualAddress;
	RST_t rst = (RST_t) RST_VirtualAddress;
	u32 cea = rst->cea;
	u32 errcs = integ->errcs;

	/* clear the CPU bus error bits */
	integ->errcs = errcs & ~(ERRCS_ucr_m | ERRCS_sae_m);

	printk(KERN_DEBUG "RC32434 bus error: ERRCS=%04x, CEA=%08x\n",
	       errcs, cea);
	printk(KERN_DEBUG "  EPC=%08lx, RA=%08lx\n",
	       regs->cp0_epc, regs->regs[31]);

	/*
	 * The 79EB434 is getting bus errors, caused by CPU accesses
	 * to undecoded addresses, occuring always inside memcpy().
	 * CEA always contains the physaddr 0x04040000 when the bus
	 * error occurs, which is 256kB past the end of physical
	 * memory on the 79EB434. Using kgdb, I could find no obvious
	 * s/w bugs that would generate these bus errors, so at this
	 * point they appear to be a h/w glitch. And they also appear
	 * to be innocuous, so for now I am returning MIPS_BE_DISCARD.
	 * <stevel@mvista.com>
	 */
#ifdef WORKAROUND_434_BUSERR
	return MIPS_BE_DISCARD;
#else

	if (is_fixup)
		return MIPS_BE_FIXUP;

	return MIPS_BE_FATAL;
#endif
}

void __init
idt_setup(void)
{
	unsigned int pciCntlVal;
	char *argptr;
	idt_disp_str("Unix");

	_machine_restart = idt_machine_restart;
	_machine_halt = idt_machine_halt;
	_machine_power_off = idt_machine_power_off;
	__wbflush = rc32434_wbflush;
	set_io_port_base(KSEG1);

	board_time_init = rc32434_time_init;
	board_timer_setup = rc32434_timer_setup;
	board_be_handler = rc32434_be_handler;

	ioport_resource.start = rc32434_res_pci_io1.start;
	ioport_resource.end = rc32434_res_pci_io1.end;
	iomem_resource.start = rc32434_res_pci_mem1.start;
	iomem_resource.end = rc32434_res_pci_mem2.end;
	write_c0_wired(0);
	/* Disable the IP bus error for PCI scaning */
	pciCntlVal = rc32434_pci->pcic;
	pciCntlVal &= 0xFFFFFF7;
	rc32434_pci->pcic = pciCntlVal;
#ifdef CONFIG_PCI
	*epldMask = 0x0;
	rc32434_pcibridge_init();
#endif

#ifdef CONFIG_BLK_DEV_IDE
	ide_ops = &std_ide_ops;
#endif

#ifdef CONFIG_FB
	conswitchp = &dummy_con;
#endif

}

int
page_is_ram(unsigned long pagenr)
{
	return 1;
}

const char *
get_system_type(void)
{
	return "MIPS IDT32434";
}
