/*
 *
 * BRIEF MODULE DESCRIPTION
 *	RC32334 specific pci support.
 *
 * Author: Steve Longerbeam <stevel@mvista.com, or source@mvista.com>
 *
 * 2001 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/config.h>

#ifdef CONFIG_PCI

#include <linux/types.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/pci_channel.h>
#include <asm/rc32434/rc32434.h>
#include <asm/rc32434/pci.h>
#include <asm/rc32434/pcikorina.h>

#define PCI_ACCESS_READ  0
#define PCI_ACCESS_WRITE 1

#undef DEBUG
#ifdef DEBUG
#define DBG(x...) printk(x)
#else
#define DBG(x...)
#endif

#ifdef __MIPSEB__
#define SWAP_BIT 1
#else
#define SWAP_BIT 0
#endif
/* define an unsigned array for the PCI registers */
unsigned int korinaCnfgRegs[25] = {
	KORINA_CNFG1,
	KORINA_CNFG2,
	KORINA_CNFG3,
	KORINA_CNFG4,
	KORINA_CNFG5,
	KORINA_CNFG6,
	KORINA_CNFG7,
	KORINA_CNFG8,
	KORINA_CNFG9,
	KORINA_CNFG10,
	KORINA_CNFG11,
	KORINA_CNFG12,
	KORINA_CNFG13,
	KORINA_CNFG14,
	KORINA_CNFG15,
	KORINA_CNFG16,
	KORINA_CNFG17,
	KORINA_CNFG18,
	KORINA_CNFG19,
	KORINA_CNFG20,
	KORINA_CNFG21,
	KORINA_CNFG22,
	KORINA_CNFG23,
	KORINA_CNFG24
};
unsigned int pciConfigAddr;	/*used for writing pci config values */
int loopCount;			/*used for the loop */
unsigned int dummyRead;		/*used to flush CPU write buffers */
extern struct resource rc32434_res_pci_io1;
extern struct resource rc32434_res_pci_io2;
extern struct resource rc32434_res_pci_mem1;
extern struct resource rc32434_res_pci_mem2;

extern char *__init prom_getcmdline(void);

#define PCI_CFG_SET(slot,func,off) \
	(rc32434_pci->pcicfga = (0x80000000 | ((slot)<<11) | \
			    ((func)<<8) | (off)))

static int
config_access(u8 type, u8 bus, u8 devfn, u8 where, u32 * data)
{
	/* 
	 * config cycles are on 4 byte boundary only
	 */
	u8 slot = PCI_SLOT(devfn);
	u8 func = PCI_FUNC(devfn);

	if (bus != 0 || slot > 5) {
		*data = 0xFFFFFFFF;
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	/* Setup address */
	PCI_CFG_SET(slot, func, where);
	rc32434_sync();

	if (type == PCI_ACCESS_WRITE)
		rc32434_pci->pcicfgd = *data;

	else
		*data = rc32434_pci->pcicfgd;
	rc32434_sync();

	/*
	 * Revisit: check for master or target abort.
	 */
	return 0;
}

static inline int
config_write(u8 bus, u8 devfn, u8 where, u32 data)
{
	return config_access(PCI_ACCESS_WRITE, bus, devfn, where, &data);
}

static inline int
config_read(u8 bus, u8 devfn, u8 where, u32 * data)
{
	return config_access(PCI_ACCESS_READ, bus, devfn, where, data);
}

/*
 * We can't address 8 and 16 bit words directly.  Instead we have to
 * read/write a 32bit word and mask/modify the data we actually want.
 */
static int
read_config_byte(struct pci_dev *dev, int where, u8 * val)
{
	u32 data = 0;

	if (config_read(dev->bus->number, dev->devfn, where, &data))
		return -1;

	*val = (data >> ((where & 3) << 3)) & 0xff;
	DBG("cfg read byte: bus %d dev_fn %x where %x: val %x\n",
	    dev->bus->number, dev->devfn, where, *val);

	return PCIBIOS_SUCCESSFUL;
}

static int
read_config_word(struct pci_dev *dev, int where, u16 * val)
{
	u32 data = 0;

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (config_read(dev->bus->number, dev->devfn, where, &data))
		return -1;

	*val = (data >> ((where & 3) << 3)) & 0xffff;
	DBG("cfg read word: bus %d dev_fn %x where %x: val %x\n",
	    dev->bus->number, dev->devfn, where, *val);

	return PCIBIOS_SUCCESSFUL;
}

static int
read_config_dword(struct pci_dev *dev, int where, u32 * val)
{
	u32 data = 0;

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (config_read(dev->bus->number, dev->devfn, where, &data))
		return -1;

	*val = data;
	DBG("cfg read dword: bus %d dev_fn %x where %x: val %x\n",
	    dev->bus->number, dev->devfn, where, *val);

	return PCIBIOS_SUCCESSFUL;
}

static int
write_config_byte(struct pci_dev *dev, int where, u8 val)
{
	u32 data = 0;

	if (config_read(dev->bus->number, dev->devfn, where, &data))
		return -1;

	data = (data & ~(0xff << ((where & 3) << 3))) |
	    (val << ((where & 3) << 3));

	if (config_write(dev->bus->number, dev->devfn, where, data))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}

static int
write_config_word(struct pci_dev *dev, int where, u16 val)
{
	u32 data = 0;

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (config_read(dev->bus->number, dev->devfn, where, &data))
		return -1;

	data = (data & ~(0xffff << ((where & 3) << 3))) |
	    (val << ((where & 3) << 3));

	if (config_write(dev->bus->number, dev->devfn, where, data))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}

static int
write_config_dword(struct pci_dev *dev, int where, u32 val)
{
	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (config_write(dev->bus->number, dev->devfn, where, val))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}

static struct pci_ops rc32434_pci_ops = {
	read_config_byte,
	read_config_word,
	read_config_dword,
	write_config_byte,
	write_config_word,
	write_config_dword
};

struct pci_channel mips_pci_channels[] = {
	{&rc32434_pci_ops, &rc32434_res_pci_io1,
	 &rc32434_res_pci_mem1, PCI_DEVFN(1, 0), PCI_DEVFN(6, 0)},
	{NULL, NULL, NULL, 0, 0}
};

unsigned __init int
pcibios_assign_all_busses(void)
{
	return 1;
}

static void
rc32434_dump_pci_bridge(void)
{
	int i;
	u32 val;
	printk("RC32434 PCI Bridge Config:\n");

	printk("PCI_MEM1_BASE: 0x%08x\n", rc32434_pci->pcilba[0].a);
	printk("PCI_IO1_BASE: 0x%08x\n", rc32434_pci->pcilba[1].a);
	printk("PCI_MEM2_BASE: 0x%08x\n", rc32434_pci->pcilba[2].a);
	printk("PCI_IO2_BASE:  0x%08x\n", rc32434_pci->pcilba[3].a);

	for (i = 0; i < 17; i++) {
		config_read(0, 0, i * 4, &val);
		printk("dword %d\t%08x\n", i, val);
	}
}

void __init
rc32434_pcibridge_init(void)
{

	unsigned int pcicValue, pcicData = 0;
	unsigned int dummyRead, pciCntlVal;
	printk("PCI: Initializing PCI\n");
	pcicValue = rc32434_pci->pcic;
	pcicValue = (pcicValue >> PCIM_SHFT) & PCIM_BIT_LEN;
	if (!((pcicValue == PCIM_H_EA) ||
	      (pcicValue == PCIM_H_IA_FIX) || (pcicValue == PCIM_H_IA_RR))) {
		/* Not in Host Mode, return ERROR */
		return;
	}

	/* Enables the Idle Grant mode, Arbiter Parking */
	pcicData |= (PCIC_igm_m | PCIC_eap_m | PCIC_en_m);
	rc32434_pci->pcic = pcicData;	/* Enable the PCI bus Interface */
	/* Zero out the PCI status & PCI Status Mask */
	for (;;) {
		pcicData = rc32434_pci->pcis;
		if (!(pcicData & PCIS_rip_m))
			break;
	}

	rc32434_pci->pcis = 0;
	rc32434_pci->pcism = 0xFFFFFFFF;
	/* Zero out the PCI decoupled registers */
	rc32434_pci->pcidac = 0;	/* disable PCI decoupled accesses at initialization */
	rc32434_pci->pcidas = 0;	/* clear the status */
	rc32434_pci->pcidasm = 0x0000007F;	/* Mask all the interrupts */
	/* Mask PCI Messaging Interrupts */
	rc32434_pci_msg->pciiic = 0;
	rc32434_pci_msg->pciiim = 0xFFFFFFFF;
	rc32434_pci_msg->pciioic = 0;
	rc32434_pci_msg->pciioim = 0;

	/* Setup PCILB0 as Memory Window */
	rc32434_pci->pcilba[0].a = (unsigned int) (PCI_ADDR_START);

	/* setup the PCI map address as same as the local address */

	rc32434_pci->pcilba[0].m = (unsigned int) (PCI_ADDR_START);

	/* Setup PCILBA1 as MEM */
	rc32434_pci->pcilba[0].c = ((SIZE_16MB & 0x1f) << PCILBAC_size_b);

	dummyRead = rc32434_pci->pcilba[0].c;	/* flush the CPU write Buffers */

	rc32434_pci->pcilba[1].a = 0x60000000;

	rc32434_pci->pcilba[1].m = 0x60000000;
	/* setup PCILBA2 as IO Window */
	rc32434_pci->pcilba[1].c = ((SIZE_256MB & 0x1f) << PCILBAC_size_b);

	dummyRead = rc32434_pci->pcilba[1].c;	/* flush the CPU write Buffers */
	rc32434_pci->pcilba[2].a = 0x18C00000;

	rc32434_pci->pcilba[2].m = 0x18FFFFFF;
	/* setup PCILBA2 as IO Window */
	rc32434_pci->pcilba[2].c = ((SIZE_4MB & 0x1f) << PCILBAC_size_b);

	dummyRead = rc32434_pci->pcilba[2].c;	/* flush the CPU write Buffers */

	rc32434_pci->pcilba[3].a = 0x18800000;

	rc32434_pci->pcilba[3].m = 0x18800000;
	rc32434_pci->pcilba[3].c = (((SIZE_1MB & 0x1ff) << PCILBAC_size_b) |
				    PCILBAC_msi_m);
	dummyRead = rc32434_pci->pcilba[2].c;	/* flush the CPU write Buffers */
	/* Setup PCILBA3 as IO Window */

	pciConfigAddr = (unsigned int) (0x80000004);
	for (loopCount = 0; loopCount < 24; loopCount++) {
		rc32434_pci->pcicfga = pciConfigAddr;
		dummyRead = rc32434_pci->pcicfga;
		rc32434_pci->pcicfgd = korinaCnfgRegs[loopCount];
		dummyRead = rc32434_pci->pcicfgd;
		pciConfigAddr += 4;
	}
	rc32434_pci->pcitc =
	    (unsigned int) ((PCITC_RTIMER_VAL & 0xff) << PCITC_rtimer_b) |
	    ((PCITC_DTIMER_VAL & 0xff) << PCITC_dtimer_b);

	pciCntlVal = rc32434_pci->pcic;
	pciCntlVal &= ~(PCIC_tnr_m);
	rc32434_pci->pcic = pciCntlVal;
	pciCntlVal = rc32434_pci->pcic;

#ifdef __MIPSEB__
#define ENDIANNESS_BIT 1
#else
#define ENDIANNESS_BIT 0
#endif
	rc32434_sync();
}

#endif				/* CONFIG_PCI */
