/*
 *
 * BRIEF MODULE DESCRIPTION
 *	Low level uart routines to directly access a 16550 uart.
 *
 * Author: Steve Longerbeam <stevel@mvista.com, or source@mvista.com>
 *
 * 2001 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/types.h>
#include <asm/rc32300/rc32300.h>

#define SERIAL_BASE RC32300_UART0_BASE

#define SER_CMD       (SERIAL_BASE + 0x14)
#define SER_DATA      (SERIAL_BASE + 0x00)
#define TX_BUSY       0x20

#define TIMEOUT       0xffff
#undef SLOW_DOWN

static const char digits[16] = "0123456789abcdef";

#ifdef SLOW_DOWN
static inline void slow_down()
{
	int k;
	for (k = 0; k < 10000; k++);
}
#else
#define slow_down()
#endif

void putch(const unsigned char c)
{
	unsigned char ch;
	int i = 0;

	do {
		ch = rc32300_readb(SER_CMD);
		slow_down();
		i++;
		if (i > TIMEOUT) {
			break;
		}
	} while (0 == (ch & TX_BUSY));
	rc32300_writeb(c, SER_DATA);
}

void puts(unsigned char *cp)
{
	unsigned char ch;
	int i = 0;

	while (*cp) {
		do {
			ch = rc32300_readb(SER_CMD);
			slow_down();
			i++;
			if (i > TIMEOUT) {
				break;
			}
		} while (0 == (ch & TX_BUSY));
		rc32300_writeb(*cp++, SER_DATA);
	}
	putch('\r');
	putch('\n');
}

void fputs(unsigned char *cp)
{
	unsigned char ch;
	int i = 0;

	while (*cp) {

		do {
			ch = rc32300_readb(SER_CMD);
			slow_down();
			i++;
			if (i > TIMEOUT) {
				break;
			}
		} while (0 == (ch & TX_BUSY));
		rc32300_writeb(*cp++, SER_DATA);
	}
}


void put64(uint64_t ul)
{
	int cnt;
	unsigned ch;

	cnt = 16;		/* 16 nibbles in a 64 bit long */
	putch('0');
	putch('x');
	do {
		cnt--;
		ch = (unsigned char) (ul >> cnt * 4) & 0x0F;
		putch(digits[ch]);
	} while (cnt > 0);
}

void put32(unsigned u)
{
	int cnt;
	unsigned ch;

	cnt = 8;		/* 8 nibbles in a 32 bit long */
	putch('0');
	putch('x');
	do {
		cnt--;
		ch = (unsigned char) (u >> cnt * 4) & 0x0F;
		putch(digits[ch]);
	} while (cnt > 0);
}
