/*
 *
 * BRIEF MODULE DESCRIPTION
 *    PROM library initialization code for the IDT 79S334A and 79EB355
 *    boards, assumes the boot code is IDT/sim.
 *
 * Author: Steve Longerbeam <stevel@mvista.com, or source@mvista.com>
 *
 * This file was derived from Carsten Langgaard's 
 * arch/mips/mips-boards/xx files.
 *
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 1999,2000 MIPS Technologies, Inc.  All rights reserved.
 *
 * 2001,2002 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/config.h>
#include <linux/kdev_t.h>
#include <linux/major.h>
#include <linux/console.h>
#include <asm/bootinfo.h>
#include <asm/page.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/bootmem.h>
#include <linux/ioport.h>
#include <asm/rc32300/rc32300.h>

char arcs_cmdline[CL_SIZE];

/* Kernel Boot parameters */
static unsigned char bootparm[]="ip=192.168.1.3:192.168.1.1::netmask";

#ifndef CONFIG_BOOT_FROM_PROM

#ifdef CONFIG_IDT_79EB355
/*
 * NVRAM on the 79EB355 is internal to the DS1511 RTC and uses
 * indexed addressing.
 */
static inline u8 nvreadb(int offset)
{
	writeb((u8)offset, &rtc->nvram_addr);
	return readb(&rtc->nvram_data);
}
#elif defined(CONFIG_IDT_79S334)
static inline u8 nvreadb(int offset)
{
	return *((u8*)KSEG1ADDR(NVRAM_BASE + offset));
}
#endif

#ifndef CONFIG_IDT_79RP355
static inline u8 env_readb(int env_index)
{
	return nvreadb(NVRAM_ENVSTART_OFF + env_index);
}

/*
 * Parses environment variable strings in NVRAM, copying strings
 * beginning with "bootparm?=" to arcs_cmdline[]. For example,
 *
 *    netaddr=10.0.1.95
 *    bootaddr=10.0.0.139
 *    bootfile=vmlinus
 *    bootparm1=root=/dev/nfs
 *    bootparm2=ip=10.0.1.95
 *
 * is parsed to:
 *
 *	root=/dev/nfs ip=10.0.1.95
 *
 * in arcs_cmdline[].
 */
static void prom_init_cmdline(void)
{
	int env_size, env_index, arcs_index;

	env_index = arcs_index = 0;

	/* stored size is 2 bytes, always big endian order */
	env_size = (int)((nvreadb(NVRAM_ENVSIZE_OFF) << 8) +
			 nvreadb(NVRAM_ENVSIZE_OFF+1));
	
	if (env_size < 0 || env_size > 512)
		return;	/* invalid total env size */

	while (env_index < env_size && arcs_index < sizeof(arcs_cmdline)) {
		char env_str[100];
		int i, arcs_len;
		/*
		  first byte is length of this env variable string,
		  including the length.
		*/
		int env_len = env_readb(env_index);
		int max_len = min(100, env_size - env_index);
		
		if (env_len == 0 || env_len > max_len)
			break;	/* invalid env variable size */
		
		/* copy the env string */
		for (i=0; i<env_len; i++)
			env_str[i] = env_readb(env_index + i);
		
		if (strncmp(&env_str[1], "bootparm", 8) == 0) {
			/*
			  copy to arcs, skipping over length byte and
			  "bootparm?=" string, a total of 11 chars.
			*/
			arcs_len = env_len - 11;

			/* will this string fit in arcs ? */
			if (arcs_index + arcs_len + 1 > sizeof(arcs_cmdline))
				break; /* nope */

			memcpy(&arcs_cmdline[arcs_index],
			       &env_str[11], arcs_len);
			arcs_index += arcs_len;
			/* add a blank between env variables */
			arcs_cmdline[arcs_index++] = ' ';
#ifdef CONFIG_IDT_79EB355
		} else if (strncmp(&env_str[1], "ethaddr", 7) == 0) {
			/* copy to arcs, skipping over length byte */
			arcs_len = env_len - 1;

			/* will this string fit in arcs ? */
			if (arcs_index + arcs_len + 1 > sizeof(arcs_cmdline))
				break; /* nope */

			memcpy(&arcs_cmdline[arcs_index],
			       &env_str[1], arcs_len);
			arcs_index += arcs_len;
			/* add a blank between env variables */
			arcs_cmdline[arcs_index++] = ' ';
#endif
		}
		
		/* increment to next prom env variable */
		env_index += env_len;
	}

	arcs_cmdline[arcs_index] = '\0';
}
#endif /* !CONFIG_IDT_79RP355 */
#endif /* !BOOT_FROM_PROM */

extern unsigned long mips_machgroup;
extern unsigned long mips_machtype;

const char *get_system_type(void)
{
#ifdef CONFIG_IDT_79EB355
	return "IDT 79EB355";
#elif defined(CONFIG_IDT_79RP355)
	return "IDT 79RP355";
#else
	return "IDT 79S334A";
#endif
}

struct resource rc32300_res_ram = {
	"RAM",
	0,
	RAM_SIZE,
	IORESOURCE_MEM
};

char * __init prom_getcmdline(void)
{
	return &(arcs_cmdline[0]);
}

int __init prom_init(int argc, char **argv, char **envp, int *prom_vec)
{
	/* set up command line */
#if defined(CONFIG_BOOT_FROM_PROM) || defined(CONFIG_IDT_79RP355)
	sprintf(arcs_cmdline,"%s\n",bootparm);
#else
	prom_init_cmdline();
#endif

	/* set our arch type */
	mips_machgroup = MACH_GROUP_IDT;

#ifdef CONFIG_IDT_79EB355
	mips_machtype = MACH_IDT79EB355;
#elif defined(CONFIG_IDT_79RP355)
	mips_machtype = MACH_IDT79RP355;
#else
	mips_machtype = MACH_IDT79S334;
#endif

	add_memory_region(0,
			  rc32300_res_ram.end - rc32300_res_ram.start,
			  BOOT_MEM_RAM);

	return 0;
}

void prom_free_prom_memory(void)
{
}

