/*
  * arch/mips/lib/rockhopper.c
  *
  * NEC Rockhopper baseboard initialisation.
  *
  * Author: MontaVista Software, Inc. <source@mvista.com>
  *
  * Based on:
  *     - arch/mips/ddb5xxx code by Jun Sun
  *     - BSB-VR0300 Rockhopper V2.1 Base Board User's Manual
  *       (NEC, February 6, 2002)
  *     - Rockhopper Base Board schematic diagram Rev. 2.2
  *       (NEC, June 24, 2002)
  *     - ALi M1535+ South Bridge and Power Management Controller (ver 1.20)
  *       (Acer Laboratories Inc.)
  *     - ALi M1535+/M1535D+ IDE, USB, AC-Link Audio and Modem Controller
  *       (Acer Laboratories Inc.)
  *     - ALi M1535/M1535D/M1535+/M1535D+ Super I/O with FIR Controller
  *       (Acer Laboratories Inc.)
  *
  * TODO/issues:
  *     - secondary IDE channel is not working. Unexpected IDE interrupt
  *       from secondary channel occured immediately after attaching the
  *       handler in IDE driver. Further investigation needed.
  *
  *     - floppy disk is not working on Rockhopper v2.1. It looks like this
  *       is due to missing pull-ups and wrong 3.3v pool-ups (5v pull-ups
  *       needed). It looks like this is corrected on Rockhopper v2.2, but
  *       it is not verified. 
  *
  *     - IrDA port is not working. It was not my goal to make it working,
  *       but I have tried to properly configure it in this file. Driver
  *       ali-ircc started up after this, but seems unusable.
  *
  *     - Audio controller is not usable, at least on board I am working with.
  *       During initialisation, M5451 resets the WM9707 AC'97 codec. Codec
  *       is AC'97 master, therefore it is expected that it will start to
  *       generate bit clocks for AC'97 interface. This is not happens, so
  *       timeout occured on attempt to access to AC'97.
  *
  * 2003 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */

#include <linux/config.h>

#include <linux/types.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/delay.h>

#include <asm/bootinfo.h>
#include <asm/irq.h>
#include <asm/vr7701.h>

#undef ROCKHOPPER_ENABLE_SECONDARY_IDE

/* 
 * M1535 IRQ mapping 
 * Feel free to change this, although it shouldn't be needed
 */
#define M1535_IRQ_INTA  7
#define M1535_IRQ_INTB  9
#define M1535_IRQ_INTC  10
#define M1535_IRQ_INTD  11

#define M1535_IRQ_USB   9
#define M1535_IRQ_IDE   14
#define M1535_IRQ_IDE2  15
#define M1535_IRQ_PS2   12
#define M1535_IRQ_RTC   8
#define M1535_IRQ_FDC   6
#define M1535_IRQ_AUDIO 5
#define M1535_IRQ_COM1  4
#define M1535_IRQ_COM2  4
#define M1535_IRQ_IRDA  3
#define M1535_IRQ_KBD   1
#define M1535_IRQ_TMR   0

/* Rockhopper "slots" assignment; this is hard-coded ... */
#define ROCKHOPPER_M5451_SLOT  6
#define ROCKHOPPER_M1535_SLOT  7
#define ROCKHOPPER_M5229_SLOT  16
#define ROCKHOPPER_M5237_SLOT  10
#define ROCKHOPPER_PMU_SLOT    28
/* ... and hard-wired. */
#define ROCKHOPPER_PCI1_SLOT   8
#define ROCKHOPPER_PCI2_SLOT   9
#define ROCKHOPPER_PCI3_SLOT   5
#define ROCKHOPPER_PCI4_SLOT   6
#define ROCKHOPPER_PCNET_SLOT  1

#define M1535_IRQ_MASK(n) (1 << (n))

#define M1535_IRQ_EDGE  (M1535_IRQ_MASK(M1535_IRQ_TMR)  | \
                         M1535_IRQ_MASK(M1535_IRQ_KBD)  | \
                         M1535_IRQ_MASK(M1535_IRQ_COM1) | \
                         M1535_IRQ_MASK(M1535_IRQ_COM2) | \
                         M1535_IRQ_MASK(M1535_IRQ_IRDA) | \
                         M1535_IRQ_MASK(M1535_IRQ_RTC)  | \
                         M1535_IRQ_MASK(M1535_IRQ_FDC)  | \
                         M1535_IRQ_MASK(M1535_IRQ_PS2))

#define M1535_IRQ_LEVEL (M1535_IRQ_MASK(M1535_IRQ_IDE)  | \
                         M1535_IRQ_MASK(M1535_IRQ_USB)  | \
                         M1535_IRQ_MASK(M1535_IRQ_INTA) | \
                         M1535_IRQ_MASK(M1535_IRQ_INTB) | \
                         M1535_IRQ_MASK(M1535_IRQ_INTC) | \
                         M1535_IRQ_MASK(M1535_IRQ_INTD))

/* configuration ports */
#define M1535_IO_CONFIG_PORT 0x3f0
#define M1535_IO_INDEX_PORT  0x3f0
#define M1535_IO_DATA_PORT   0x3f1

#define CONFIG_ENTER_KEY1 0x51
#define CONFIG_ENTER_KEY2 0x23
#define CONFIG_LEAVE_KEY  0xBB

#define CONFIG_SELECT     0x07

/* Interrupt level/edge control port */
#define M1535_IO_ELC1     0x4d0	/* Edge/Level control for master PIC */
#define M1535_IO_ELC2     0x4d1	/* Edge/Level control for slave PIC */

#define M1535_PMU_IO_BASE 0x500	/* Power management unit base I/O address */
#define M5229_IDE_IO_BASE 0xf000	/* Default IDE I/O base */

/* SuperIO controller logical devices */
#define M1535_SIO_DEV_FDC    0x00	/* Floppy */
#define M1535_SIO_DEV_PP     0x03	/* Parallel port */
#define M1535_SIO_DEV_UART1  0x04	/* UART 1 */
#define M1535_SIO_DEV_UART2  0x05	/* UART 2 */
#define M1535_SIO_DEV_KBD    0x07	/* Keyboard */
#define M1535_SIO_DEV_UART3  0x0B	/* UART 3 */

/* M1535 Configuration Registers */
#define M1535_PIC        0x40	/* PCI Interface Control */
#define M1535_PIC_PASV_REL  0x10	/* Passive release feature */
#define M1535_PIC_DEL_TR    0x08	/* Delayed transaction enable */
#define M1535_PIC_DMA_LB    0x01	/* DMA Line Buffer enable */

#define M1535_IORC       0x41	/* I/O Recovery Control */
#define M1535_IORC_PS2_KBD  0x80	/* PS/2 keyboard present */
#define M1535_IORC_PS2_MICE 0x40	/* PS/2 mouse present */
#define M1535_IORC_RP       0x3c	/* I/O recovery period */
#define M1535_IORC_REN      0x01	/* I/O recovery enable */

#define M1535_ISAC1      0x42	/* Internal ISA Bus Cycle Control */
#define M1535_ISAC2      0x43	/* Internal ISA Bus Cycle Control 2 */

#define M1535_IDEIRT     0x44	/* IDE Interrupt Routing Table */
#define M1535_IDEIRT_SRC    0x80	/* PCI Soft Reset Control */
#define M1535_IDEIRT_IO_DEC 0x40	/* On-chip I/O decode (1-subtractive) */
#define M1535_IDEIRT_INTEDG 0x10	/* IDE Interrupt signal transform to edge */
#define M1535_IDEIRT_INTRT  0x0F	/* IDE Interrupt route table */

#define M1535_PIPM       0x45	/* PCI Interrupt Polling Mode */
#define M1535_PIPM_PIPM_EN  0x80	/* PCI Interrupt Polling Mode enable */
#define M1535_PIPM_ROMKBCS  0x40	/* ROMKBCSJ activated when accessing
					   0x60 - 0x66 ports */
#define M1535_PIPM_DT_TIMO  0x08	/* Delayed transaction Timeout cntr En */
#define M1535_PIPM_DDMA     0x02	/* Distributed DMA Enable */
#define M1535_PIPM_PAR_EN   0x01	/* Parity Check Enable */

#define M1535_PIRT1      0x48	/* PCI Routing Table 1 */
#define M1535_PIRT2      0x49	/* PCI Routing Table 2 */
#define M1535_PIRT3      0x4a	/* PCI Routing Table 3 */
#define M1535_PIRT4      0x4b	/* PCI Routing Table 4 */

#define M1535_PCC        0x50	/* Parity Check Control */
#define M1535_PCC_IDE       0x4000	/* IDE parity check enable */
#define M1535_PCC_DMA       0x2000	/* DMA master parity check enable */
#define M1535_PCC_USB       0x1000	/* USB parity check enable */

#define M1535_USBDC      0x52	/* USB Device Control */
#define M1535_USBDC_HCEN    0x4000	/* On-chip USB host controller 1 enable */
#define M1535_USBDC_RTCEN   0x2000	/* RTC ports enable */

#define M1535_IDEIC      0x58	/* IDE Interface Control */
#define M1535_IDEIC_EN        0x40	/* IDE controller enable */
#define M1535_IDEIC_IDSEL_A27 0x00	/* IDSEL address select */
#define M1535_IDEIC_IDSEL_A26 0x10
#define M1535_IDEIC_IDSEL_A24 0x20
#define M1535_IDEIC_IDSEL_A15 0x30
#define M1535_IDEIC_SBEN      0x08	/* Secondary bus signals enable */
#define M1535_IDEIC_PBEN      0x04	/* Primary bus signals enable */

#define M1535_RAM        0x6d	/* ROM Address Mapping */
#define M1535_RAM_MUST_BE_1   0x40	/* Undocumented */
#define M1535_RAM_DMA_RREL    0x20	/* DMA Master Cycle Retry Release PCI bus */
#define M1535_RAM_PC_PCI_DMA  0x08	/* PC/PCI DMA function enable */

#define M1535_DDMAS      0x71

#define M1535_USBIDS     0x72	/* USB 1 IDSEL select */
#define M1535_USBIDS_IRQSYNC  0x80	/* IRQ Output synchronisation */
#define M1535_USBIDS_SICONT   0x20	/* Repeat serial IRQ continuous mode */
#define M1535_USBIDS_PMU_IDSEL      0x0c	/* PMU IDSEL addres select */
#define M1535_USBIDS_PMU_IDSEL_A28  0x00	/*    A28 */
#define M1535_USBIDS_PMU_IDSEL_A14  0x04	/*    A14 */
#define M1535_USBIDS_PMU_IDSEL_A17  0x08	/*    A17 */
#define M1535_USBIDS_PMU_IDSEL_A19  0x0c	/*    A19 */
#define M1535_USBIDS_USB_IDSEL      0x03	/* USB IDSEL addres select */
#define M1535_USBIDS_USB_IDSEL_A31  0x00	/*     A31 */
#define M1535_USBIDS_USB_IDSEL_A22  0x01	/*     A22 */
#define M1535_USBIDS_USB_IDSEL_A21  0x02	/*     A21 */
#define M1535_USBIDS_USB_IDSEL_A13  0x03	/*     A13 */

#define M1535_USBIR      0x74	/* USB Interrupt Routing Table */
#define M1535_IDEIRT2    0x75	/* Secondary IDE Interrupt selection */
#define M1535_IDEIRT2_INTEDG        0x10	/* IDE Interrupt signal transform 
						   to edge */
#define M1535_AMDC       0x77	/* Audio/Modem Device Control */
#define M1535_AMDC_MDM_DIS          0x40	/* M5453 Modem Disable */
#define M1535_AMDC_MDM_IDSEL        0x30	/* Modem IDSEL address select */
#define M1535_AMDC_MDM_IDSEL_A19    0x00	/*     A19 */
#define M1535_AMDC_MDM_IDSEL_A20    0x10	/*     A20 */
#define M1535_AMDC_MDM_IDSEL_A13    0x20	/*     A13 */
#define M1535_AMDC_MDM_IDSEL_A14    0x30	/*     A14 */
#define M1535_AMDC_AUD_DIS          0x08	/* M5451 Audio device Disable */
#define M1535_AMDC_AUD_IDSEL        0x06	/* Audio IDSEL address select */
#define M1535_AMDC_AUD_IDSEL_A17    0x00	/*     A17 */
#define M1535_AMDC_AUD_IDSEL_A19    0x02	/*     A19 */
#define M1535_AMDC_AUD_IDSEL_A14    0x04	/*     A14 */
#define M1535_AMDC_AUD_IDSEL_A15    0x06	/*     A15 */

#define M1535_USB2IDS    0x7D	/* USB 2 IDSEL select */
#define M1535_USB2IDS_USB_IDSEL      0xc0	/* USB IDSEL addres select */
#define M1535_USB2IDS_USB_IDSEL_A26  0x00	/*     A26 */
#define M1535_USB2IDS_USB_IDSEL_A21  0x40	/*     A21 */
#define M1535_USB2IDS_USB_IDSEL_A22  0x80	/*     A22 */
#define M1535_USB2IDS_USB_IDSEL_A17  0xc0	/*     A17 */

#define M1535_USB2DC     0x7E	/* USB 2 Device Control */
#define M1535_USB2DC_EN              0x80	/* USB2 Enable */
#define M1535_USB2DC_IDE_PLL_PWRDN   0x40	/* Enable IDE's PLL powerdown in S1 */
#define M1535_USB2DC_AUDIO_MAP_HIGH  0x01	/* Mapping audio master memory
						   access address region */

#define M5229_PRGIF      0x09	/* Programming interface */
#define M5229_PRGIF_CHSTAT_REPT      0x40	/* Device reporting IDE channel
						   status */
#define M5229_PRGIF_PRI_EN           0x20	/* Enable primary IDE */
#define M5229_PRGIF_SEC_EN           0x10	/* Enable secondary IDE */
#define M5229_PRGIF_SEC_FIXED        0x08	/* Secondary channel support fixed
						   mode of operation */
#define M5229_PRGIF_SEC_NATIVE       0x04	/* Secondary channel mode is native */
#define M5229_PRGIF_PRI_FIXED        0x02	/* Primary channel support fixed
						   mode of operation */
#define M5229_PRGIF_PRI_NATIVE       0x01	/* Primary channel mode is native */

#define M5229_UDMAR     0x4a	/* Ultra DMA 66 Control Register */
#define M5229_UDMAR_ATA_ATAPI4       0x4000	/* ATA/ATAPI-4 support enable */
#define M5229_UDMAR_UDMA_80C_CBLDET  0x0800	/* UltraDMA 80-conductor cable
						   detect enable/disable */
#define M5229_UDMAR_UDMA_EN          0x0100	/* UltraDMA Enable */
#define M5229_UDMAR_UDMA_BCDIS       0x0020	/* UltraDMA byte count disable */
#define M5229_UDMAR_SIMPLEX_RO       0x0004	/* Simplex bit option for IDE
						   bus master */
#define M5229_UDMAR_SCCD40           0x0002	/* Secondary channel cable det. */
#define M5229_UDMAR_PCCD40           0x0001	/* Primary channel cable det. */

#define M5229_CCA3       0x50	/* Class Code Attribute Register 3 */
#define M5229_CCA3_IDE_EN            0x01	/* Enable IDE function */
#define M5229_CCA3_PI                0x02	/* Return actual programming iface */

#define M5229_RAT        0x51	/* Reset and Testing register */
#define M5229_RAT_CHIPRST            0x80	/* Chip reset */
#define M5229_RAT_SOFTRST            0x40	/* Soft reset (except cfg space) */
#define M5229_RAT_RSTCH2             0x20	/* Channel 2 reset */
#define M5229_RAT_RSTCH1             0x10	/* Channel 1 reset */

#define M5229_CDRC       0x53	/* CD-ROM control */
#define M5229_CDRC_ID_RO             0x80	/* Sub-system vendor/dev is R/O */
#define M5229_CDRC_COMPAT_MASK       0x08	/* Mask base address during compat
						   mode */
#define M5229_CDRC_CDR_PIO_FIFO      0x02	/* CD-ROM FIFO operation in PIO
						   mode */
#define M5229_CDRC_CDR_DMA           0x01	/* Enable CD-ROM DMA mode */

#define M5229_UDMAP      0x56	/* Ultra DMA timing setting for Primary channl */
#define M5229_UDMAS      0x56	/* Ultra DMA timing setting for Primary channl */
#define M5229_UDMA_DEV1_EN          0x80	/* Enable Ultra DMA for device 1 */
#define M5229_UDMA_DEV1_MODE        0x70	/* UDMA mode for device 1 */
#define M5229_UDMA_DEV1_MODE4       0x00	/*     mode 4 */
#define M5229_UDMA_DEV1_MODE3       0x10	/*     mode 3 */
#define M5229_UDMA_DEV1_MODE2       0x20	/*     mode 2 */
#define M5229_UDMA_DEV1_MODE1       0x30	/*     mode 1 */
#define M5229_UDMA_DEV1_MODE0       0x40	/*     mode 0 */
#define M5229_UDMA_DEV1_MODE5       0x70	/*     mode 5 */
#define M5229_UDMA_DEV0_EN          0x08	/* Enable Ultra DMA for device 0 */
#define M5229_UDMA_DEV0_MODE        0x70	/* UDMA mode for device 0 */
#define M5229_UDMA_DEV0_MODE4       0x00	/*     mode 4 */
#define M5229_UDMA_DEV0_MODE3       0x01	/*     mode 3 */
#define M5229_UDMA_DEV0_MODE2       0x02	/*     mode 2 */
#define M5229_UDMA_DEV0_MODE1       0x03	/*     mode 1 */
#define M5229_UDMA_DEV0_MODE0       0x04	/*     mode 0 */
#define M5229_UDMA_DEV0_MODE5       0x07	/*     mode 5 */

#define M5229_PCAS       0x58	/* Primary channel address setup timing */
#define M5229_PCCB       0x59	/* Primary channel command block timing */
#define M5229_PCDT0      0x5A	/* Primary channel drive 0 data rw timing */
#define M5229_PCDT1      0x5B	/* Primary channel drive 1 data rw timing */
#define M5229_SCAS       0x5C	/* Secondary channel address setup timing */
#define M5229_SCCB       0x5D	/* Secondary channel command block timing */
#define M5229_SCDT0      0x5E	/* Secondary channel drive 0 data rw timing */
#define M5229_SCDT1      0x5F	/* Secondary channel drive 1 data rw timing */

static struct pci_ops *rockhopper_pci_ops = NULL;

static int intmap[] = { 0, 8, 0, 2, 4, 5, 7, 6, 0, 1, 3, 9, 11, 0, 13, 15 };

/* Write value to SuperIO PnP register */
static void __init
m1535_cfg(u8 device, u8 reg, u8 val)
{
	/* Enter config mode */
	outb(CONFIG_ENTER_KEY1, M1535_IO_CONFIG_PORT);
	outb(CONFIG_ENTER_KEY2, M1535_IO_CONFIG_PORT);

	/* Select device */
	outb(CONFIG_SELECT, M1535_IO_INDEX_PORT);
	outb(device, M1535_IO_DATA_PORT);

	/* Write to the register */
	outb(reg, M1535_IO_INDEX_PORT);
	outb(val, M1535_IO_DATA_PORT);

	/* Leave config mode */
	outb(CONFIG_LEAVE_KEY, M1535_IO_CONFIG_PORT);
}

static struct pci_dev *__init
ali_dev(int slot)
{
	static struct pci_dev __initdata dev;
	static struct pci_bus __initdata bus;
	bus.ops = rockhopper_pci_ops;
	bus.parent = NULL;
	dev.bus = &bus;
	dev.devfn = PCI_DEVFN(slot, 0);
	return &dev;
}

#define ali_pciread(dev, bwl, sfx, type) \
static inline type __init                                          \
m##dev##_pci_getcfg##sfx(u8 reg)                                   \
{                                                                  \
        type val;                                                  \
        pci_read_config_##bwl (ali_dev(ROCKHOPPER_M##dev##_SLOT),  \
                               reg, &val);                         \
        return val;                                                \
}

#define ali_pciwrite(dev, bwl, sfx, type) \
static inline void __init                                          \
m##dev##_pci_setcfg##sfx (u8 reg, type val)                        \
{                                                                  \
        pci_write_config_##bwl (ali_dev(ROCKHOPPER_M##dev##_SLOT), \
                                reg, val);                         \
}

#define ali_pcidev(dev) \
ali_pciread(dev, byte,, u8)      \
ali_pciread(dev, word, w, u16)   \
ali_pciread(dev, dword, l, u32)  \
ali_pciwrite(dev, byte,, u8)     \
ali_pciwrite(dev, word, w, u16)  \
ali_pciwrite(dev, dword, l, u32)

ali_pcidev(1535)
    ali_pcidev(5229)
    ali_pcidev(5451)
    ali_pcidev(5237)

/* M1535 keyboard/mouse initialisation */
static void __init
rockhopper_keyboard_mouse_init(void)
{
	/* Select PS/2 mouse IRQ number */
	m1535_cfg(M1535_SIO_DEV_KBD, 0x72, M1535_IRQ_PS2);

	/* Select PS/2-style mouse and keyboard */
	m1535_pci_setcfg(M1535_IORC, m1535_pci_getcfg(M1535_IORC) |
			 M1535_IORC_PS2_KBD | M1535_IORC_PS2_MICE);
}

/* M1535 floppy disk controller initialisation */
static void __init
rockhopper_floppy_init(void)
{
	/* Enable the FDC */
	m1535_cfg(M1535_SIO_DEV_FDC, 0x30, 1);

	/* Select the FDC IRQ */
	m1535_cfg(M1535_SIO_DEV_FDC, 0x70, M1535_IRQ_FDC);
}

/* M1535 UART (COM1 and COM2) initialisation */
static void __init
rockhopper_uart_init(void)
{
	printk("Configuring ALI M1535 Super I/O UART.\n");
	m1535_cfg(M1535_SIO_DEV_UART1, 0x30, 1);	/* Enable UART1 */
	m1535_cfg(M1535_SIO_DEV_UART2, 0x60, 0x03);	/* Select 0x3f8 addr */
	m1535_cfg(M1535_SIO_DEV_UART2, 0x61, 0xf8);
	m1535_cfg(M1535_SIO_DEV_UART1, 0x70, M1535_IRQ_COM1);
	m1535_cfg(M1535_SIO_DEV_UART1, 0xF0, 0);	/* No special features */
	m1535_cfg(M1535_SIO_DEV_UART1, 0xF1, 0);
	m1535_cfg(M1535_SIO_DEV_UART1, 0xF2, 0);

	m1535_cfg(M1535_SIO_DEV_UART2, 0x30, 0x81);	/* Enable UART2, FIR */
	m1535_cfg(M1535_SIO_DEV_UART2, 0x60, 0x03);	/* Select 0x300 addr */
	m1535_cfg(M1535_SIO_DEV_UART2, 0x61, 0x00);
	m1535_cfg(M1535_SIO_DEV_UART2, 0x70, M1535_IRQ_IRDA);
	m1535_cfg(M1535_SIO_DEV_UART2, 0x74, 0x01);	/* Using DMA1 */
	m1535_cfg(M1535_SIO_DEV_UART2, 0xF0, 0x00);	/* IBM tranciever,
							   no CVROFF,
							   SIR, no 8MHz */
	m1535_cfg(M1535_SIO_DEV_UART2, 0xF1, 0x0d);	/* IrDA, HDX, 
							   Tx - active hi
							   Rx - active lo */
	m1535_cfg(M1535_SIO_DEV_UART2, 0xF2, 0x06);	/* enable HDX timer */

	m1535_cfg(M1535_SIO_DEV_UART3, 0x30, 0x01);	/* Enable UART3 */
	m1535_cfg(M1535_SIO_DEV_UART3, 0x60, 0x02);	/* Select 0x2f8 addr */
	m1535_cfg(M1535_SIO_DEV_UART3, 0x61, 0xf8);
	m1535_cfg(M1535_SIO_DEV_UART3, 0x70, M1535_IRQ_COM2);
	m1535_cfg(M1535_SIO_DEV_UART3, 0xf0, 0x00);	/* Nothing special */
	m1535_cfg(M1535_SIO_DEV_UART3, 0xf1, 0x00);

	m1535_pci_setcfg(M1535_RAM, M1535_RAM_MUST_BE_1 | M1535_RAM_PC_PCI_DMA);
	m1535_pci_setcfg(M1535_DDMAS, 0xEF);
}

/* Set up PCI bus modes and interrupt routing */
static void __init
rockhopper_pci_init(void)
{
	/* 
	 * Interrupts from PCI peripheral are indicated by level;
	 * interrupts from legacy XT I/O traditionally indicated by edge.
	 * We MUST to configure level->edge transition for level
	 * interrupts.
	 */
	if (M1535_IRQ_LEVEL & M1535_IRQ_EDGE)
		panic("Rockhopper interrupt selected both as level and edge\n");
	outb(M1535_IRQ_LEVEL & 0xFF, M1535_IO_ELC1);
	outb((M1535_IRQ_LEVEL >> 8) & 0xFF, M1535_IO_ELC2);

	/* Setup interrupt mappings */
	m1535_pci_setcfg(M1535_PIRT1, (intmap[M1535_IRQ_INTA] << 0) |
			 (intmap[M1535_IRQ_INTB] << 4));
	m1535_pci_setcfg(M1535_PIRT2, (intmap[M1535_IRQ_INTC] << 0) |
			 (intmap[M1535_IRQ_INTD] << 4));
	m1535_pci_setcfg(M1535_PIRT3, 0);
	m1535_pci_setcfg(M1535_PIRT4, intmap[M1535_IRQ_AUDIO] << 4);

	/* Disable modem, enable audio */
	m1535_pci_setcfg(M1535_AMDC, M1535_AMDC_MDM_DIS |
			 M1535_AMDC_AUD_IDSEL_A17);
}

/* enable and configure IDE controller on Ali M1535 chip */
static void __init
rockhopper_ide_init(void)
{
	m5229_pci_setcfg(M5229_RAT, M5229_RAT_CHIPRST);
	udelay(1000);

	/* Enable IDE controller */
	m1535_pci_setcfg(M1535_IDEIC, M1535_IDEIC_EN |
			 M1535_IDEIC_IDSEL_A27 | M1535_IDEIC_PBEN
#ifdef ROCKHOPPER_ENABLE_SECONDARY_IDE
			 | M1535_IDEIC_SBEN
#endif
	    );

	/* IDE interrupt routing */
	m1535_pci_setcfg(M1535_IDEIRT, intmap[M1535_IRQ_IDE]);
	m1535_pci_setcfg(M1535_IDEIRT2, intmap[M1535_IRQ_IDE2]);

	/* Enable IDE parity check */
	m1535_pci_setcfgw(M1535_PCC, m1535_pci_getcfgw(M1535_PCC) |
			  M1535_PCC_IDE);

	m5229_pci_setcfg(M5229_PRGIF, M5229_PRGIF_CHSTAT_REPT |
			 M5229_PRGIF_PRI_EN |
			 M5229_PRGIF_SEC_EN |
			 M5229_PRGIF_PRI_FIXED |
			 M5229_PRGIF_SEC_FIXED |
			 M5229_PRGIF_PRI_NATIVE | M5229_PRGIF_SEC_NATIVE);

	m5229_pci_setcfg(M5229_CCA3, m5229_pci_getcfg(M5229_CCA3) |
			 M5229_CCA3_IDE_EN | M5229_CCA3_PI);

	/* Enable UDMA, select 40-pin connectors */
	m5229_pci_setcfgw(M5229_UDMAR, M5229_UDMAR_ATA_ATAPI4 |
			  M5229_UDMAR_UDMA_EN |
			  M5229_UDMAR_UDMA_BCDIS | M5229_UDMAR_UDMA_80C_CBLDET);

	m5229_pci_setcfg(M5229_CDRC, M5229_CDRC_ID_RO |
			 M5229_CDRC_COMPAT_MASK |
			 M5229_CDRC_CDR_PIO_FIFO | M5229_CDRC_CDR_DMA);

	m5229_pci_setcfg(M5229_UDMAP, M5229_UDMA_DEV0_EN |
			 M5229_UDMA_DEV0_MODE4 |
			 M5229_UDMA_DEV1_EN | M5229_UDMA_DEV1_MODE4);
	m5229_pci_setcfg(M5229_UDMAS, M5229_UDMA_DEV0_EN |
			 M5229_UDMA_DEV0_MODE4 |
			 M5229_UDMA_DEV1_EN | M5229_UDMA_DEV1_MODE4);

	m5229_pci_setcfg(M5229_PCAS, 0x02);
	m5229_pci_setcfg(M5229_PCCB, 0x22);
	m5229_pci_setcfg(M5229_PCDT0, 0x22);
	m5229_pci_setcfg(M5229_PCDT1, 0x22);
	m5229_pci_setcfg(M5229_SCAS, 0x02);
	m5229_pci_setcfg(M5229_SCCB, 0x22);
	m5229_pci_setcfg(M5229_SCDT0, 0x22);
	m5229_pci_setcfg(M5229_SCDT1, 0x22);

	m5229_pci_setcfgl(0x20, M5229_IDE_IO_BASE);
}

static void __init
rockhopper_usb_init(void)
{
	/* Enable USB parity check */
	m1535_pci_setcfgw(M1535_PCC, m1535_pci_getcfgw(M1535_PCC) |
			  M1535_PCC_USB);
	/* 
	 * Enable USB controller1, disable USB controller 2.
	 * Rockhopper wire USB connectors to USB Pad 1 and USB Pad 2.
	 * According to M1535 documentation Pad1 and Pad2 can be
	 * connected to the same USB controller only. We select USB1;
	 * USB1 port 1 connected to Pad1 and USB1 port 2 connected to
	 * Pad2.
	 */
	m1535_pci_setcfgw(M1535_USBDC, m1535_pci_getcfgw(M1535_USBDC) &
			  ~M1535_USBDC_HCEN);
	m1535_pci_setcfg(M1535_USB2DC, m1535_pci_getcfg(M1535_USB2DC) &
			 ~M1535_USB2DC_EN);

	m1535_pci_setcfg(M1535_USBIDS, (m1535_pci_getcfg(M1535_USBIDS) &
					~M1535_USBIDS_USB_IDSEL) |
			 M1535_USBIDS_USB_IDSEL_A21);

	/* Select USB interrupts */
	m1535_pci_setcfg(M1535_USBIR, intmap[M1535_IRQ_USB]);
}

static void __init
rockhopper_pmu_init(void)
{
	pci_write_config_word(ali_dev(ROCKHOPPER_PMU_SLOT), 0xE0,
			      M1535_PMU_IO_BASE);
}

int __init
rockhopper_detect(struct pci_ops *ops)
{
	int detected = 0;
	int i;
	rockhopper_pci_ops = ops;
	/* May be my board was slightly broken... */
	for (i = 0; !detected && i < 10; i++)
		detected |= (m1535_pci_getcfgw(0) == PCI_VENDOR_ID_AL);
	printk("Rockhopper detected: %s\n", detected ? "yes" : "no");
	return detected;
}

void __init
rockhopper_init(struct pci_ops *ops)
{
	rockhopper_pci_ops = ops;
	printk("Configuring Rockhopper baseboard/Ali M1535+ south bridge\n");
	rockhopper_keyboard_mouse_init();
	rockhopper_uart_init();
	rockhopper_floppy_init();
	rockhopper_pci_init();
	rockhopper_ide_init();
	rockhopper_usb_init();
	rockhopper_pmu_init();
}

struct irq_map_entry {
	u16 bus;
	u8 slot;
	u8 irq;
};
static struct irq_map_entry int_map[] = {
	{0, ROCKHOPPER_M5451_SLOT, M1535_IRQ_AUDIO},	/* Audio controller */
	{0, ROCKHOPPER_PCI1_SLOT, M1535_IRQ_INTD},	/* PCI slot #1 */
	{0, ROCKHOPPER_PCI2_SLOT, M1535_IRQ_INTC},	/* PCI slot #2 */
	{0, ROCKHOPPER_M5237_SLOT, M1535_IRQ_USB},	/* USB host controller */
	{0, ROCKHOPPER_M5229_SLOT, M1535_IRQ_IDE},	/* IDE controller */
	{1, ROCKHOPPER_PCNET_SLOT, M1535_IRQ_INTD},	/* AMD Am79c973 on-board 
							   ethernet */
	{1, ROCKHOPPER_PCI3_SLOT, M1535_IRQ_INTB},	/* PCI slot #3 */
	{1, ROCKHOPPER_PCI4_SLOT, M1535_IRQ_INTC}	/* PCI slot #4 */
};

static int pci_intlines[] =
    { M1535_IRQ_INTA, M1535_IRQ_INTB, M1535_IRQ_INTC, M1535_IRQ_INTD };

/* Determine the Rockhopper IRQ line number for the PCI device */
int __init
rockhopper_pci_irq(struct pci_dev *dev)
{
	u8 pin;
	struct pci_bus *bus;
	int slot;
	int i;

	pci_read_config_byte(dev, PCI_INTERRUPT_PIN, &pin);
	if (pin == 0 || pin > 4)
		pin = 1;

	bus = dev->bus;
	slot = PCI_SLOT(dev->devfn);

	/* 
	 * Board installed into rockhopper may have another PCI bridge.
	 * The code below assumes that bridge uses traditional interrupt
	 * swizzle.
	 */
	while (bus != NULL && bus->number > 1) {
		slot = PCI_SLOT(bus->self->devfn);
		bus = bus->parent;
		pin = (((pin - 1) + slot) % 4) + 1;
	}
	if (bus == NULL)
		return -1;

	for (i = 0; i < sizeof (int_map) / sizeof (int_map[0]); i++) {
		if (int_map[i].bus == bus->number && int_map[i].slot == slot) {
			int line;
			for (line = 0; line < 4; line++)
				if (pci_intlines[line] == int_map[i].irq)
					break;
			if (line < 4)
				return pci_intlines[(line + (pin - 1)) % 4];
			else
				return int_map[i].irq;
		}
	}
	return -1;
}
