/*
 * arch/mips/intrinsity/common/pci.c
 *
 * Intrinsity FastMATH - specific PCI support
 *
 * Author: MontaVista Software, Inc. <source@mvista.com>
 *
 * 2004 (c) MontaVista, Software, Inc.  This file is licensed under the terms
 * of the GNU General Public License version 2.  This program is licensed
 * "as is" without any warranty of any kind, whether express or implied.
 */

/*
 * This module provides the glue between Linux's PCI subsystem
 * and the hardware.  We basically provide glue for accessing
 * configuration space, and set up the translation for I/O
 * space accesses.
 */
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/mm.h>

#include <asm/intrinsity/fastmath.h>
#include <asm/io.h>
#include <asm/pci_channel.h>

extern int pciauto_assign_resources(int busno, struct pci_channel *hose);

#define	FM_PCI_BUSNUM_BASE	2

/*
 * This macro calculates the offset into config space where
 * a given bus, device/function, and offset live on the fm
 */

#define CFGOFFSET(bus,devfn,where) ((((bus)+FM_PCI_BUSNUM_BASE)<<16)+((devfn)<<8)+(where))

/*
 * Using the above offset, this macro calcuates the actual
 * address.
 */
#define CFGADDR(dev,where) (FM_PCI_CONFIG_BASE + CFGOFFSET(dev->bus->number,dev->devfn,where))

/*
 * Read/write 32-bit values in config space.
 */

#define READCFG32(addr)       swab32(*(u32 *)((addr)&~3))
#define WRITECFG32(addr,data) *(u32 *)((addr)&~3) = swab32(data)

/*
 * Read/write access functions for various sizes of values
 * in config space.
 */

static int fm_pci_read_config_byte(struct pci_dev *dev, int where, u8 * val)
{
	u32 data = 0;
	u32 cfgaddr = CFGADDR(dev, where);

	data = READCFG32(cfgaddr);
	*val = (data >> ((where & 3) << 3)) & 0xff;

	return PCIBIOS_SUCCESSFUL;
}

static int fm_pci_read_config_word(struct pci_dev *dev, int where, u16 * val)
{
	u32 data = 0;
	u32 cfgaddr = CFGADDR(dev, where);

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	data = READCFG32(cfgaddr);
	*val = (data >> ((where & 3) << 3)) & 0xffff;

	return PCIBIOS_SUCCESSFUL;
}

static int fm_pci_read_config_dword(struct pci_dev *dev, int where, u32 * val)
{
	u32 data = 0;
	u32 cfgaddr = CFGADDR(dev, where);

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	data = READCFG32(cfgaddr);
	*val = data;

	/* a hack to prevent the tsi400 from being reconfigured */
	if (where == PCI_VENDOR_ID) {
		u32 vendor;
		u32 device;
		vendor = data & 0xffff;
		device = (data >> 16) & 0xffff;
		if (vendor == 0x10e3 && device == 0x0400)
			return -1;
	}

	return PCIBIOS_SUCCESSFUL;
}

static int fm_pci_write_config_byte(struct pci_dev *dev, int where, u8 val)
{
	u32 data = 0;
	u32 cfgaddr = CFGADDR(dev, where);

	data = READCFG32(cfgaddr);

	data = (data & ~(0xff << ((where & 3) << 3))) |
	    (val << ((where & 3) << 3));

	WRITECFG32(cfgaddr, data);

	return PCIBIOS_SUCCESSFUL;
}

static int fm_pci_write_config_word(struct pci_dev *dev, int where, u16 val)
{
	u32 data = 0;
	u32 cfgaddr = CFGADDR(dev, where);

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	data = READCFG32(cfgaddr);

	data = (data & ~(0xffff << ((where & 3) << 3))) |
	    (val << ((where & 3) << 3));

	WRITECFG32(cfgaddr, data);

	return PCIBIOS_SUCCESSFUL;
}

static int fm_pci_write_config_dword(struct pci_dev *dev, int where, u32 val)
{
	u32 cfgaddr = CFGADDR(dev, where);

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	WRITECFG32(cfgaddr, val);

	return PCIBIOS_SUCCESSFUL;
}

struct pci_ops fm_pci_ops = {
	fm_pci_read_config_byte,
	fm_pci_read_config_word,
	fm_pci_read_config_dword,
	fm_pci_write_config_byte,
	fm_pci_write_config_word,
	fm_pci_write_config_dword
};

static struct resource pci_io_resource = {
	"pci IO space",
	FM_PCI_IO_BASE,
	FM_PCI_IO_BASE + FM_PCI_SIZE - 1,
	IORESOURCE_IO
};

static struct resource pci_mem_resource = {
	"pci memory space",
	FM_PCI_MEM_BASE,
	FM_PCI_MEM_BASE + FM_PCI_SIZE - 1,
	IORESOURCE_MEM
};

struct pci_channel mips_pci_channels[] = {
	{&fm_pci_ops, &pci_io_resource, &pci_mem_resource, 0, 1},
	{(struct pci_ops *) NULL, (struct resource *) NULL,
	 (struct resource *) NULL, (int) NULL, (int) NULL}
};

void __init pcibios_fixup(void)
{
}

void __init pcibios_fixup_irqs(void)
{
}

void __init pcibios_fixup_resources(struct pci_dev *dev)
{
}

unsigned int pcibios_assign_all_busses(void)
{
	return 1;
}
