/*
 * arch/mips/intrinsity/common/irq.c
 *
 * Intrinsity FastMips/FastMATH interrupt routines.
 *
 * Author: MontaVista Software, Inc. <ppopov@mvista.com> or <source@mvista.com>
 *
 * 2001-2004 (c) MontaVista, Software, Inc.  This file is licensed under the 
 * terms of the GNU General Public License version 2.  This program is licensed
 * "as is" without any warranty of any kind, whether express or implied.
 */
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel_stat.h>
#include <linux/module.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/timex.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/delay.h>

#include <asm/bitops.h>
#include <asm/bootinfo.h>
#include <asm/io.h>
#include <asm/mipsregs.h>
#include <asm/system.h>
#include <asm/intrinsity/fastmath.h>

extern void breakpoint(void);
extern asmlinkage void fm_IRQ(void);
extern void set_debug_traps(void);
extern irq_cpustat_t irq_stat[NR_CPUS];
unsigned int local_bh_count[NR_CPUS];
unsigned int local_irq_count[NR_CPUS];

extern unsigned int do_IRQ(int irq, struct pt_regs *regs);
extern void __init init_generic_irq(void);

static void fm_setup_local_irq(unsigned int irq_nr, struct hw_interrupt_type *h)
{
	if (irq_nr > FM_MAX_INTR)
		return;
	irq_desc[irq_nr].handler = h;
	asm("sync");
}

static void enable_irq_gpio(unsigned int irq)
{
	switch (irq) {
	case FM_IRQ_UART:
		FPGA(CONFIG) |= FM_FPGA_UART_ENB;
		break;
	case FM_IRQ_COUNT:
		FPGA(CONFIG) |= FM_FPGA_COUNT_ENB;
		break;
	case FM_IRQ_I2C:
		FPGA(CONFIG) |= FM_FPGA_I2C_ENB;
		break;
	case FM_IRQ_LM86_ALERT:
		FPGA(CONFIG) |= FM_FPGA_LM86_ALERT_ENB;
		break;
	case FM_IRQ_PCI_A:
		FPGA(CONFIG) |= FM_FPGA_PCI_A_ENB;
		break;
	}
	asm("sync");
}

static void disable_irq_gpio(unsigned int irq)
{
	switch (irq) {
	case FM_IRQ_UART:
		FPGA(CONFIG) &= ~FM_FPGA_UART_ENB;
		break;
	case FM_IRQ_COUNT:
		FPGA(CONFIG) &= ~FM_FPGA_COUNT_ENB;
		break;
	case FM_IRQ_I2C:
		FPGA(CONFIG) &= ~FM_FPGA_I2C_ENB;
		break;
	case FM_IRQ_LM86_ALERT:
		FPGA(CONFIG) &= ~FM_FPGA_LM86_ALERT_ENB;
		break;
	case FM_IRQ_PCI_A:
		FPGA(CONFIG) &= ~FM_FPGA_PCI_A_ENB;
		break;
	}
	asm("sync");
}

static unsigned int startup_irq_gpio(unsigned int irq)
{
	GPIO(STATUS) = FM_GPIO_STATUS_LL_INT | FM_GPIO_STATUS_INT;
	asm("sync");
	enable_irq_gpio(irq);
	return 0;		/* Never anything pending  */
}

static void shutdown_irq_gpio(unsigned int irq)
{
	disable_irq_gpio(irq);
	GPIO(STATUS) = FM_GPIO_STATUS_LL_INT | FM_GPIO_STATUS_INT;
	asm("sync");
}

static void mask_and_ack_irq_gpio(unsigned int irq)
{
	disable_irq_gpio(irq);
	GPIO(STATUS) = FM_GPIO_STATUS_LL_INT | FM_GPIO_STATUS_INT;

	if (irq == FM_IRQ_COUNT)	/* special handling here */
		FPGA(CONFIG) |= FM_FPGA_COUNT_RESET;

	asm("sync");
}

static void end_irq_gpio(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		enable_irq_gpio(irq);
}

static struct hw_interrupt_type gpio_irq_type = {
	"GPIO",
	startup_irq_gpio,
	shutdown_irq_gpio,
	enable_irq_gpio,
	disable_irq_gpio,
	mask_and_ack_irq_gpio,
	end_irq_gpio,
	NULL
};

static void enable_irq_dma(unsigned int irq)
{
	set_c0_status(FM_IE_DMA);
}

static void disable_irq_dma(unsigned int irq)
{
	clear_c0_status(FM_IE_DMA);
}

static unsigned int startup_irq_dma(unsigned int irq)
{
	enable_irq_dma(irq);
	return 0;		/* Never anything pending  */
}

#define shutdown_irq_dma		disable_irq_dma
#define mask_and_ack_irq_dma	disable_irq_dma

static void end_irq_dma(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		enable_irq_dma(irq);
}

static struct hw_interrupt_type dma_irq_type = {
	"DMA",
	startup_irq_dma,
	shutdown_irq_dma,
	enable_irq_dma,
	disable_irq_dma,
	mask_and_ack_irq_dma,
	end_irq_dma,
	NULL
};

void __init init_IRQ(void)
{
	memset(irq_desc, 0, sizeof (irq_desc));
	set_except_vector(0, fm_IRQ);
	init_generic_irq();

	/* associate all interrupts with a handler type */
	fm_setup_local_irq(FM_IRQ_UART, &gpio_irq_type);
	fm_setup_local_irq(FM_IRQ_COUNT, &gpio_irq_type);
	fm_setup_local_irq(FM_IRQ_I2C, &gpio_irq_type);
	fm_setup_local_irq(FM_IRQ_LM86_ALERT, &gpio_irq_type);
	fm_setup_local_irq(FM_IRQ_PCI_A, &gpio_irq_type);
	fm_setup_local_irq(FM_IRQ_DMA, &dma_irq_type);

	/* setup, then enable all interrupts */
	fm_fpga_interrupt_setup();
	asm("sync");		/* make sure fpga changes are done before gpio setup */
	fm_gpio_interrupt_setup();
	fm_dma_interrupt_setup();
	set_c0_status(ALLINTS);

#ifdef CONFIG_REMOTE_DEBUG
	/* If local serial I/O used for debug port, enter kgdb at once */
	puts("Waiting for kgdb to connect...");
	set_debug_traps();
	breakpoint();
#endif
}

void fm_dma_interrupt(struct pt_regs *regs)
{
	do_IRQ(FM_IRQ_DMA, regs);
}

void fm_gpio_interrupt(struct pt_regs *regs)
{
	unsigned int fpga_status;
	unsigned int fpga_config;

	fpga_status = FPGA(STATUS);
	fpga_config = FPGA(CONFIG);
	/* PCI-A? */
	if ((fpga_status & FM_FPGA_PCI_A_INT)
	    && (fpga_config & FM_FPGA_PCI_A_ENB))
		do_IRQ(FM_IRQ_PCI_A, regs);
	/* UART? */
	else if ((fpga_status & FM_FPGA_UART_INT)
		 && (fpga_config & FM_FPGA_UART_ENB))
		do_IRQ(FM_IRQ_UART, regs);
	/* Counter? */
	else if ((fpga_status & FM_FPGA_COUNT_INT)
		 && (fpga_config & FM_FPGA_COUNT_ENB))
		do_IRQ(FM_IRQ_COUNT, regs);
	/* I2C? */
	else if ((fpga_status & FM_FPGA_I2C_INT)
		 && (fpga_config & FM_FPGA_I2C_ENB))
		do_IRQ(FM_IRQ_I2C, regs);
	/* LM86 Alert? */
	else if ((fpga_status & FM_FPGA_LM86_ALERT_INT)
		 && (fpga_config & FM_FPGA_LM86_ALERT_ENB))
		do_IRQ(FM_IRQ_LM86_ALERT, regs);
	else {
		/* try to clear it anyway */
		GPIO(STATUS) |= FM_GPIO_STATUS_INT;
		asm("sync");
	}
}
