/*
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Code to handle ATI XILLEON IRQs
 *
 * Copyright (C) 1992 Linus Torvalds
 * Copyright (C) 1994 - 2000 Ralf Baechle
 * Copyright (C) 2001 Robert Lembree
 */
#include <linux/kernel.h>
#include <linux/irq.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/kernel_stat.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/random.h>
#include <linux/sched.h>

#include <asm/ati/xilleon.h>
#include <asm/ati/xilleonint.h>
#include <asm/ati/xilleonreg_kernel.h>
#include <linux/pci_ids.h>

#include <asm/gdb-stub.h>

#include <asm/system.h>

#if 0
#define DEBUG_INT(x...) printk(x)
#else
#define DEBUG_INT(x...)
#endif

extern unsigned long chipId;

extern asmlinkage void xilleonIRQ(void);
extern unsigned int do_IRQ(int irq, struct pt_regs *regs);

#ifdef CONFIG_GDB_CONSOLE
extern void __init register_gdb_console(void);
#endif

static unsigned int cached_pcu_int_mask =  0x0;
static unsigned int cached_ext_pci_int_mask = 0x0;
static unsigned int cached_sb_int_mask = 0x0;

static void enable_xilleon_irq(unsigned int irq_nr);
extern void __init init_generic_irq(void);
extern void __init prom_printf(char *fmt, ...);

#ifdef DBG_BSP_TRACE_ENABLE
extern void dbg_trace_to_fb(const char *fmt, ...);
extern long long dbg_trace_get_timestamp64(void);
#else
#   define dbg_trace_to_fb(x...)
#   define dbg_trace_get_timestamp64() (0)
#endif
extern void xilleon_timer_interrupt(struct pt_regs *regs);
void xilleon_irqdispatch_pcu(struct pt_regs *regs);

#define ALLINTS     (IE_IRQ1 | IE_IRQ2 | IE_IRQ3 | IE_IRQ4 | IE_IRQ5)

static unsigned int startup_xilleon_irq(unsigned int irq_nr)
{

	enable_xilleon_irq(irq_nr);
	
	return 0; /* never anything pending */
}

#define shutdown_xilleon_irq	disable_xilleon_irq

static void disable_xilleon_irq(unsigned int irq_nr)
{
	unsigned long flags;
	u32 mask;
	int aChipId;
	
	DEBUG_INT("disabling IRQ %d\n", irq_nr);

	save_and_cli(flags);

	aChipId = GETFLD_REGMM16(HBIU_DEVICE_ID, DEVICE_ID);
	
	if (irq_nr < EXTPCI_IRQBASE) {            // handle the southbridge
		
		if (irq_nr >= SB_IRQBASE + NUM_SB_IRQS)
			return;
		
		irq_nr -= SB_IRQBASE;
		
		cached_sb_int_mask &= ~(1 << irq_nr);
		SETREG_REGMM32(MIPS_SB_INT_CNTL, cached_sb_int_mask);
		
	} else if (irq_nr < FIRMWARE_IRQBASE) {   // handle the ext PCI

		if (irq_nr >= EXTPCI_IRQBASE + NUM_EXT_PCI_IRQS)
			return;
		
		irq_nr -= EXTPCI_IRQBASE;
		
		cached_ext_pci_int_mask &= ~(1 << irq_nr);
        
		if (XK_CHIP_X225_A11(aChipId))
			SETREG_REGMM32(MIPS_EXT_PCI_INT_CNTL, (cached_ext_pci_int_mask | MIPS_EXT_PCI_INT_CNTL__MIPS_INTR2_EN__MASK));
		else
			SETREG_REGMM32(MIPS_EXT_PCI_INT_CNTL, cached_ext_pci_int_mask);
		
	} else if (irq_nr < PCU_IRQBASE) {        // handle the firmware
		;

	} else if (irq_nr < NR_IRQS) {            // handle the PCU
		
		if (irq_nr >= PCU_IRQBASE + NUM_PCU_IRQS)
			return;
		
		irq_nr -= PCU_IRQBASE;
		
		mask = GETREG_REGMM32(MIPS_PCU_INT_CNTL);
		mask &= ~(1 << irq_nr);

		if (XK_CHIP_X225_A11(aChipId)) {
			cached_pcu_int_mask &= ~(1 << irq_nr);
			SETREG_REGMM32(PCU_INT_CNTL, cached_pcu_int_mask);

		} else
			SETREG_REGMM32(MIPS_PCU_INT_CNTL, mask);
		
	} else {                               // got a bad IRQ
		DEBUG_INT("Got bad IRQ: %d\n", irq_nr);
	}
	
	
	restore_flags(flags);
}

static void enable_xilleon_irq(unsigned int irq_nr)
{
	unsigned long flags;
	u32 mask;
	int aChipId;
	
	save_and_cli(flags);

	aChipId = GETFLD_REGMM16(HBIU_DEVICE_ID, DEVICE_ID);

	DEBUG_INT("enabling IRQ %d\n", irq_nr);
	
	if (irq_nr < EXTPCI_IRQBASE) {            // handle the southbridge
		
		if (irq_nr >= SB_IRQBASE + NUM_SB_IRQS)
			return;
		
		irq_nr -= SB_IRQBASE;

		cached_sb_int_mask |= (1 << irq_nr);
		SETREG_REGMM32(MIPS_SB_INT_CNTL, cached_sb_int_mask);
		
	} else if (irq_nr < FIRMWARE_IRQBASE) {   // handle the ext PCI
		
		if (irq_nr >= EXTPCI_IRQBASE + NUM_EXT_PCI_IRQS)
			return;
		
		irq_nr -= EXTPCI_IRQBASE;
		
		cached_ext_pci_int_mask |= (1 << irq_nr);

		if (XK_CHIP_X225_A11(aChipId))
			SETREG_REGMM32(MIPS_EXT_PCI_INT_CNTL, (cached_ext_pci_int_mask | MIPS_EXT_PCI_INT_CNTL__MIPS_INTR2_EN__MASK));
		else
			SETREG_REGMM32(MIPS_EXT_PCI_INT_CNTL, cached_ext_pci_int_mask);
		
	} else if (irq_nr < PCU_IRQBASE) {        // handle the firmware
		;

	} else if (irq_nr < NR_IRQS) {            // handle the PCU
		
		if (irq_nr >= PCU_IRQBASE + NUM_PCU_IRQS)
			return;
		
		irq_nr -= PCU_IRQBASE;
		
		mask = GETREG_REGMM32(MIPS_PCU_INT_CNTL);
        
		mask  |= (1 << irq_nr);
        
		if (XK_CHIP_X225_A11(aChipId))
			cached_pcu_int_mask  |= (1 << irq_nr);

		SETREG_REGMM32(MIPS_PCU_INT_CNTL, mask);
		SETREG_REGMM32(PCU_INT_CNTL, cached_pcu_int_mask);
		SETFLAG_REGMM32(GEN_INT_CNTL, PCU_INT_EN);
		SETFLAG_REGMM32(MIPS_EXT_PCI_INT_CNTL, MIPS_INTR2_EN);

        } else { // got a bad IRQ
		printk("Got bad IRQ: %d\n", irq_nr);
	}
       	
	restore_flags(flags);
}

static void mask_and_ack_xilleon_irq(unsigned int irq_nr)
{
}

static void end_xilleon_irq(unsigned int irq_nr)
{
}

struct hw_interrupt_type xilleon_irq_type = {
 	"XILLEON Int",
	startup_xilleon_irq,
	shutdown_xilleon_irq,
	enable_xilleon_irq,
	disable_xilleon_irq,
	mask_and_ack_xilleon_irq,
	end_xilleon_irq,
	NULL
};

#ifdef DBG_BSP_TRACE_ENABLE
static char *sb_names[] = {
	"XILLEON_IDE_INT",
	"XILLEON_USB_INT",
	"X225_IDE_INT_0",
	"X225_IDE_INT_1",
	"XILLEON_LPC0_INT",
	"XILLEON_LPC1_INT",
	"XILLEON_LPC2_INT",
	"XILLEON_LPC3_INT",
	"XILLEON_DAIO1_INT",
	"XILLEON_DAIO2_INT",
	"X225_MBA_INT",
	NULL
};
#endif

#if defined(DBG_BSP_TRACE_ENABLE) && defined(DBG_BSP_TRACE_CORRECT)
#   define STACK_DEPTH 256
static int nesting_level = 0;
static int correction_stack[STACK_DEPTH];
static int self_preemp_c[256];

void xilleon_trace_init(void)
{
	int i;

	for (i = 0; i < 256; i++)
		self_preemp_c[i] = -1;
}

void xilleon_trace_setup(int irq_nr)
{
	int flags;

	save_and_cli(flags);

	self_preemp_c[irq_nr]++;

	if (self_preemp_c[irq_nr])
		panic("IRQ %d self-preemption\n", irq_nr);

	correction_stack[nesting_level] = 0;

	nesting_level++;
	self_preemp_c[irq_nr]--;

	if (nesting_level >= STACK_DEPTH)
		panic("nesting_level >= STACK_DEPTH\n");

	restore_flags(flags);
}

void xilleon_trace_correct(int *diff, int base)
{
	int i, flags;

	save_and_cli(flags);
    
	/* decerease nesting level */
	if (nesting_level > 0)
		nesting_level--;
  
	if (correction_stack[nesting_level] <= *diff) {
		*diff -= correction_stack[nesting_level]; /* correct this result */

		/* add this result to all levels of nesting */
		for (i = 0; i < nesting_level; i++)
			correction_stack[i] += *diff;
	}

	restore_flags(flags);
}
#else
#   define xilleon_trace_setup(x)
#   define xilleon_trace_correct(x,y)
#   define xilleon_trace_init()
#endif

/* Split xilleon_irqdispatch into four functions for efficiency sake */
void xilleon_irqdispatch_sb(struct pt_regs *regs)
{
	u32 status, i;
	int irq = 0;
	long long in, out;
	int diff;

	xilleon_trace_setup(SB_IRQBASE);

	in = dbg_trace_get_timestamp64();
	dbg_trace_to_fb("+sb\n");

	status = GETREG_REGMM32(SB_INT_STATUS) & cached_sb_int_mask;
	DEBUG_INT("sb status is %#0x\n", status);

	for (i = 0; status; i++) {
		if(status & 1) {
			long long in1, out1;
			int diff1;

			in1 = dbg_trace_get_timestamp64();

			irq = i + SB_IRQBASE;
			DEBUG_INT("xilleon_irqdispatch_sb: %d\n", irq);
			do_IRQ(irq, regs);

			out1 = dbg_trace_get_timestamp64();
			diff1 = out1 - in1;

			dbg_trace_to_fb(" bsp %05d %s\n", 
					diff1, 
					sb_names[i] ? sb_names[i]: "<null>");
		}
		status >>= 1;
	}

	out = dbg_trace_get_timestamp64();
	diff = out - in;

	xilleon_trace_correct(&diff, SB_IRQBASE);

	dbg_trace_to_fb("-sb  %05d\n", diff);
}

#ifdef DBG_BSP_TRACE_ENABLE
static char *pci_names[] = {
	"XILLEON_INTR0_INT",
	"XILLEON_INTR1_INT",
	"XILLEON_INTR2_INT",
	"XILLEON_INTR3_INT",
	NULL
};
#endif

void xilleon_irqdispatch_pci(struct pt_regs *regs)
{
	u32 status, i;
	int irq = 0;
	long long in, out;
	int diff;
	int aChipId;

	xilleon_trace_setup(EXTPCI_IRQBASE);

	aChipId = GETFLD_REGMM16(HBIU_DEVICE_ID, DEVICE_ID);

	in = dbg_trace_get_timestamp64();
	dbg_trace_to_fb("+pci\n");
    
	status = GETREG_REGMM32(EXT_PCI_INT_STATUS);

	if (XK_CHIP_X225_A11(aChipId) && (status & EXT_PCI_INT_STATUS__INTR2__MASK))
		xilleon_irqdispatch_pcu (regs);

	status &= cached_ext_pci_int_mask;
    
	DEBUG_INT("pci status is %#0x\n", status);
    
	for (i = 0; status; i++) {
		if(status & 1) {
			long long in1, out1;
			int diff1;

			in1 = dbg_trace_get_timestamp64();

			irq = i + EXTPCI_IRQBASE;
			DEBUG_INT("xilleon_irqdispatch_pci: %d\n", irq);
			do_IRQ(irq, regs);

			out1 = dbg_trace_get_timestamp64();
			diff1 = out1 - in1;

			dbg_trace_to_fb(" bsp %05d %s\n", 
					diff1, 
					pci_names[i] ? pci_names[i]: "<null>");
		}
		status >>= 1;
	}

	out = dbg_trace_get_timestamp64();
	diff = out - in;

	xilleon_trace_correct(&diff, EXTPCI_IRQBASE);

	dbg_trace_to_fb("-pci %05d\n", diff);
}

void xilleon_irqdispatch_fw(struct pt_regs *regs)
{
	long long in, out;
	int diff;

	xilleon_trace_setup(FIRMWARE_IRQBASE);

	in = dbg_trace_get_timestamp64();
	dbg_trace_to_fb("+fw\n");

	DEBUG_INT("fw status is %#0x\n", GETREG_REGMM32(GEN_INT_STATUS));

	do_IRQ(FIRMWARE_IRQBASE, regs);

	out = dbg_trace_get_timestamp64();
	diff = out - in;

	xilleon_trace_correct(&diff, FIRMWARE_IRQBASE);

	dbg_trace_to_fb("-fw  %05d\n", diff);
}

#ifdef DBG_BSP_TRACE_ENABLE
static char *pcu_names[] = {
	"XILLEON_DMA_INT",
	"XILLEON_UIRT_INT",
	"XILLEON_UART2_INT",
	"XILLEON_UART1_INT",
	"XILLEON_TIMER_INT",
	"XILLEON_RTC_INT",
	"not-used",
	"not-used",
	"XILLEON_FB0_INT",
	"XILLEON_FB1_INT",
	"XILLEON_FB2_INT",
	"XILLEON_FB3_INT",
	"XILLEON_FB4_INT",
	"XILLEON_FB5_INT",
	"XILLEON_FB6_INT",
	"XILLEON_FB7_INT",
	"XILLEON_I2C0_INT",
	"XILLEON_I2C1_INT",
	"XILLEON_SMCA_INT",
	"XILLEON_SMCB_INT",
	"XILLEON_SMCC_INT",
	"XILLEON_HDCP_STAT_INT",
	"XILLEON_I2C_STAT_INT",
	"X225_UIRT2_INT",
	"X225_HOST_WRITE_DETECT_INT",
	NULL
};
#endif

void xilleon_irqdispatch_pcu(struct pt_regs *regs)
{
	u32 status, i, mask;
	int irq = 0;
	long long in, out;
	int diff;

	xilleon_trace_setup(PCU_IRQBASE);

	in = dbg_trace_get_timestamp64();
	dbg_trace_to_fb("+pcu\n");

	mask = GETREG_REGMM32(MIPS_PCU_INT_CNTL);
	status = GETREG_REGMM32(PCU_INT_STATUS) & /*cached_pcu_int_*/mask;
    
	DEBUG_INT("pcu status is %#0x\n", status);

	for (i = 0; status; i++) {
		if(status & 1) {
			long long in1, out1;
			int diff1;

			in1 = dbg_trace_get_timestamp64();

			irq = i + PCU_IRQBASE;
			DEBUG_INT("xilleon_irqdispatch_pcu: %d\n", irq);
			do_IRQ(irq, regs);

			out1 = dbg_trace_get_timestamp64();
			diff1 = out1 - in1;

			dbg_trace_to_fb(" bsp %05d %s\n", 
					diff1, 
					pcu_names[i] ? pcu_names[i]: "<null>");

		}
		status >>= 1;
	}

	out = dbg_trace_get_timestamp64();
	diff = out - in;

	xilleon_trace_correct(&diff, PCU_IRQBASE);

	dbg_trace_to_fb("-pcu %05d\n", diff);   
}

/*
 * IRQs on the ATI Xilleon are arranged in groups thusly:
 *
 *	MIPS IRQ	Source
 *      --------        ------
 *             0	    Software (ignored)
 *             1        Software (ignored)
 *             2        Hardware (HW0, ignored)	
 *             3        Peripheral Control Unit (PCU) (HW1)
 *             4        Firmware (HW2)
 *             5        PCI (HW3)
 *             6        Southbridge (HW4)
 *             7        R4k timer (INT 5)
 *
 * Each MIPS IRQ has multiple causes.  For simplicity's
 * sake, we just call a different function for each
 * MIPS interrupt, and let the upper layers of code
 * translate to the kernel's interrupt numbers.
 *
 * We handle the IRQ according to _our_ priority which is:
 *
 * Highest ----     Peripheral Control Unit (PCU)
 *                  Firmware
 *                  PCI
 *                  Southbridge
 * Lowest  ----     R4k Timer
 */


void xilleon_irqdispatch (struct pt_regs *regs)
{
	unsigned int cause;
	unsigned int status;
    
	cause = read_c0_cause();
	status = read_c0_status() & ALLINTS;
	cause &= status;

	if (cause & IE_IRQ1) {

		clear_c0_status(IE_IRQ1 | IE_IRQ2 | IE_IRQ3 | IE_IRQ4 | IE_IRQ5); 

		xilleon_irqdispatch_pcu(regs);

	} else if (cause & IE_IRQ2) {

		clear_c0_status(IE_IRQ2 | IE_IRQ3 | IE_IRQ4 | IE_IRQ5); 

		xilleon_irqdispatch_fw(regs);

	} else if (cause & IE_IRQ3) {

		clear_c0_status(IE_IRQ3 | IE_IRQ4 | IE_IRQ5); 

		xilleon_irqdispatch_pci(regs);

	} else if (cause & IE_IRQ4) {

		clear_c0_status(IE_IRQ4 | IE_IRQ5); 

		xilleon_irqdispatch_sb(regs);

	} else if (cause & IE_IRQ5) {
		long long in, out;
		int diff;

		xilleon_trace_setup(XILLEON_CP0_TIMER_INT);

		in = dbg_trace_get_timestamp64();
		dbg_trace_to_fb("+r4k\n");

		xilleon_timer_interrupt(regs);

		out = dbg_trace_get_timestamp64();
		diff = out - in;

		xilleon_trace_correct(&diff, XILLEON_CP0_TIMER_INT);

		dbg_trace_to_fb("-r4k %05d\n", diff);

	}

	set_c0_status(status);
}

#ifdef CONFIG_KGDB
extern void breakpoint(void);
extern int gdb_enter;
#endif

void __init init_IRQ (void)
{
	int i = 0;

	prom_printf("Initializing IRQs\n");

	/* 
	 * shut off interrupts for the whole chip
	 */

	CLEARFLAG_REGMM32(GEN_INT_CNTL, CHIP_INT_EN);

	/* 
	 * turn off all block level interrupts by writing "0" to all
	 * bit position in the interrupt registers
	 */

	SETREG_REGMM32(MIPS_GEN_INT_CNTL, 0x0000);
	SETREG_REGMM32(MIPS_PCU_INT_CNTL, 0x0000);
	SETREG_REGMM32(MIPS_SB_INT_CNTL, 0x0000);
	SETREG_REGMM32(MIPS_EXT_PCI_INT_CNTL, 0x0000);

	/* Now safe to set the exception vector. */

	set_except_vector(0, xilleonIRQ);

	xilleon_trace_init();

	init_generic_irq();

	for (i = 1; i < NR_IRQS; i++) {
		irq_desc[i].status	= IRQ_DISABLED;
		irq_desc[i].action	= 0;
		irq_desc[i].depth	= 1;
		irq_desc[i].handler     = &xilleon_irq_type;
	}

	/* 
	 * turn on interrupts now
	 */

	SETFLAG_REGMM32(GEN_INT_CNTL, CHIP_INT_EN);


#ifdef CONFIG_KGDB
	if (gdb_enter == 0) {
		goto gdb_hook_skip;
	}

	prom_printf("Wait for gdb client connection ...\n");
	set_debug_traps();
	breakpoint();
	prom_printf("Connected\n");
	goto gdb_hook_done;
 gdb_hook_skip:
	prom_printf("Skipping gdb client connection.\n");
 gdb_hook_done:
#endif
#ifdef CONFIG_GDB_CONSOLE
	register_gdb_console();
#endif
	
}

