/*
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 1999,2000 MIPS Technologies, Inc.  All rights reserved.
 *
 * ########################################################################
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ########################################################################
 *
 * Setting up the clock on the MIPS boards.
 *
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel_stat.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>

#include <asm/mipsregs.h>
#include <asm/ati/xilleon.h>
#include <asm/ptrace.h>
#include <asm/div64.h>

#include <linux/mc146818rtc.h>
#include <linux/timex.h>

#include <asm/ati/prom.h>
#include <asm/ati/xilleonint.h>

#define SEC_1               (27*1000*1000)
#define SEC_10              (SEC_1*10)

static unsigned long r4k_offset; /* Amount to increment compare reg each time */
static unsigned long r4k_cur;    /* What counter should be at next timer irq */
extern unsigned int mips_counter_frequency;
extern int setup_irq(unsigned int irq, struct irqaction * new);
extern unsigned int do_IRQ(int irq, struct pt_regs *regs);

#define ALLINTS (IE_SW0 | IE_SW1 | IE_IRQ0 | IE_IRQ1 | IE_IRQ2 | IE_IRQ3 | IE_IRQ4 | IE_IRQ5)


/* 
 * Figure out the r4k offset, the amount to increment the compare
 * register for each time tick. 
 * Use the RTC to calculate offset.
 */
static unsigned long __init cal_r4koff(void)
{
    unsigned long count;
    int lChipId = GETFLD_REGMM16( HBIU_DEVICE_ID, DEVICE_ID );

#ifdef CONFIG_IKOS
    count = 100000/2;
#else

    unsigned int flags;

    __save_and_cli(flags);

    if (!XK_CHIP_X220(lChipId)) {
        /* Start real time clock */
        SETREG_REGMM32(X225_RTC_CONT_REG1, GETREG_REGMM32(X225_RTC_CONT_REG1) | 1);
#if 1
        SETREG_REGMM32(X225_TIMER5_INITIALIZE_REG, SEC_10);
        SETREG_REGMM32(X225_TIMER5_READBACK_REG, SEC_10);
        SETREG_REGMM32(X225_TIMER_CONT_REG1, GETREG_REGMM32(X225_TIMER_CONT_REG1) | X225_TIMER_CONT_REG1__TIMER5_ENABLE__MASK);
        SETREG_REGMM32(X225_TIMER_CONT_REG0, GETREG_REGMM32(X225_TIMER_CONT_REG0) | X225_TIMER_CONT_REG0__TIMER5_RETRIGGER__MASK);
        count = GETREG_REGMM32(X225_TIMER5_READBACK_REG) - (SEC_1/10);
        write_c0_count(0);
        while (GETREG_REGMM32(X225_TIMER5_READBACK_REG) > count);

#else
        /* Start counter exactly on falling edge of update flag */
        count = GETREG_REGMM32(X225_RTC_READBACK_REG);
        while(GETREG_REGMM32(X225_RTC_READBACK_REG) == count) ;

        /* Start r4k counter. */
        write_c0_count(0);
        count = GETREG_REGMM32(X225_RTC_READBACK_REG);

        /* Read counter exactly on falling edge of update flag */
        while(GETREG_REGMM32(X225_RTC_READBACK_REG) == count) ;
#endif
    } else { /* x220 chip */
        /* Start real time clock */
        SETREG_REGMM32(RTC_CONT_REG1, GETREG_REGMM32(RTC_CONT_REG1) | 1);
#if 1
        SETREG_REGMM32(TIMER5_INITIALIZE_REG, SEC_10);
        SETREG_REGMM32(TIMER5_READBACK_REG, SEC_10);
        SETREG_REGMM32(TIMER_CONT_REG1, GETREG_REGMM32(TIMER_CONT_REG1) | TIMER_CONT_REG1__TIMER5_ENABLE__MASK);
        SETREG_REGMM32(TIMER_CONT_REG0, GETREG_REGMM32(TIMER_CONT_REG0) | TIMER_CONT_REG0__TIMER5_RETRIGGER__MASK);
        count = GETREG_REGMM32(TIMER5_READBACK_REG) - (SEC_1/10);
        write_c0_count(0);
        while (GETREG_REGMM32(TIMER5_READBACK_REG) > count);

#else
        /* Start counter exactly on falling edge of update flag */ 
        count = GETREG_REGMM32(RTC_READBACK_REG);
        while(GETREG_REGMM32(RTC_READBACK_REG) == count) ;

        /* Start r4k counter. */
        write_c0_count(0);
        count = GETREG_REGMM32(RTC_READBACK_REG);

        /* Read counter exactly on falling edge of update flag */ 
        while(GETREG_REGMM32(RTC_READBACK_REG) == count) ;
#endif
    }

    mips_counter_frequency = read_c0_count() * 10;

    /* restore interrupts */
    __restore_flags(flags);
#endif

    return (mips_counter_frequency / HZ);
}


void __init xilleon_time_init(void)
{
        unsigned int est_freq;

        /* Set Data mode - binary. */ 
        CMOS_WRITE(CMOS_READ(RTC_CONTROL) | RTC_DM_BINARY, RTC_CONTROL);

    printk("calculating r4koff... ");
    r4k_offset = cal_r4koff();
    //printk("%08lx(%d)\n", r4k_offset, (int) r4k_offset);

    est_freq = 2*r4k_offset*HZ; 
    est_freq += 5000;    /* round */
    est_freq -= est_freq%10000;
    printk("CPU frequency %d.%02d MHz\n", est_freq/1000000, 
           (est_freq%1000000)*100/1000000);
    r4k_cur = (read_c0_count() + r4k_offset);
}

void __init xilleon_timer_setup(struct irqaction *irq)
{
        /* we are using the cpu counter for timer interrupts */
    setup_irq(XILLEON_CP0_TIMER_INT, irq);

        /* to generate the first timer interrupt */
    r4k_cur = (read_c0_count() + r4k_offset);
    write_c0_compare(r4k_cur);
    clear_c0_status(ALLINTS);
    set_c0_status((IE_IRQ1 | IE_IRQ2 | IE_IRQ3 | IE_IRQ4 | IE_IRQ5));
}

void xilleon_timer_interrupt(struct pt_regs *regs)
{
    do_IRQ(XILLEON_CP0_TIMER_INT, regs);
}
