/*
 *  Module name: stw_flash.c
 *
 *  Descriptions:
 *
 *  This driver is ported from ATI pmon pflash.c, nflash.c, and sflash.c.
 *
 *  A hardware specific device (low-level) driver for the NAND, NOR, 
 *  and serial flash devices.
 *
 */

#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/ioport.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>

#include <asm/io.h>
#include <asm/ati/xilleon.h>
#include <asm/ati/stw_flash.h>

#ifdef CONFIG_MTD_NAND_ECC
#include <linux/mtd/nand_ecc.h>
#endif


/* NAND Flash */

#ifndef PFLASH_DEFAULT_CMD_TRIGGER_TIMEOUT
#define PFLASH_DEFAULT_CMD_TRIGGER_TIMEOUT 10
#endif

#ifndef PFLASH_DEFAULT_GET_STATUS_TIMEOUT
#define PFLASH_DEFAULT_GET_STATUS_TIMEOUT 10
#endif

#ifndef PFLASH_DEFAULT_GET_CHIP_ID_TIMEOUT
#define PFLASH_DEFAULT_GET_CHIP_ID_TIMEOUT 10
#endif

#ifndef PFLASH_DEFAULT_CLEAR_STATE_MACHINE_TIMEOUT
#define PFLASH_DEFAULT_CLEAR_STATE_MACHINE_TIMEOUT 10
#endif

#ifndef PFLASH_DEFAULT_BLOCK_ERASE_TIMEOUT
#define PFLASH_DEFAULT_BLOCK_ERASE_TIMEOUT 10
#endif

#ifndef FLASH_MS_TIMER_COUNT
#define FLASH_MS_TIMER_COUNT 1000
#endif



/************************************************************************/

#define SAMSUNG_ID          0xEC
#define SAMSUNG_K9F6408Q0C  0x39
#define SAMSUNG_K9F6408U0C  0xE6
#define SAMSUNG_K9F6408U0B  0xE6
#define SAMSUNG_K9F2808Q0C  0x33
#define SAMSUNG_K9F2808U0C  0x73
#define SAMSUNG_K9F2816Q0C  0x43
#define SAMSUNG_K9F2816U0C  0x53
#define SAMSUNG_K9F2808Q0B  0x33
#define SAMSUNG_K9F2808U0B  0x73
#define SAMSUNG_K9F5608Q0B  0x35
#define SAMSUNG_K9F5608U0B  0x75
#define SAMSUNG_K9F5616Q0B  0x45
#define SAMSUNG_K9F5616U0B  0x55
#define SAMSUNG_K9F5608U0A  0x75
#define SAMSUNG_K9F1208Q0A  0x36
#define SAMSUNG_K9F1208U0A  0x76
#define SAMSUNG_K9F1216Q0A  0x46
#define SAMSUNG_K9F1216U0A  0x56

#define TOSHIBA_ID          0x98
#define TOSHIBA_TC58V16BFT  0xEA
#define TOSHIBA_TC58V32AFT  0xE5
#define TOSHIBA_TC58V64AFT  0xE6
#define TOSHIBA_TC58V64BFT  0xE6
#define TOSHIBA_TC58128FT   0x73
#define TOSHIBA_TC58128AFT  0x73
#define TOSHIBA_TC58256FT   0x75
#define TOSHIBA_TC58256AFT  0x75
#define TOSHIBA_TH58512FT   0x76
#define TOSHIBA_TH58100FT   0x79

#define AMD_ID              0x01
#define AMD_AM30LV0064D     0xE6

#define PFLASH_CMD_TRIGGER_BLOCK_ERASE  0x1
#define PFLASH_CMD_TRIGGER_STATUS_RESET 0x2
#define PFLASH_CMD_TRIGGER_STATUS_READ  0x3
#define PFLASH_CMD_TRIGGER_ID_READ      0x4

#define SPARE_AREA_ECC0_L_OFFSET       0x0
#define SPARE_AREA_ECC1_L_OFFSET       0x1
#define SPARE_AREA_ECC2_L_OFFSET       0x2
#define SPARE_AREA_ECC0_U_OFFSET       0x3
#define SPARE_AREA_VALID_BLOCK_OFFSET  0x5
#define SPARE_AREA_ECC1_U_OFFSET       0x6
#define SPARE_AREA_ECC2_U_OFFSET       0x7

#define SPARE_AREA_VALID_BLOCK_VALUE   0xFF

#define IBT_GOOD_BLOCK          0x00000000
#define IBT_FACTORY_BAD_BLOCK   0x000000FF
#define IBT_DETECTED_BAD_BLOCK  0x0000FF00
#define IBT_TABLE_0_BLOCK       0x00FF0000
#define IBT_TABLE_1_BLOCK       0xFF000000
#define IBT_MONITOR_BOOT_BLOCK  0xFFFF0000

#define IBT_UNKNOWN_BLOCK  (-1)

#define MAKE_MAGIC(a,b,c,d) (((a)<<0)|((b)<<8)|((c)<<16)|((d)<<24))

#define MONITORBOOT_MAGIC  MAKE_MAGIC('M','O','N','0')

#define MEG 0x100000

/************************************************************************/

/* NOR flash */

#ifndef NFLASH_DEFAULT_WRITE_TIMEOUT
#define NFLASH_DEFAULT_WRITE_TIMEOUT 1
#endif

#ifndef NFLASH_DEFAULT_CHIP_ERASE_TIMEOUT
#define NFLASH_DEFAULT_CHIP_ERASE_TIMEOUT 15000
#endif

#ifndef NFLASH_DEFAULT_BLOCK_ERASE_TIMEOUT
#define NFLASH_DEFAULT_BLOCK_ERASE_TIMEOUT 400
#endif

#define KILO     0x400

#define NFF_CORE (1<<0)
#define NFF_USER (1<<1)

#define NFT_AT49xV16x4  0xC0
#define NFT_AT49xV16x4T 0xC2
#define NFT_AT49xV32x   0xC8
#define NFT_AT49xV32xT  0xC9
#define NFT_AT29LV320D  0xF9
#define NFT_AT29LV320DT 0xF6
#define NFT_AT29LV64xD  0x22D7


static int nflashExists(uint32 aper, uint32 baseoffset);
static int nflashWaitToComplete(uint32 aper, uint32 baseoffset, uint32 chipoffset, uint32 timeout);
static int nflashWrite8(uint32 aper, uint32 baseoffset, uint32 chipoffset, uint32 value);
static int nflashWrite16(uint32 aper, uint32 baseoffset, uint32 chipoffset, uint32 value);
static uint32 nflashRead8(uint32 aper, uint32 baseoffset, uint32 chipoffset);
static uint32 nflashRead16(uint32 aper, uint32 baseoffset, uint32 chipoffset);
static uint32 nflashRead32(uint32 aper, uint32 baseoffset, uint32 chipoffset);


/************************************************************************/

/* Serial Flash */

#ifndef SFLASH_DEFAULT_WRITE_TIMEOUT
#define SFLASH_DEFAULT_WRITE_TIMEOUT 1
#endif

#ifndef SFLASH_DEFAULT_BLOCK_ERASE_TIMEOUT
#define SFLASH_DEFAULT_BLOCK_ERASE_TIMEOUT 3000
#endif

#ifndef SFLASH_DEFAULT_CHIP_ERASE_TIMEOUT
#define SFLASH_DEFAULT_CHIP_ERASE_TIMEOUT 6000
#endif

#ifndef SFLASH_DEFAULT_SOFT_RESET_TIMEOUT
#define SFLASH_DEFAULT_SOFT_RESET_TIMEOUT 1
#endif

#ifndef SFLASH_DEFAULT_STATUS_TIMEOUT
#define SFLASH_DEFAULT_STATUS_TIMEOUT 1
#endif


/************************************************************************/

/* NAND flash */

/*
#ifdef CONFIG_PCU_ARBITER
#define DEBUG_PCU_ARBITER
#endif
*/

typedef struct
{
    uint32 magic;
    uint32 checksum; // currently unused
    uint32 timestamp;
    uint32 table[2048];
}
IBT_DATA;

#ifdef CONFIG_PCU_ARBITER
uint32 pcuNand = 0;   /* defualt */
uint32 fbcAperCtrl[6] = {0x0, 0x0, 0x0, 0x0, 0x0, 0x0};
#endif

/************************************************************************/


#ifndef CONFIG_STW4X225
uint32 sck_prescale;
#endif

static int sflash_exists;
static int sflash_pcu_mode = PCU_MODE_SFLASH_SEPST10;

static uint32 pflash_exists;
static uint32 pflash_mask, flash_mask;
static uint32 ibt_found;
static int32 ibt_cs = CS_INVALID;
static int32 ibt_block0 = IBT_UNKNOWN_BLOCK;
static int32 ibt_block1 = IBT_UNKNOWN_BLOCK;
static int pflash_pcu_mode = PCU_MODE_PFLASH_32MBIT;

static char blockbuffer[16384];

static union
{
char array[16384];
IBT_DATA data;
}
ibt;

/************************************************************************/

static int pflashGetSpare8(uint32 offset, uint8 *data);
static int pflashSetSpare8(uint32 offset, uint32 data);
static int pflashIsBlockFactoryInvalid(CHIP_SEL chipsel, uint32 block);
static int pflashFindInvalidBlockTable(void);
static int pflashWriteInvalidBlockTables(void);
static int pflashWriteInvalidBlockTable0(void);
static int pflashWriteInvalidBlockTable1(void);
static int pflashCreateInvalidBlockTable(void);

int pflashReadInvalidBlockTable(void);

/************************************************************************/

int pflashInit(void)
{
    uint32 r;
    uint32 size = 0;
    uint32 size0 = 0;
    uint32 size1 = 0;

#ifdef CONFIG_STW5X226
    pcuSetMode(PCU_MODE_PFLASH_32MBIT);
#endif

    // detect if pflash exists
    pflash_mask = 0;
    pflash_exists = (pflashClearStateMachine() == 0);

    if(pflash_exists)
    {
        if(size1 = pflashGetChipSize(CS1))
        {
            size = size1;
            ibt_cs = CS1;
            pflash_mask |= (1<<PFLASH1_NUM);
        }

        if(size0 = pflashGetChipSize(CS0))
        {
            size = size0;
            ibt_cs = CS0;
            pflash_mask |= (1<<PFLASH0_NUM);
        }


#ifdef STW_FLASH_DEBUG
	printk("pflash size: 0x%x  size0: 0x%x size1: 0x%x\n", size, size0, size1);
#endif

        // reenable pflash registers for detected size chip
        switch(size)
        {
        case 4*MEG:
            pflash_pcu_mode = PCU_MODE_PFLASH_32MBIT;
            r = GETREG_REGMM32(PFLASH_CNTL);
            MODIFYFLD(r, PFLASH_CNTL, PF32MBIT_EN, 1);
            MODIFYFLD(r, PFLASH_CNTL, PF64MBIT_EN, 0);
            MODIFYFLD(r, PFLASH_CNTL, PF128MBIT_EN, 0); 
            SETREG_REGMM32(PFLASH_CNTL, r); 
            break;

        case 8*MEG:
            pflash_pcu_mode = PCU_MODE_PFLASH_64MBIT;
            r = GETREG_REGMM32(PFLASH_CNTL);
            MODIFYFLD(r, PFLASH_CNTL, PF32MBIT_EN, 0);
            MODIFYFLD(r, PFLASH_CNTL, PF64MBIT_EN, 1);
            MODIFYFLD(r, PFLASH_CNTL, PF128MBIT_EN, 0); 
            SETREG_REGMM32(PFLASH_CNTL, r); 
            break;

        case 16*MEG:
            pflash_pcu_mode = PCU_MODE_PFLASH_128MBIT;
            r = GETREG_REGMM32(PFLASH_CNTL);
            MODIFYFLD(r, PFLASH_CNTL, PF32MBIT_EN, 0);
            MODIFYFLD(r, PFLASH_CNTL, PF64MBIT_EN, 0);
            MODIFYFLD(r, PFLASH_CNTL, PF128MBIT_EN, 1); 
            SETREG_REGMM32(PFLASH_CNTL, r); 
            break;
        }

#ifdef CONFIG_STW5X226
	pcuSetMode(pflashGetPcuMode());
#endif

        if(pflashFindInvalidBlockTable())
        {
	  printk("WARNING: pflashInit: pflash is disabled\n");
	  return 0;
        }
    }

    return pflash_mask;
}

/************************************************************************/

int pflashGetPcuMode(void)
{
    return pflash_pcu_mode;
}

/************************************************************************/

uint32 pflashGetTotalBlocks(void)
{
    if(pflash_exists)
    {
        if (GETFLD(PFLASH_CNTL,PF32MBIT_EN))
        {
            return 512;
        }
        else if (GETFLD(PFLASH_CNTL,PF64MBIT_EN))
        {
            return 1024;
        }
        else if (GETFLD(PFLASH_CNTL,PF128MBIT_EN))
        {
            return 1024;
        }
    }

    return 0;
}

/************************************************************************/

uint32 pflashGetTotalPages(void)
{
    if(pflash_exists)
    {
        if (GETFLD(PFLASH_CNTL,PF32MBIT_EN))
        {
            return 16;
        }
        else if (GETFLD(PFLASH_CNTL,PF64MBIT_EN))
        {
            return 16;
        }
        else if (GETFLD(PFLASH_CNTL,PF128MBIT_EN))
        {
            return 32;
        }
    }

    return 0;
}

/************************************************************************/

uint32 pflashGetBlockSizeInBytes(void)
{
    if(pflash_exists)
    {
        if (GETFLD(PFLASH_CNTL,PF32MBIT_EN))
        {
            return 8192;
        }
        else if (GETFLD(PFLASH_CNTL,PF64MBIT_EN))
        {
            return 8192;
        }
        else if (GETFLD(PFLASH_CNTL,PF128MBIT_EN))
        {
            return 16384;
        }
    }

    return 0;
}

/************************************************************************/

uint32 pflashGetChipStartOffset(CHIP_SEL chipsel)
{
    if(pflash_exists)
    {
        if(chipsel == CS1)
        {
            if (GETFLD(PFLASH_CNTL,PF32MBIT_EN))
            {
                return 0x00400000;
            }
            else if (GETFLD(PFLASH_CNTL,PF64MBIT_EN))
            {
                return 0x00800000;
            }
            else if (GETFLD(PFLASH_CNTL,PF128MBIT_EN))
            {
                return 0x01000000;
            }
            else // some other device is selected in STRAPS_VALUE register
            {
                printk("\nERROR: pflashGetChipStartOffset: no pflash devices found\n");

                return 0xFFFFFFFF; //error
            }
        }
    }

    return 0x00000000;
}

/************************************************************************/

uint32 pflashGetBlockStartOffset(CHIP_SEL chipsel, uint32 block)
{    
    if(pflash_exists)
    {
        return pflashGetChipStartOffset(chipsel) + (block * pflashGetBlockSizeInBytes());
    }

    return 0;
}

/************************************************************************/

int pflashChipErase(CHIP_SEL chipsel, uint32 flags)
{
    uint32 block;
    uint32 blocks = pflashGetTotalBlocks();
    int error = 0;

    if((flags & PFF_ALLOW_ALL) || (pflashFindInvalidBlockTable() == 0))
    {
        for(block=0; block<blocks; block++)
        {
            if(pflashIsBlockInvalid(chipsel, block, flags) == 0)
            {
	      printk(".");

                if(pflashBlockErase(chipsel, block, flags))
                {
                    printk("ERROR: pflashChipErase: unable to erase block %d\n", block);
                    
                    error = -1;
                }
                else
                {
                    if((flags & PFF_VERIFY) && pflashBlockEraseVerify(chipsel, block))
                    {
                        printk("ERROR: pflashChipErase: verify failed on block %d\n", block);

                        error = -1;
                    }
                }
            }
            else
            {
	      printk("B");
            }
        }

        printk("\n");
    }

    return error;
}

/************************************************************************/

int pflashBlockErase(CHIP_SEL chipsel, uint32 block, uint32 flags)
{
    uint32 status=PFLASH_OK;
    uint32 timeout;

    if(pflash_exists)
    {
        if(pflashFindInvalidBlockTable() == 0)
        {
            if(pflashIsBlockInvalid(chipsel, block, flags))
            {
                printk("\nERROR: pflashBlockErase: attempt to erase invalid block %d\n", block);

                return(PFLASH_BLOCK_INVALID);
            }
            else
            {
                pflashClearStateMachine();

                // set the address
                SETFLD(PFLASH_BE_ADDR,BLOCK_ERASE_ADDR, pflashGetBlockStartOffset(chipsel, block) >> PFLASH_BE_ADDR__BLOCK_ERASE_ADDR__SHIFT);
                // trigger the block erase command
                SETFLD(PFLASH_CNTL, CMD_TRIG, PFLASH_CMD_TRIGGER_BLOCK_ERASE);
                // wait until done
                status = pflashWaitCMDTriggerDone();

                if(status == PFLASH_OK)
                {
                    timeout = PFLASH_DEFAULT_BLOCK_ERASE_TIMEOUT * FLASH_MS_TIMER_COUNT;

                    while(GETFLD(PFLASH_STATUS, BLOCK_ERASE_DONE) == 0)
                    {
                        if(--timeout == 0)
                        {
                            printk("\nERROR: pflashBlockErase: timeout waiting for BLOCK_ERASE_DONE on block %d\n", block);

                            return(PFLASH_BLOCK_ERASE_NOT_DONE);
                        }

                        udelay(1000);
                    }

                    status = pflashGetStatus(chipsel);

                    if(status)
                    {
                        printk("\nERROR: pflashBlockErase: bad status on block %d\n", block);
                    }

                    SETFLD(PFLASH_STATUS, BLOCK_ERASE_DONE, 0x0);

                    pflashClearStateMachine();
                }
            }
        }
        else
        {
            printk("\nERROR: pflashBlockErase: invalid block table not found\n");

            status= PFLASH_IBT_NOT_FOUND;
        }
    }

    return(status);
}

/************************************************************************/

int pflashBlockEraseVerify(CHIP_SEL chipsel, uint32 block)
{
    uint32 status=0;
    uint32 offset;
    uint32 blocksize;
    uint32 chipoffset;
    uint32 blockoffset;
    uint32 actual;

    if(pflash_exists)
    {
        chipoffset = pflashGetChipStartOffset(chipsel);
        blocksize = pflashGetBlockSizeInBytes();
        blockoffset = block * blocksize;

        pflashClearStateMachine();

        for(offset=0; offset<blocksize; offset+=4)
        {
            actual = GETMEM_PFLASHMM32(chipoffset + blockoffset + offset);

            if(actual != 0xFFFFFFFF)
            {
                printk("\nERROR: pflashBlockEraseVerify: chip=%d, block=%d, offset=%d, actual=0x%08X\n", chipsel, block, offset, actual);

                status = PFLASH_VERIFY_FAILED;
                break;
            }
        }

        pflashClearStateMachine();
    }

    return status;
}

/************************************************************************/

int pflashPageRead(CHIP_SEL chipsel, uint32 block, uint32 page, char *buffer, uint32 flags)
{
    uint32 offset;
    uint32 chipoffset;
    uint32 blockoffset;
    uint32 pageoffset;
    uint32 blocksize;
    uint32 *p = (uint32 *) buffer;
#ifdef CONFIG_MTD_NAND_ECC
    uint8 read_ecc[3];
    uint8 calc_ecc[3];
#endif

    if(pflash_exists)
    {
        blocksize = pflashGetBlockSizeInBytes();
        chipoffset = pflashGetChipStartOffset(chipsel);
        blockoffset = block * blocksize;
        pageoffset = page * PFLASH_PAGE_SIZE;

        pflashClearStateMachine();

        for(offset=0; offset<PFLASH_PAGE_SIZE; offset+=4)
        {
            *p++ = GETMEM_PFLASHMM32(chipoffset + blockoffset + pageoffset + offset);

            //if (block < 4) {
	    //if (page < 2) {
	    //printk("chipoff=%08X, blockoff=%04X, pageoff=%04X, off=%04X, value=%08X\n", 
	    //chipoffset, blockoffset, pageoffset, offset, *(p-1));
	    //printk("KSEG1: 0x%x APER_PCU_BASE: 0x%x  PFLASH_BASE: 0x%x\n",
	    //KSEG1, APER_PCU_BASE, PFLASH_BASE); 
	    //}
        }

        pflashClearStateMachine();

#ifdef CONFIG_MTD_NAND_ECC
        if(flags & PFF_ECC)
        {
            // lower half of page
            nand_calculate_ecc(NULL, buffer, calc_ecc);

            pflashGetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC0_L_OFFSET, &read_ecc[0]);
            pflashGetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC1_L_OFFSET, &read_ecc[1]);
            pflashGetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC2_L_OFFSET, &read_ecc[2]);

            //printk("\n[L] read_ecc = %02X %02X %02X    calc_ecc = %02X %02X %02X\n", read_ecc[0], read_ecc[1], read_ecc[2], calc_ecc[0], calc_ecc[1], calc_ecc[2]);

            if(nand_correct_data(NULL, buffer, read_ecc, calc_ecc) == -1)
            {
                printk("\nERROR: pflashPageRead: ECC correction error: chip=%d, block=%d, page=%d (lower)\n", chipsel, block, page);

                return -1;
            }

            // upper half of page
            nand_calculate_ecc(NULL, buffer+(PFLASH_PAGE_SIZE/2), calc_ecc);

            pflashGetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC0_U_OFFSET, &read_ecc[0]);
            pflashGetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC1_U_OFFSET, &read_ecc[1]);
            pflashGetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC2_U_OFFSET, &read_ecc[2]);

            //printk("[U] read_ecc = %02X %02X %02X    calc_ecc = %02X %02X %02X\n", read_ecc[0], read_ecc[1], read_ecc[2], calc_ecc[0], calc_ecc[1], calc_ecc[2]);

            if(nand_correct_data(NULL, buffer+(PFLASH_PAGE_SIZE/2), read_ecc, calc_ecc) == -1)
            {
                printk("\nERROR: pflashPageRead: ECC correction error: chip=%d, block=%d, page=%d (upper)\n", chipsel, block, page);

                return -1;
            }
        }
#endif

        return 0;
    }

    return -1;
}

/************************************************************************/

int pflashBlockRead(CHIP_SEL chipsel, uint32 block, char *buffer, uint32 flags)
{
    uint32 blocksize;
    uint32 page;

    if(pflash_exists)
    {
        blocksize = pflashGetBlockSizeInBytes();

        for(page=0; page<(blocksize/PFLASH_PAGE_SIZE); page++)
        {
            if(pflashPageRead(chipsel, block, page, buffer, flags))
            {
                return -1;
            }

            buffer += PFLASH_PAGE_SIZE;
        }

        return 0;
    }

    return -1;
}

/************************************************************************/

int pflashChipRead(CHIP_SEL chipsel, uint32 block_start, uint32 blocks, char *addr, uint32 flags)
{
    uint32 blocksize;
    uint32 totalblocks;
    uint32 block;
    uint32 blocksremaining;

    blocksize = pflashGetBlockSizeInBytes();
    totalblocks = pflashGetTotalBlocks();
    blocksremaining = blocks;

    printk("Reading %d blocks starting at %d...\n", blocks, block_start);

    for(block=block_start; blocksremaining && (block<totalblocks); block++)
    {
        if(pflashIsBlockInvalid(chipsel, block, 0))
        {
	  printk("B");
        }
        else
        {
	  printk(".");

            if(pflashBlockRead(chipsel, block, addr, flags)) break;

            addr += blocksize;
            blocksremaining -= 1;
        }
    }

    printk("\n%d blocks read.\n", blocks-blocksremaining);

    return 0;
}

/************************************************************************/

int pflashPageWrite(CHIP_SEL chipsel, uint32 block, uint32 page, char *buffer, uint32 flags)
{
    uint32 offset;
    uint32 blocksize;
    uint32 chipoffset;
    uint32 pageoffset;
    uint32 blockoffset;
    uint32 status=0;
#ifdef CONFIG_MTD_NAND_ECC
    uint8 calc_ecc_lower[3];
    uint8 calc_ecc_upper[3];
#endif

    if(pflash_exists)
    {
        if(!(flags & PFF_ALLOW_IBT))
        {
            if(pflashFindInvalidBlockTable() == 0)
            {
                if(pflashIsBlockInvalid(chipsel, block, flags))
                {
                    printk("\nERROR: pflashPageWrite: attempt to write to an invalid block\n");

                    return -1;
                }
            }
        }

        chipoffset = pflashGetChipStartOffset(chipsel);
        blocksize = pflashGetBlockSizeInBytes();
        blockoffset = block * blocksize;
        pageoffset = page * PFLASH_PAGE_SIZE;

#ifdef CONFIG_MTD_NAND_ECC
        if(flags & PFF_ECC)
        {
            nand_calculate_ecc(NULL, buffer, calc_ecc_lower);
            nand_calculate_ecc(NULL, buffer+(PFLASH_PAGE_SIZE/2), calc_ecc_upper);
        }
#endif

        pflashClearStateMachine();

        for(offset=0; offset<PFLASH_PAGE_SIZE; offset+=4)
        {
            SETMEM_PFLASHMM32(chipoffset + blockoffset + pageoffset + offset, *((uint32 *)buffer)++);
        }

        status = pflashGetStatus(chipsel);

        pflashClearStateMachine();

#ifdef CONFIG_MTD_NAND_ECC
        if(flags & PFF_ECC)
        {
            // update ecc
            pflashSetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC0_L_OFFSET, calc_ecc_lower[0]);
            pflashSetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC1_L_OFFSET, calc_ecc_lower[1]);
            pflashSetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC2_L_OFFSET, calc_ecc_lower[2]);
            pflashSetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC0_U_OFFSET, calc_ecc_upper[0]);
            pflashSetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC1_U_OFFSET, calc_ecc_upper[1]);
            pflashSetSpare8(chipoffset + blockoffset + pageoffset + SPARE_AREA_ECC2_U_OFFSET, calc_ecc_upper[2]);
        }
#endif
    }

    return status;
}


int pflashBlockWrite(CHIP_SEL chipsel, uint32 block, char *buffer, uint32 flags)
{
    uint32 blocksize;
    uint32 page;

    if(pflash_exists)
    {
        if(!(flags & PFF_ALLOW_IBT))
        {
            if(pflashFindInvalidBlockTable() == 0)
            {
                if(pflashIsBlockInvalid(chipsel, block, flags))
                {
                    printk("\nERROR: pflashBlockWrite: attempt to write to an invalid block\n");

                    return -1;
                }
            }
        }

        blocksize = pflashGetBlockSizeInBytes();

        for(page=0; page<(blocksize/PFLASH_PAGE_SIZE); page++)
        {
            if(pflashPageWrite(chipsel, block, page, buffer, flags))
            {
                return -1;
            }

            buffer += PFLASH_PAGE_SIZE;
        }

        return 0;
    }

    return -1;
}

/************************************************************************/

int pflashChipWrite(CHIP_SEL chipsel, uint32 block_start, char *addr, uint32 size, uint32 flags)
{
    uint32 block;
    uint32 blocks;
    uint32 blocksize;
    uint32 totalblocks;
    uint32 blocksremaining;
    char *p;

    totalblocks = pflashGetTotalBlocks();
    blocksize = pflashGetBlockSizeInBytes();
    blocks = (size+blocksize-1) / blocksize;

    printk("Erasing %d blocks starting at %d...\n", blocks, block_start);

    blocksremaining = blocks;

    for(block=block_start; blocksremaining && (block<totalblocks); block++)
    {
        if(pflashIsBlockInvalid(chipsel, block, 0))
        {
            printk("B"); 
        }
        else
        {
	  printk(".");

            if(pflashBlockErase(chipsel, block, 0)) break;

            blocksremaining -= 1;
        }
    }


    printk("\n");
    printk("Verifying %d blocks starting at %d...\n", blocks, block_start);

    blocksremaining = blocks;

    for(block=block_start; blocksremaining && (block<totalblocks); block++)
    {
        if(pflashIsBlockInvalid(chipsel, block, 0))
        {
	  printk("B");
        }
        else
        {
            if(pflashBlockEraseVerify(chipsel, block))
            {
	      printk("!");
	      pflashSetBlockInvalid(chipsel, block);
	      continue;
            }
            else
            {
	      printk(".");
            }

            blocksremaining -= 1;
        }
    }

    printk("\n");
    printk("Writing %d blocks starting at %d...\n", blocks, block_start);

    blocksremaining = blocks;
    p = addr;

    for(block=block_start; blocksremaining && (block<totalblocks); block++)
    {
        if(pflashIsBlockInvalid(chipsel, block, 0))
        {
	  printk("B");
        }
        else
        {
	  printk(".");

            if(pflashBlockWrite(chipsel, block, p, flags)) break;

            p += blocksize;
            blocksremaining -= 1;
        }
    }

    printk("\n");
    printk("Verifying %d blocks starting at %d...\n", blocks, block_start);

    blocksremaining = blocks;
    p = addr;

    for(block=block_start; blocksremaining && (block<totalblocks); block++)
    {
        if(pflashIsBlockInvalid(chipsel, block, 0))
        {
	  printk("B");
        }
        else
        {
            if(pflashBlockVerify(chipsel, block, p, flags))
            {
	      printk("!");
                pflashSetBlockInvalid(chipsel, block);

                printk("\nERROR: pflashChipWrite: aborted\n");

                return -1;
            }
            else
            {
	      printk(".");
            }

            p += blocksize;
            blocksremaining -= 1;
        }
    }

    printk("\n");
    printk("%d blocks written.\n", blocks-blocksremaining);

    return 0;
}

/************************************************************************/

int pflashPageVerify(CHIP_SEL chipsel, uint32 block, uint32 page, char *buffer, uint32 flags)
{
    static char pagebuffer[PFLASH_PAGE_SIZE];
    uint32 offset;
    uint8 actual;
    uint8 expected;
    int error;

    if(pflash_exists)
    {
        if((error = pflashPageRead(chipsel, block, page, pagebuffer, flags)))
        {
            return error;
        }

        for(offset=0; offset<PFLASH_PAGE_SIZE; offset++)
        {
            actual = pagebuffer[offset];
            expected = buffer[offset];

            if(actual != expected)
            {
                printk("\nERROR: pflashPageVerify: chip=%d, block=%d, page=%d, offset=%d, actual=0x%02X, expected=0x%02X\n", chipsel, block, page, offset, actual, expected);

                return PFLASH_VERIFY_FAILED;
            }
        }

        return 0;
    }

    return -1;
}

/************************************************************************/

int pflashBlockVerify(CHIP_SEL chipsel, uint32 block, char *buffer, uint32 flags)
{
    uint32 blocksize;
    uint32 page;
    int error;

    if(pflash_exists)
    {
        blocksize = pflashGetBlockSizeInBytes();

        for(page=0; page<(blocksize/PFLASH_PAGE_SIZE); page++)
        {
            if((error = pflashPageVerify(chipsel, block, page, buffer, flags)))
            {
                return error;
            }

            buffer += PFLASH_PAGE_SIZE;
        }

        return 0;
    }

    return -1;
}

/************************************************************************/

int pflashGetStatus(CHIP_SEL chipsel)
{
    uint32 status=PFLASH_OK;
    uint32 timeout;

    if(pflash_exists)
    {
        pflashClearStateMachine();

        // set the address
        SETFLD(PFLASH_BE_ADDR, BLOCK_ERASE_ADDR, pflashGetChipStartOffset(chipsel) >> PFLASH_BE_ADDR__BLOCK_ERASE_ADDR__SHIFT);
        // trigger the status read command
        SETFLD(PFLASH_CNTL, CMD_TRIG, PFLASH_CMD_TRIGGER_STATUS_READ);
        // wait until complete
        status = pflashWaitCMDTriggerDone();

        if(status == PFLASH_OK)
        {
            timeout = PFLASH_DEFAULT_GET_STATUS_TIMEOUT * FLASH_MS_TIMER_COUNT;

            while(!(GETFLD(PFLASH_STATUS, STATUS_RD_DONE)))
            {
                if(--timeout == 0)
                {
                    printk("\nERROR: pflashGetStatus: timeout waiting for STATUS_RD_DONE\n");

                    return PFLASH_TIME_OUT;
                }

                udelay(1000);
            }

            if(status = GETFLD(PFLASH_STATUS, STATUS_ERROR))
            {
                printk("\nERROR: pflashGetStatus: PFLASH_STATUS=%08X\n", status);

                SETFLD(PFLASH_STATUS, STATUS_ERROR, 0x0);

                status = PFLASH_STATUS_ERROR;
            }

            SETFLD(PFLASH_STATUS, STATUS_RD_DONE, 0x0); // reset the done flag

            pflashClearStateMachine();
        }
    }

    return status;
}

/************************************************************************/

int pflashClearStateMachine(void)
{
    uint32 data;
    uint32 timeout;

    timeout = PFLASH_DEFAULT_CLEAR_STATE_MACHINE_TIMEOUT * FLASH_MS_TIMER_COUNT;

    // Wait for SEPROM_BUSY to go low
    while(GETFLD(SFLASH_CNTL2_STATUS, SEPROM_BUSY))
    {
        if(--timeout == 0)
        {
            printk("\nERROR: pflashClearStateMachine: timeout waiting for SEPROM_BUSY\n");

            return(PFLASH_SEPROM_BUSY);
        }

        udelay(1000);
    }

    // Clear The Read/Write cycle of state machine
    data = GETREG(PFLASH_CNTL);
    MODIFYFLD(data, PFLASH_CNTL, CMD_TRIG,    0);
    MODIFYFLD(data, PFLASH_CNTL, READ_CYCLE,  0);
    MODIFYFLD(data, PFLASH_CNTL, WRITE_CYCLE, 0);
    MODIFYFLD(data, PFLASH_CNTL, SPARE_EN, 0);	
    SETREG(PFLASH_CNTL, data);

    // Clear the status
    SETREG(PFLASH_STATUS, 0);

    return 0;
}

/************************************************************************/

int pflashWaitCMDTriggerDone(void)
{
    uint32 timeout;

    if(pflash_exists)
    {
        timeout = PFLASH_DEFAULT_CMD_TRIGGER_TIMEOUT * FLASH_MS_TIMER_COUNT;

        while(GETFLD(PFLASH_CNTL, CMD_TRIG) != 0)
        {
            if(--timeout == 0)
            {
                printk("\nERROR: pflashWaitCMDTriggerDone: timeout waiting for CMD_TRIG\n");

                return(PFLASH_CMD_TRIG_BUSY);
            }

            udelay(1000);
        }
    }

    return PFLASH_OK;
}

/************************************************************************/

uint32 pflashGetChipID(CHIP_SEL chipsel)
{
    uint32 status = PFLASH_OK;
    uint32 id = 0;
    uint32 timeout;

    if(pflash_exists)
    {
        pflashClearStateMachine();

        // set the address
        SETFLD(PFLASH_BE_ADDR, BLOCK_ERASE_ADDR, pflashGetChipStartOffset(chipsel) >> PFLASH_BE_ADDR__BLOCK_ERASE_ADDR__SHIFT);
        // trigger the ID read command
        SETFLD(PFLASH_CNTL, CMD_TRIG, PFLASH_CMD_TRIGGER_ID_READ);
        // wait until complete
        status = pflashWaitCMDTriggerDone();

        if(status == PFLASH_OK)
        {
            timeout = PFLASH_DEFAULT_GET_CHIP_ID_TIMEOUT * FLASH_MS_TIMER_COUNT;

            while(GETFLD(PFLASH_STATUS, ID_READ_DONE) == 0)
            {
                if(--timeout == 0)
                {
                    printk("\nERROR: pflashGetChipID: timeout waiting for ID_READ_DONE\n");

                    return 0;
                }

                udelay(1000);
            }

            id = GETFLD(PFLASH_ID_STAT_DATA, PF_ID_DATA);

            SETFLD(PFLASH_STATUS, ID_READ_DONE, 0x0);

            pflashClearStateMachine();
        }
    }

    return id;
}

/************************************************************************/

char *pflashGetIdString(CHIP_SEL chipsel)
{
    uint32 id;
    static char s[32];

    if(pflash_exists)
    {
        id = pflashGetChipID(chipsel);

        switch(id & 0xFF)
        {
        case SAMSUNG_ID:
            strcpy(s, "Samsung ");

            switch((id & 0xFF00) >> 8)
            {
            case SAMSUNG_K9F6408Q0C:  strcat(s, "K9F6408Q0");  break;
            case SAMSUNG_K9F6408U0C:  strcat(s, "K9F6408U0");  break;
            case SAMSUNG_K9F2808Q0C:  strcat(s, "K9F2808Q0");  break;
            case SAMSUNG_K9F2808U0C:  strcat(s, "K9F2808U0");  break;
            case SAMSUNG_K9F2816Q0C:  strcat(s, "K9F2816Q0");  break;
            case SAMSUNG_K9F2816U0C:  strcat(s, "K9F2816U0");  break;
            case SAMSUNG_K9F5608Q0B:  strcat(s, "K9F5608Q0");  break;
            case SAMSUNG_K9F5608U0B:  strcat(s, "K9F5608U0");  break;
            case SAMSUNG_K9F5616Q0B:  strcat(s, "K9F5616Q0");  break;
            case SAMSUNG_K9F5616U0B:  strcat(s, "K9F5616U0");  break;
            case SAMSUNG_K9F1208Q0A:  strcat(s, "K9F1208Q0");  break;
            case SAMSUNG_K9F1208U0A:  strcat(s, "K9F1208U0");  break;
            case SAMSUNG_K9F1216Q0A:  strcat(s, "K9F1216Q0");  break;
            case SAMSUNG_K9F1216U0A:  strcat(s, "K9F1216U0");  break;
            default:                  strcat(s, "unknown");    break;
            }
            break;

        case TOSHIBA_ID:
            strcpy(s, "Toshiba ");

            switch((id & 0xFF00) >> 8)
            {
            case TOSHIBA_TC58V16BFT:  strcat(s, "TC58V16FT");  break;
            case TOSHIBA_TC58V32AFT:  strcat(s, "TC58V32FT");  break;
            case TOSHIBA_TC58V64AFT:  strcat(s, "TC58V64FT");  break;
            case TOSHIBA_TC58128FT :  strcat(s, "TC58128FT");  break;
            case TOSHIBA_TC58256FT :  strcat(s, "TC58256FT");  break;
            case TOSHIBA_TH58512FT :  strcat(s, "TH58512FT");  break;
            case TOSHIBA_TH58100FT :  strcat(s, "TH58100FT");  break;
            default:                  strcat(s, "unknown");    break;
            }
            break;

        case AMD_ID:
            strcpy(s, "AMD ");

            switch((id & 0xFF00) >> 8)
            {
            case AMD_AM30LV0064D:     strcat(s, "AM30LV0064"); break;
            default:                  strcat(s, "unknown");    break;
            }
            break;

        default:
            strcpy(s, "Unknown");
            break;
        }
    }
    else
    {
        strcpy(s, "Not found");
    }

    return s;
}

/************************************************************************/

uint32 pflashGetChipSize(CHIP_SEL chipsel)
{
    uint32 id;
    int size;

    id = pflashGetChipID(chipsel);
    size = 0;

    switch(id & 0xFF)
    {
    case SAMSUNG_ID:
        switch((id & 0xFF00) >> 8)
        {
        case SAMSUNG_K9F6408Q0C:  size =   8*MEG;  break;
        case SAMSUNG_K9F6408U0C:  size =   8*MEG;  break;
        case SAMSUNG_K9F2808Q0C:  size =  16*MEG;  break;
        case SAMSUNG_K9F2808U0C:  size =  16*MEG;  break;
        case SAMSUNG_K9F2816Q0C:  size =  16*MEG;  break;
        case SAMSUNG_K9F2816U0C:  size =  16*MEG;  break;
        case SAMSUNG_K9F5608Q0B:  size =  32*MEG;  break;
        case SAMSUNG_K9F5608U0B:  size =  32*MEG;  break;
        case SAMSUNG_K9F5616Q0B:  size =  32*MEG;  break;
        case SAMSUNG_K9F5616U0B:  size =  32*MEG;  break;
        case SAMSUNG_K9F1208Q0A:  size =  64*MEG;  break;
        case SAMSUNG_K9F1208U0A:  size =  64*MEG;  break;
        case SAMSUNG_K9F1216Q0A:  size =  64*MEG;  break;
        case SAMSUNG_K9F1216U0A:  size =  64*MEG;  break;
        }
        break;

    case TOSHIBA_ID:
        switch((id & 0xFF00) >> 8)
        {
        case TOSHIBA_TC58V16BFT:  size =   2*MEG;  break;
        case TOSHIBA_TC58V32AFT:  size =   4*MEG;  break;
        case TOSHIBA_TC58V64AFT:  size =   8*MEG;  break;
        case TOSHIBA_TC58128FT :  size =  16*MEG;  break;
        case TOSHIBA_TC58256FT :  size =  32*MEG;  break;
        case TOSHIBA_TH58512FT :  size =  64*MEG;  break;
        case TOSHIBA_TH58100FT :  size = 128*MEG;  break;
        }
        break;

    case AMD_ID:
        switch((id & 0xFF00) >> 8)
        {
        case AMD_AM30LV0064D:     size =  8*MEG;  break;
        }
        break;
    }

    //printk("DEBUG - pflashGetChipSize - chipId: 0x%x size: 0x%x\n", id, size);

    return size;
}

/************************************************************************/

int pflashIsBlockInvalid(CHIP_SEL chipsel, uint32 block, uint32 flags)
{
    if((ibt_cs == CS0) && (chipsel == CS1)) block += pflashGetTotalBlocks();

    if(flags & PFF_ALLOW_ALL) return 0; // debug only - will erase ibt
   
    if(ibt.data.table[block] == IBT_GOOD_BLOCK) return 0;

    if(flags & PFF_ALLOW_IBT)
    {
        if(ibt.data.table[block] == IBT_TABLE_0_BLOCK) return 0;
        if(ibt.data.table[block] == IBT_TABLE_1_BLOCK) return 0;
    }

    if(flags & PFF_ALLOW_MB)
    {
        if(ibt.data.table[block] == IBT_MONITOR_BOOT_BLOCK) return 0;
    }

    return 1;
}

/************************************************************************/

int pflashSetBlockInvalid(CHIP_SEL chipsel, uint32 block)
{
    if((ibt_cs == CS0) && (chipsel == CS1)) block += pflashGetTotalBlocks();

    if(ibt.data.table[block] == IBT_GOOD_BLOCK)
    {
        ibt.data.table[block] = IBT_DETECTED_BAD_BLOCK;
    }

    return pflashWriteInvalidBlockTables();
}

/************************************************************************/

int pflashSetAutoExec(char *command, uint32 *rblock)
{
    uint32 block;
    uint32 total_blocks;

    total_blocks = pflashGetTotalBlocks();

    // search for an existing monitor boot block
    if(pflashGetAutoExec(&block) == 0)
    {
        printk("Creating autoexec block...\n");

        // create a new monitor boot block in the first available good block
        for(block=0; block<total_blocks; block++)
        {
            if(pflashIsBlockInvalid(ibt_cs, block, PFF_ALLOW_MB) == 0)
            {
                ibt.data.table[block] = IBT_MONITOR_BOOT_BLOCK;

                if(pflashWriteInvalidBlockTables())
                {
                    printk("ERROR: pflashSetAutoExec: unable to create autoexec block\n");

                    return -1;
                }

                break;
            }
        }
    }
    else
    {
        printk("Updating autoexec block...\n");
    }

    if(block == total_blocks)
    {
        printk("ERROR: pflashSetAutoExec: unable to create autoexec block\n");
    }
    else
    {
        if(pflashBlockErase(ibt_cs, block, PFF_ALLOW_MB) == 0)
        {
            if(pflashBlockEraseVerify(ibt_cs, block) == 0)
            {
                memset(blockbuffer, 0, sizeof(blockbuffer));

                *((uint32 *) blockbuffer) = MONITORBOOT_MAGIC;

                strncpy(&blockbuffer[4], command, sizeof(blockbuffer)-5);

                if(pflashBlockWrite(ibt_cs, block, blockbuffer, PFF_ECC | PFF_ALLOW_MB) == 0)
                {
                    if(pflashBlockVerify(ibt_cs, block, blockbuffer, PFF_ECC) == 0)
                    {
                        *rblock = block;

                        return 0;
                    }
                    else
                    {
                        printk("ERROR: pflashSetAutoExec: autoexec block verify error\n");
                    }
                }
                else
                {
                    printk("ERROR: pflashSetAutoExec: unable to write autoexec block\n");
                }
            }
            else
            {
                printk("ERROR: pflashSetAutoExec: autoexec block erase verify error\n");
            }
        }
        else
        {
            printk("ERROR: pflashSetAutoExec: unable to erase autoexec block\n");
        }
    }

    return -1;
}

/************************************************************************/

char * pflashGetAutoExec(uint32 *rblock)
{
    uint32 block;
    uint32 total_blocks;

    total_blocks = pflashGetTotalBlocks();

    // search for an existing monitor boot block
    for(block=0; block<total_blocks; block++)
    {
        if(ibt.data.table[block] == IBT_MONITOR_BOOT_BLOCK)
        {
            if(pflashBlockRead(ibt_cs, block, blockbuffer, PFF_ECC) == 0)
            {
                if(*((uint32 *) blockbuffer) == MONITORBOOT_MAGIC)
                {
                    *rblock = block;

                    return &blockbuffer[4];
                }
            }
        }
    }

    return 0;
}

/************************************************************************/

int pflashGetBufferSize(uint32 imagesize)
{
    uint32 block_size = pflashGetBlockSizeInBytes();

    return ((imagesize + block_size - 1) / block_size) * block_size;
}

/************************************************************************/

static int pflashGetSpare8(uint32 offset, uint8 *data)
{
    pflashClearStateMachine();
    SETFLD_REGMM32(PFLASH_CNTL, SPARE_EN, 1);
    *data = GETMEM_PFLASHMM8(offset);
    SETFLD_REGMM32(PFLASH_CNTL, SPARE_EN, 0);
    pflashClearStateMachine();

    return 0;
}

/************************************************************************/

static int pflashSetSpare8(uint32 offset, uint32 data)
{
    pflashClearStateMachine();
    SETFLD_REGMM32(PFLASH_CNTL, SPARE_EN, 1);
    SETMEM_PFLASHMM8(offset, data);
    SETFLD_REGMM32(PFLASH_CNTL, SPARE_EN, 0);
    pflashClearStateMachine();

    return 0;
}

/************************************************************************/
// This function is currently for Samsung NAND flash only

static int pflashIsBlockFactoryInvalid(CHIP_SEL chipsel, uint32 block)
{
    uint8 page0_value;
    uint8 page1_value;
    uint32 start_offset;

    // TODO: add check for Toshiba and use its algorithm

    start_offset = pflashGetBlockStartOffset(chipsel, block);

    pflashGetSpare8(start_offset + SPARE_AREA_VALID_BLOCK_OFFSET, &page0_value);
    pflashGetSpare8(start_offset + SPARE_AREA_VALID_BLOCK_OFFSET + PFLASH_PAGE_SIZE, &page1_value);

    if((page0_value != SPARE_AREA_VALID_BLOCK_VALUE) || (page1_value != SPARE_AREA_VALID_BLOCK_VALUE))
    {
        //printk("WARNING: pflashIsBlockFactoryInvalid: bad block=%d, page0=0x%02X, page1=0x%02X\n", block, page0_value, page1_value);

        return PFLASH_BLOCK_INVALID;
    }
    else
    {
        return PFLASH_OK;
    }
}

/************************************************************************/

static int pflashCreateInvalidBlockTable(void)
{
    uint32 block;
    uint32 total_blocks;

    //printk("Creating pflash invalid block table...\n");

    ibt_block0 = IBT_UNKNOWN_BLOCK;
    ibt_block1 = IBT_UNKNOWN_BLOCK;

    total_blocks = pflashGetTotalBlocks();

    ibt.data.timestamp = 0;

    if(ibt_cs == CS0) // pflash exists on CS0
    {
        for(block=0; block<total_blocks; block++)
        {
            if(pflashIsBlockFactoryInvalid(CS0, block) == PFLASH_BLOCK_INVALID)
            {
                printk("WARNING: pflashCreateInvalidBlockTable: pflash chip 0 block %d is invalid\n", block);

                ibt.data.table[block] = IBT_FACTORY_BAD_BLOCK;
            }
            else
            {
                ibt.data.table[block] = IBT_GOOD_BLOCK;

                if(ibt_block0 == IBT_UNKNOWN_BLOCK)
                    ibt_block0 = block;
                else if(ibt_block1 == IBT_UNKNOWN_BLOCK)
                    ibt_block1 = block;
            }
        }

        if(pflash_mask & (1<<PFLASH1_NUM)) // pflash exists on CS1
        {
            for(block=0; block<total_blocks; block++)
            {
                if(pflashIsBlockFactoryInvalid(CS1, block) == PFLASH_BLOCK_INVALID)
                {
                    printk("WARNING: pflashCreateInvalidBlockTable: pflash chip 1 block %d is invalid\n", block);

                    ibt.data.table[total_blocks+block] = IBT_FACTORY_BAD_BLOCK;
                }
                else
                {
                    ibt.data.table[total_blocks+block] = IBT_GOOD_BLOCK;

/*
                    if(ibt_block0 == IBT_UNKNOWN_BLOCK)
                        ibt_block0 = total_blocks+block;
                    else if(ibt_block1 == IBT_UNKNOWN_BLOCK)
                        ibt_block1 = total_blocks+block;
*/
                }
            }
        }
    }
    else if(ibt_cs == CS1) // pflash exists only on CS1
    {
        for(block=0; block<total_blocks; block++)
        {
            if(pflashIsBlockFactoryInvalid(CS1, block) == PFLASH_BLOCK_INVALID)
            {
                printk("WARNING: pflashCreateInvalidBlockTable: pflash chip 1 block %d is invalid\n", block);

                ibt.data.table[block] = IBT_FACTORY_BAD_BLOCK;
            }
            else
            {
                ibt.data.table[block] = IBT_GOOD_BLOCK;

                if(ibt_block0 == IBT_UNKNOWN_BLOCK)
                    ibt_block0 = block;
                else if(ibt_block1 == IBT_UNKNOWN_BLOCK)
                    ibt_block1 = block;
            }
        }
    }
    else
    {
        printk("ERROR: pflashCreateInvalidBlockTable: unable to create table\n");

        return -1;
    }

    ibt.data.table[ibt_block0] = IBT_TABLE_0_BLOCK;
    ibt.data.table[ibt_block1] = IBT_TABLE_1_BLOCK;

    ibt_found = 1;

    return pflashWriteInvalidBlockTables();
}

/************************************************************************/

static int pflashWriteInvalidBlockTables(void)
{
    int retval = 0;

    ibt.data.timestamp += 1;

    if(pflashWriteInvalidBlockTable0()) retval = -1;
    if(pflashWriteInvalidBlockTable1()) retval = -1;

    return retval;
}

/************************************************************************/

static int pflashWriteInvalidBlockTable0(void)
{
    if(ibt_block0 != IBT_UNKNOWN_BLOCK)
    {
        //printk("Erasing block %d...\n", ibt_block0);

        if(pflashBlockErase(ibt_cs, ibt_block0, PFF_ALLOW_IBT))
        {
            printk("ERROR: pflashWriteInvalidBlockTable0: unable to erase chip %d block %d\n", ibt_cs, ibt_block0);

            return -1;
        }
        else
        {
            //printk("Writing pflash invalid block table 0 to block %d...\n", ibt_block0);

            ibt.data.magic = MAKE_MAGIC('I','B','T','0');

            if(pflashBlockWrite(ibt_cs, ibt_block0, ibt.array, PFF_ECC | PFF_ALLOW_IBT))
            {
                printk("ERROR: pflashWriteInvalidBlockTable0: unable to write chip %d block %d\n", ibt_cs, ibt_block0);

                return -1;
            }
            else
            {
                if(pflashBlockVerify(ibt_cs, ibt_block0, ibt.array, PFF_ECC))
                {
                    printk("ERROR: pflashWriteInvalidBlockTable0: verify error\n");

                    return -1;
                }
            }
        }
    }
    else
    {
        printk("ERROR: pflashWriteInvalidBlockTable0: block not found\n");

        return -1;
    }

    return 0;
}

/************************************************************************/

static int pflashWriteInvalidBlockTable1(void)
{
    if(ibt_block1 != IBT_UNKNOWN_BLOCK)
    {
        //printk("Erasing block %d...\n", ibt_block1);

        if(pflashBlockErase(ibt_cs, ibt_block1, PFF_ALLOW_IBT))
        {
            printk("ERROR: pflashWriteInvalidBlockTable1: unable to erase chip %d block %d\n", ibt_cs, ibt_block1);

            return -1;
        }
        else
        {
            //printk("Writing pflash invalid block table 1 to block %d...\n", ibt_block1);

            ibt.data.magic = MAKE_MAGIC('I','B','T','1');

            if(pflashBlockWrite(ibt_cs, ibt_block1, ibt.array, PFF_ECC | PFF_ALLOW_IBT))
            {
                printk("ERROR: pflashWriteInvalidBlockTable1: unable to write chip %d block %d\n", ibt_cs, ibt_block1);

                return -1;
            }
            else
            {
                if(pflashBlockVerify(ibt_cs, ibt_block1, ibt.array, PFF_ECC))
                {
                    printk("ERROR: pflashWriteInvalidBlockTable1: verify error\n");

                    return -1;
                }
            }
        }
    }
    else
    {
        printk("ERROR: pflashWriteInvalidBlockTable1: block not found\n");

        return -1;
    }

    return 0;
}

/************************************************************************/

int pflashReadInvalidBlockTable(void)
{
    uint32 block;
    char pagebuffer[PFLASH_PAGE_SIZE];
    char pagebuffer1[PFLASH_PAGE_SIZE];
    uint32 total_blocks;
    uint32 ibt_timestamp0 = 0;
    uint32 ibt_timestamp1 = 0;
    int status;

    ibt_block0 = IBT_UNKNOWN_BLOCK;
    ibt_block1 = IBT_UNKNOWN_BLOCK;

    total_blocks = pflashGetTotalBlocks();

    // search for the invalid block tables by looking for the magic numbers (IBT0/1)
    for(block=0; block<total_blocks; block++)
    {
        status = pflashPageRead(ibt_cs, block, 0, pagebuffer, PFF_ECC);
        pflashPageRead(ibt_cs, block, 1, pagebuffer1, 0); // required 
	
	/* X226 debugging */
	//printk("pflashReadIBT - total block: 0x%x  block: 0x%x\n", total_blocks, block);

        if(status == 0)
        {
	  //if(block < 10) printk("block %d magic='%c%c%c%c'\n", block, pagebuffer[0], pagebuffer[1], pagebuffer[2], pagebuffer[3]);

            if(pagebuffer[0] == 'I' && pagebuffer[1] == 'B' && pagebuffer[2] == 'T')
            {
                if(pagebuffer[3] == '0')
                    ibt_block0 = block;
                else if(pagebuffer[3] == '1')
                    ibt_block1 = block;

                if((ibt_block0 != IBT_UNKNOWN_BLOCK) && (ibt_block1 != IBT_UNKNOWN_BLOCK))
                {
                    break;
                }
            }
        }
    }

    // attempt to read invalid block table #0
    if(ibt_block0 != IBT_UNKNOWN_BLOCK)
    {
        printk("Reading pflash invalid block table 0 at block %d...\n", ibt_block0);

        if(pflashBlockRead(ibt_cs, ibt_block0, ibt.array, PFF_ECC) == 0)
        {
            ibt_timestamp0 = ibt.data.timestamp;
        }
        else
        {
            printk("WARNING: pflashReadInvalidBlockTable: unable to read table 0\n");
        }
    }
    else
    {
        printk("WARNING: pflashReadInvalidBlockTable: table 0 not found\n");
    }

    // attempt to read invalid block table #1
    if(ibt_block1 != IBT_UNKNOWN_BLOCK)
    {
        printk("Reading pflash invalid block table 1 at block %d...\n", ibt_block1);

        if(pflashBlockRead(ibt_cs, ibt_block1, ibt.array, PFF_ECC) == 0)
        {
            ibt_timestamp1 = ibt.data.timestamp;
        }
        else
        {
            printk("WARNING: pflashReadInvalidBlockTable: unable to read table 1\n");
        }
    }
    else
    {
        printk("WARNING: pflashReadInvalidBlockTable: table 1 not found\n");
    }

    if(ibt_timestamp0 || ibt_timestamp1)
    {
        if(ibt_timestamp0 < ibt_timestamp1)
        {
            // read table #1 data
            if(pflashBlockRead(ibt_cs, ibt_block1, ibt.array, PFF_ECC) == 0)
            {
                // overwrite table #0 with table #1 data
                pflashWriteInvalidBlockTable0();

                return 0;
            }
        }
        else if(ibt_timestamp0 > ibt_timestamp1)
        {
            // read table #0 data
            if(pflashBlockRead(ibt_cs, ibt_block0, ibt.array, PFF_ECC) == 0)
            {
                // overwrite table #1 with table #0 data
                pflashWriteInvalidBlockTable1();

                return 0;
            }
        }
        else
        {
            // read table #0 data
            if(pflashBlockRead(ibt_cs, ibt_block0, ibt.array, PFF_ECC) == 0)
            {
                return 0;
            }
        }
    }

    // neither invalid block table could be read
    return -1;
}

/************************************************************************/

static int pflashFindInvalidBlockTable(void)
{
    if(!ibt_found)
    {
        if(pflashReadInvalidBlockTable())
        {
            printk("\nWARNING: pflashFindInvalidBlockTable: unable to find table - creating it...\n");

            if(pflashCreateInvalidBlockTable())
            {
                printk("ERROR: pflashFindInvalidBlockTable: unable to create table\n");

                return -1;
            }
        }

        ibt_found = 1;
    }

    return 0;
}


/*****************************************************************************/

int pflash_oob_enable(uint32_t timeout)
{
  uint32_t data;

  /* Wait for SEPROM_BUSY to go low */
  while(GETFLD_REGMM32(SFLASH_CNTL2_STATUS, SEPROM_BUSY)) {
    udelay(1000);

    if(--timeout == 0) {
      printk(KERN_INFO "\nERROR: timeout waiting for SEPROM_BUSY\n");
      return(PFLASH_SEPROM_BUSY);
    }
  }

  /* Clear The Read/Write cycle of state machine */
  data = GETREG_REGMM32(PFLASH_CNTL);
  MODIFYFLD(data, PFLASH_CNTL, CMD_TRIG,    0);
  MODIFYFLD(data, PFLASH_CNTL, READ_CYCLE,  0);
  MODIFYFLD(data, PFLASH_CNTL, WRITE_CYCLE, 0);
  MODIFYFLD(data, PFLASH_CNTL, SPARE_EN, 1);
  SETREG_REGMM32(PFLASH_CNTL, data);

  /* Clear the status */
  SETREG_REGMM32(PFLASH_STATUS, 0);

  return 0;
}


/*****************************************************************************/

int pflash_get_spare8( uint32_t offset, 
		       uint8_t *data )
{

  pflashClearStateMachine();
  SETFLD_REGMM32(PFLASH_CNTL, SPARE_EN, 1);
  *data = PFLASH_READ_8(offset);
  SETFLD_REGMM32(PFLASH_CNTL, SPARE_EN, 0);
  pflashClearStateMachine();
  /* printk(KERN_INFO "pflash_get_spare8 - offset: %x  data: %x\n", 
     offset, *data); */

  return 0;
}


/*****************************************************************************/

int pflash_set_spare8( uint32_t offset, 
		       uint32_t data )
{

  pflashClearStateMachine();
  SETFLD_REGMM32(PFLASH_CNTL, SPARE_EN, 1);
  PFLASH_WRITE_8(offset, data);
  SETFLD_REGMM32(PFLASH_CNTL, SPARE_EN, 0);
  pflashClearStateMachine();
  return 0;

}


/* 

   Xilleon PCU device NAND-flexbus arbiter

   pcuSetNandAcces
   pcuSetFlexbusAccess

   To save the default flexbus aperture settings in order to restore them after 
   nand access done.

   default (Flexbus):  BOOTROM_SEL - 1110/1111, FBUS_SELECT - 0 
   NAND flash:         BOOTROM_SEL - 0100,      FBUS_SELECT - bit 3
   serial flash:       BOOTROM_SEL - 0111,1xxx except 1110/1111, FBUS_SELECT - bit 2

*/

#ifdef CONFIG_PCU_ARBITER

int pcuSetNandAccess(void)
{

  int r=0;
  u32 flashsize, defnum;


#ifdef DEBUG_PCU_ARBITER
  printk("pcuSetNandAccess - fbcAperCtrl[0]=0x%x\n", fbcAperCtrl[0]);
#endif

  /* save the current FBC aperture settings */
  fbcAperCtrl[0] = WC_GETREG_REGMM32(FBC_APER0_CNTL);
  fbcAperCtrl[1] = WC_GETREG_REGMM32(FBC_APER1_CNTL);
  fbcAperCtrl[2] = WC_GETREG_REGMM32(FBC_APER2_CNTL);
  fbcAperCtrl[3] = WC_GETREG_REGMM32(FBC_APER3_CNTL);
  fbcAperCtrl[4] = WC_GETREG_REGMM32(FBC_APER4_CNTL);
  fbcAperCtrl[5] = WC_GETREG_REGMM32(FBC_APER5_CNTL);

  stw_flashinfo(&flashsize, &defnum);
  pcuSetMode(PCU_MODE_PFLASH_32MBIT);  

   r = GETREG_REGMM32(PFLASH_CNTL);
  MODIFYFLD(r, PFLASH_CNTL, PF32MBIT_EN, 0);
  MODIFYFLD(r, PFLASH_CNTL, PF64MBIT_EN, 0);
  MODIFYFLD(r, PFLASH_CNTL, PF128MBIT_EN, 1); 
  SETREG_REGMM32(PFLASH_CNTL, r); 

  pcuSetMode(PCU_MODE_PFLASH_128MBIT);  
  pcuNand = 1;

  return 0;
}


int pcuSetFlexbusAccess(void)
{

#ifdef DEBUG_PCU_ARBITER
  printk("pcuSetFlexbusAccess - fbcAperCtrl[0]=0x%x\n", fbcAperCtrl[0]);
#endif


  /* turn off serial flash aperture */
  SETFLD_REGMM32(APER_PCU_SFLASH_CNTL, APERSIZE, 0);

  /* turn off NAND aperture */
  SETFLD_REGMM32(APER_PCU_PFLASH_CNTL, APERSIZE, 0);

  /* disable serial flash registers */
  if(GETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN))  
    SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN,  0);
  if(GETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN))  
    SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN,  0);
  if(GETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN))  
    SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN,  0);
  if(GETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN)) 
    SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 0);
  if(GETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN)) 
    SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 0);

  /* disable NAND registers */
  SETFLD_REGMM32(PFLASH_CNTL, PF32MBIT_EN, 0);
  SETFLD_REGMM32(PFLASH_CNTL, PF64MBIT_EN, 0);
  SETFLD_REGMM32(PFLASH_CNTL, PF128MBIT_EN, 0);

  /* set flexbus mux to flexbus controller apertures */
  SETREG_REGMM32(FBUS_SELECT, 0x0);

  /* board specific switch to flexbus */
  sbdSelectFlexBus();

  /* restore the default flexbus aperture settings */ 
  WC_SETREG_REGMM32(FBC_APER0_CNTL, fbcAperCtrl[0]);
  WC_SETREG_REGMM32(FBC_APER1_CNTL, fbcAperCtrl[1]);
  WC_SETREG_REGMM32(FBC_APER2_CNTL, fbcAperCtrl[2]);
  WC_SETREG_REGMM32(FBC_APER3_CNTL, fbcAperCtrl[3]);
  WC_SETREG_REGMM32(FBC_APER4_CNTL, fbcAperCtrl[4]);
  WC_SETREG_REGMM32(FBC_APER5_CNTL, fbcAperCtrl[5]);

  pcuNand = 0;

  return 0;

}

#endif


/************************************************************************/

void dump_nand_chip_info (struct nand_chip *this)
{

  DEBUG (MTD_DEBUG_LEVEL3, "nand_chip struct:\n");
  DEBUG (MTD_DEBUG_LEVEL3, "IO_ADDR: \n");
  DEBUG (MTD_DEBUG_LEVEL3, "CTRL_ADDR: \n");
  DEBUG (MTD_DEBUG_LEVEL3, "CLE: \n");
  DEBUG (MTD_DEBUG_LEVEL3, "ALE: \n");
  DEBUG (MTD_DEBUG_LEVEL3, "NCE: \n");
  DEBUG (MTD_DEBUG_LEVEL3, "chip_lock: 0x%x\n", this->chip_lock);
  DEBUG (MTD_DEBUG_LEVEL3, "wait_queue: 0x%x\n", this->wq);
  DEBUG (MTD_DEBUG_LEVEL3, "nand_state: 0x%x\n", this->state);
  DEBUG (MTD_DEBUG_LEVEL3, "page_shift: 0x%x\n", this->page_shift);

#ifdef CONFIG_MTD_NAND_ECC
  DEBUG (MTD_DEBUG_LEVEL3, "ecc_code_buf: 0x%x, 0x%x, 0x%x, 0x%x, 0x%x, 0x%x\n", 
	 this->ecc_code_buf[0], this->ecc_code_buf[1], this->ecc_code_buf[2], 
	 this->ecc_code_buf[3], this->ecc_code_buf[4], this->ecc_code_buf[5]);
#endif

}


/*****************************************************************************/

void dump_mtd_info (struct mtd_info *mtd)
{

  DEBUG (MTD_DEBUG_LEVEL3, "mtd_info struct:\n");
  DEBUG (MTD_DEBUG_LEVEL3, "type: 0x%x\n", mtd->type);
  DEBUG (MTD_DEBUG_LEVEL3, "flags: 0x%x\n", mtd->flags);
  DEBUG (MTD_DEBUG_LEVEL3, "total size: 0x%x\n", mtd->size);
  DEBUG (MTD_DEBUG_LEVEL3, "erasesize: 0x%x\n", mtd->erasesize);
  DEBUG (MTD_DEBUG_LEVEL3, "oobblock: 0x%x\n", mtd->oobblock);
  DEBUG (MTD_DEBUG_LEVEL3, "oobsize: 0x%x\n", mtd->oobsize);
  DEBUG (MTD_DEBUG_LEVEL3, "ecctype: 0x%x\n", mtd->ecctype );
  DEBUG (MTD_DEBUG_LEVEL3, "eccsize: 0x%x\n", mtd->eccsize);
  DEBUG (MTD_DEBUG_LEVEL3, "NAND name: %s\n", mtd->name);
  DEBUG (MTD_DEBUG_LEVEL3, "index: 0x%x\n", mtd->index);
  DEBUG (MTD_DEBUG_LEVEL3, "number of erase regions: 0x%x\n", mtd->numeraseregions);
  DEBUG (MTD_DEBUG_LEVEL3, "mtd->read: 0x%x\n", mtd->read);
  DEBUG (MTD_DEBUG_LEVEL3, "mtd->read_oob: 0x%x\n", mtd->read_oob);

}

/************************************************************************************/

/* NOR Flash functions */

int nflashInit(void)
{
    uint32 flash_mask = 0;

    pcuSetMode(PCU_MODE_FLEXBUS);

#ifdef NFLASH0_NUM
    if(nflashExists(NFLASH0_APER, NFLASH0_BASE)) flash_mask |= (1<<NFLASH0_NUM);
#endif
#ifdef NFLASH1_NUM
    if(nflashExists(NFLASH1_APER, NFLASH1_BASE)) flash_mask |= (1<<NFLASH1_NUM);
#endif
#ifdef NFLASH2_NUM
    if(nflashExists(NFLASH2_APER, NFLASH2_BASE)) flash_mask |= (1<<NFLASH2_NUM);
#endif
#ifdef NFLASH3_NUM
    if(nflashExists(NFLASH3_APER, NFLASH3_BASE)) flash_mask |= (1<<NFLASH3_NUM);
#endif
#ifdef NFLASH4_NUM
    if(nflashExists(NFLASH4_APER, NFLASH4_BASE)) flash_mask |= (1<<NFLASH4_NUM);
#endif
#ifdef NFLASH5_NUM
    if(nflashExists(NFLASH5_APER, NFLASH5_BASE)) flash_mask |= (1<<NFLASH5_NUM);
#endif
#ifdef NFLASH6_NUM
    if(nflashExists(NFLASH6_APER, NFLASH6_BASE)) flash_mask |= (1<<NFLASH6_NUM);
#endif
#ifdef NFLASH7_NUM
    if(nflashExists(NFLASH7_APER, NFLASH7_BASE)) flash_mask |= (1<<NFLASH7_NUM);
#endif

    return flash_mask;
}

/************************************************************************/

uint32 nflashGetType(uint32 aper, uint32 baseoffset)
{
    uint32 type;

    pcuFlexBusWrite8(aper, baseoffset+(0x5555<<1), 0xAA);
    pcuFlexBusWrite8(aper, baseoffset+(0x2AAA<<1), 0x55);
    pcuFlexBusWrite8(aper, baseoffset+(0x5555<<1), 0x90);

    type = pcuFlexBusRead8(aper, baseoffset+(1<<1));

    pcuFlexBusWrite8(aper, baseoffset+(0x88), 0xF0);

    return type;
}

uint32 nflashGetSize(uint32 type)
{
    switch(type)
    {
    case NFT_AT49xV16x4: return 2*MEG;
    case NFT_AT49xV32x:  return 4*MEG;
    case NFT_AT29LV320D: return 4*MEG;
    case NFT_AT29LV64xD: return 8*MEG;
    }

    return 0;
}


static int nflashExists(uint32 aper, uint32 baseoffset)
{
    uint32 type;

    type = nflashGetType(aper, baseoffset);

    if((type == NFT_AT49xV16x4) || (type == NFT_AT49xV32x) || (type == NFT_AT29LV320D))
        return 1;
    else
        return 0;
}


/************************************************************************************/


/* Serial Flash */


int sflashInit(void)
{
    uint32 flash_mask = 0;

    // assume sflash exists only if strapped for sflash boot
    switch((GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK))
    {
    case STRAPS__BOOTROM_SEL__ST_M25P05:          sflash_pcu_mode = PCU_MODE_SFLASH_SEPST05;  sflash_exists = 1; break;
    case STRAPS__BOOTROM_SEL__ST_M25P10:          sflash_pcu_mode = PCU_MODE_SFLASH_SEPST10;  sflash_exists = 1; break;
    case STRAPS__BOOTROM_SEL__ST_M25P20:          sflash_pcu_mode = PCU_MODE_SFLASH_SEPST20;  sflash_exists = 1; break;
    case STRAPS__BOOTROM_SEL__ST_M25P40:          sflash_pcu_mode = PCU_MODE_SFLASH_SEPST40;  sflash_exists = 1; break;
    case STRAPS__BOOTROM_SEL__ST_M25P80:          sflash_pcu_mode = PCU_MODE_SFLASH_SEPST80;  sflash_exists = 1; break;
    case STRAPS__BOOTROM_SEL__NEXTFLASH_NX25F011: sflash_pcu_mode = PCU_MODE_SFLASH_SEPISSI;  sflash_exists = 1; break;
    case STRAPS__BOOTROM_SEL__ATMEL_AT25F1024:    sflash_pcu_mode = PCU_MODE_SFLASH_SEPATMEL; sflash_exists = 1; break;
    }

    if(sflash_exists)
    {
        pcuSetMode(sflash_pcu_mode);

        sflashCheckIdleAndSoftReset();
        sflashReadEnableSequence();

        flash_mask |= (1<<SFLASH_NUM);
    }

    return flash_mask;
}

/************************************************************************/

int sflashGetPcuMode(void)
{
    return sflash_pcu_mode;
}

/************************************************************************/

char *sflashGetIdString(void)
{
    switch((GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK))
    {
    case STRAPS__BOOTROM_SEL__ST_M25P05:          return "ST Microelectronics M25P05";
    case STRAPS__BOOTROM_SEL__ST_M25P10:          return "ST Microelectronics M25P10";
    case STRAPS__BOOTROM_SEL__ST_M25P20:          return "ST Microelectronics M25P20";
    case STRAPS__BOOTROM_SEL__ST_M25P40:          return "ST Microelectronics M25P40";
    case STRAPS__BOOTROM_SEL__ST_M25P80:          return "ST Microelectronics M25P80";
    case STRAPS__BOOTROM_SEL__NEXTFLASH_NX25F011: return "NEXFLASH NX25F011";
    case STRAPS__BOOTROM_SEL__ATMEL_AT25F1024:    return "ATMEL AT25F1024";
    }

    return "Unknown";
}

/************************************************************************/

int sflashGetChipSize(void)
{
    switch((GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK))
    {
    case STRAPS__BOOTROM_SEL__ST_M25P05:          return 64*1024;
    case STRAPS__BOOTROM_SEL__ST_M25P10:          return 128*1024;
    case STRAPS__BOOTROM_SEL__ST_M25P20:          return 256*1024;
    case STRAPS__BOOTROM_SEL__ST_M25P40:          return 512*1024;
    case STRAPS__BOOTROM_SEL__ST_M25P80:          return 1024*1024;
    case STRAPS__BOOTROM_SEL__NEXTFLASH_NX25F011: return 132*1024;
    case STRAPS__BOOTROM_SEL__ATMEL_AT25F1024:    return 128*1024;
    }

    return 0;
}

/************************************************************************/

int sflashGetBlockSize(int block)
{
    switch((GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK))
    {
    case STRAPS__BOOTROM_SEL__ST_M25P05:          return 0x8000;
    case STRAPS__BOOTROM_SEL__ST_M25P10:          return 0x8000;
    case STRAPS__BOOTROM_SEL__ST_M25P20:          return 0x10000;
    case STRAPS__BOOTROM_SEL__ST_M25P40:          return 0x10000;
    case STRAPS__BOOTROM_SEL__ST_M25P80:          return 0x10000;
    case STRAPS__BOOTROM_SEL__NEXTFLASH_NX25F011: return 0x2100;
    case STRAPS__BOOTROM_SEL__ATMEL_AT25F1024:    return 0x8000;
    }

    return 0;
}

/************************************************************************/

int sflashGetTotalBlocks(void)
{
    switch((GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK))
    {
    case STRAPS__BOOTROM_SEL__ST_M25P05:          return 2;
    case STRAPS__BOOTROM_SEL__ST_M25P10:          return 4;
    case STRAPS__BOOTROM_SEL__ST_M25P20:          return 4;
    case STRAPS__BOOTROM_SEL__ST_M25P40:          return 8;
    case STRAPS__BOOTROM_SEL__ST_M25P80:          return 16;
    case STRAPS__BOOTROM_SEL__NEXTFLASH_NX25F011: return 16;
    case STRAPS__BOOTROM_SEL__ATMEL_AT25F1024:    return 4;
    }

    return 0;
}

/************************************************************************/

int sflashGetBlockStartOffset(UINT32 block)
{
    switch((GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK))
    {
    case STRAPS__BOOTROM_SEL__ST_M25P05:          return block * 0x8000;
    case STRAPS__BOOTROM_SEL__ST_M25P10:          return block * 0x8000;
    case STRAPS__BOOTROM_SEL__ST_M25P20:          return block * 0x10000;
    case STRAPS__BOOTROM_SEL__ST_M25P40:          return block * 0x10000;
    case STRAPS__BOOTROM_SEL__ST_M25P80:          return block * 0x10000;
    case STRAPS__BOOTROM_SEL__NEXTFLASH_NX25F011: return 0;
    case STRAPS__BOOTROM_SEL__ATMEL_AT25F1024:    return 0;
    }

    return 0;
}

/************************************************************************/

int sflashChipErase(void)
{
    UINT32 regVal;
    UINT32 timeout;

    if(sflash_exists)
    {
        regVal = GETREG(SFLASH_CNTL1);
        MODIFYFLD(regVal,SFLASH_CNTL1,WRITE_ENABLE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,BCNT_OVER_WTE_EN, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,BYTE_CNT, 0x0);
#ifdef CONFIG_STW4X225
        MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x2);
#endif
        SETREG(SFLASH_CNTL1,regVal);

        SETFLD(SFLASH_CNTL2_STATUS,SEC_COMMAND, 0x6);
        SETMEM_SFLASHMM8(0x0,0x0);

        if(sflashCheckIdleAndSoftReset()) return -1;

        regVal = GETREG(SFLASH_CNTL1);
        MODIFYFLD(regVal,SFLASH_CNTL1,WRITE_ENABLE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,BYTE_CNT, 0x0);
#ifdef CONFIG_STW4X225
        MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x2);
#endif
        SETREG(SFLASH_CNTL1,regVal);

        SETFLD(SFLASH_CNTL2_STATUS,SEC_COMMAND, 0xC7); //chip erase command
        SETMEM_SFLASHMM8(0x0,0x0); //command trigger

        if(sflashCheckIdleAndSoftReset()) return -1;

        regVal=GETREG(SFLASH_CNTL1);
        MODIFYFLD(regVal,SFLASH_CNTL1,WRITE_ENABLE, 0x0);
        MODIFYFLD(regVal,SFLASH_CNTL1,BCNT_OVER_WTE_EN, 1);
        MODIFYFLD(regVal,SFLASH_CNTL1,BYTE_CNT, 0x3);
#ifdef CONFIG_STW4X225
        MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x2);
#endif
        SETREG(SFLASH_CNTL1,regVal);

        SETFLD(SFLASH_CNTL2_STATUS,SEC_COMMAND, 0x0);

#ifdef CONFIG_STWX225
        regVal = GETREG(SFLASH_CNTL1);
        MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x1);
        SETREG(SFLASH_CNTL1,regVal);
        SETFLD(SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE,0x0);
#endif

        switch((GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK))
        {
        case STRAPS__BOOTROM_SEL__ST_M25P05: timeout =  6000; break;
        case STRAPS__BOOTROM_SEL__ST_M25P10: timeout =  4000; break;
        case STRAPS__BOOTROM_SEL__ST_M25P20: timeout =  6000; break;
        case STRAPS__BOOTROM_SEL__ST_M25P40: timeout = 10000; break;
        case STRAPS__BOOTROM_SEL__ST_M25P80: timeout = 20000; break;
        default:                             timeout = SFLASH_DEFAULT_CHIP_ERASE_TIMEOUT; break;
        }

        if(sflashCheckWriteInProgress(timeout)) return -1;
        if(sflashCheckStatusRegister())  return -1;

        return 0;
    }
    else
    {
        printk("ERROR: sflashChipErase: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

int sflashBlockErase(int block)
{
    UINT32 regVal;
    UINT32 timeout;

    if(sflash_exists)
    {
        regVal = GETREG(SFLASH_CNTL1);
        MODIFYFLD(regVal,SFLASH_CNTL1,WRITE_ENABLE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,BCNT_OVER_WTE_EN, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,BYTE_CNT, 0x0);
#ifdef CONFIG_X225
        MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x2);
#endif
        SETREG(SFLASH_CNTL1,regVal);

        SETFLD(SFLASH_CNTL2_STATUS,SEC_COMMAND, 0x6);

        SETMEM_SFLASHMM8(0x0,0x0);

        if(sflashCheckStatusRegister()) return -1; 
        if(sflashCheckIdleAndSoftReset()) return -1;

        regVal = GETREG(SFLASH_CNTL1);
        MODIFYFLD(regVal,SFLASH_CNTL1,WRITE_ENABLE, 0x1);
        //MODIFYFLD(regVal,SFLASH_CNTL1,BCNT_OVER_WTE_EN, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,BYTE_CNT, 0x0);
#ifdef CONFIG_STW4X225
        MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x2);
#endif
        SETREG(SFLASH_CNTL1,regVal);

        regVal=GETREG(SFLASH_CNTL2_STATUS);
        MODIFYFLD(regVal,SFLASH_CNTL2_STATUS,SEC_COMMAND, 0xD8);
        MODIFYFLD(regVal,SFLASH_CNTL2_STATUS,SECTOR_ERASE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL2_STATUS,SECTOR_TO_ERASE, block);

        SETMEM_SFLASHMM8(0x0,0x0);

        if(sflashCheckStatusRegister()) return -1; 
        if(sflashCheckIdleAndSoftReset()) return -1;

        regVal=GETREG(SFLASH_CNTL1);
        MODIFYFLD(regVal,SFLASH_CNTL1,WRITE_ENABLE, 0x0);
        MODIFYFLD(regVal,SFLASH_CNTL1,BCNT_OVER_WTE_EN, 1);
        MODIFYFLD(regVal,SFLASH_CNTL1,BYTE_CNT, 0x3);
#ifdef CONFIG_STW4X225
        MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x2);
#endif
        SETREG(SFLASH_CNTL1,regVal);

        SETFLD(SFLASH_CNTL2_STATUS,SEC_COMMAND, 0x0);

#ifdef CONFIG_STW4X225
        regVal = GETREG(SFLASH_CNTL1);
        MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
        MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x1);
        SETREG(SFLASH_CNTL1,regVal);
        SETFLD(SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE,0x0);
#endif

        switch((GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK))
        {
        case STRAPS__BOOTROM_SEL__ST_M25P05: timeout = 3000; break;
        case STRAPS__BOOTROM_SEL__ST_M25P10: timeout = 2000; break;
        case STRAPS__BOOTROM_SEL__ST_M25P20: timeout = 3000; break;
        case STRAPS__BOOTROM_SEL__ST_M25P40: timeout = 3000; break;
        case STRAPS__BOOTROM_SEL__ST_M25P80: timeout = 3000; break;
        default:                             timeout = SFLASH_DEFAULT_BLOCK_ERASE_TIMEOUT; break;
        }

        if(sflashCheckWriteInProgress(timeout)) return -1;
        if(sflashCheckStatusRegister()) return -1; 

        return 0;
    }
    else
    {
        printk("ERROR: sflashBlockErase: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

int sflashBlockEraseVerify(uint32 block)
{
    uint32 offset;
    uint32 blocksize;
    uint32 blockoffset;
    uint8 actual;

    if(sflash_exists)
    {
        blocksize = sflashGetBlockSize(block);
        blockoffset = sflashGetBlockStartOffset(block);

        sflashReadEnableSequence();

        for(offset=0; offset<blocksize; offset++)
        {
            actual = GETMEM_SFLASHMM8(blockoffset + offset);

            if(actual != 0xFF)
            {
                printk("\nERROR: sflashBlockEraseVerify: block=%d, offset=%d, actual=0x%02X, expected=0xFF\n", block, offset, actual);
                return -1;
            }
        }

        return 0;
    }
    else
    {
        printk("ERROR: sflashBlockEraseVerify: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

int sflashChipWrite(uint32 block_start, char *addr, uint32 size)
{
    uint32 block;
    uint32 blocks;
    uint32 blocksize;
    uint32 totalblocks;
    uint32 blocksremaining;

    if(sflash_exists)
    {
        totalblocks = sflashGetTotalBlocks();
        blocksize = sflashGetBlockSize(0);
        blocks = blocksremaining = (size+blocksize-1) / blocksize;

        printk("Erasing chip...\n");

        if(sflashChipErase() == 0)
        {
            printk("Writing/verifying %d blocks starting at %d...\n", blocks, block_start);

            for(block=block_start; blocksremaining && (block<totalblocks); block++)
            {
                printk("block %d: ", block);

                printk("W");
                if(sflashBlockWrite(block, addr)) break;

                printk("V");
                if(sflashBlockVerify(block, addr)) break;

                addr += blocksize;
                blocksremaining -= 1;

                printk("\n");
            }

            printk("\n");
        }

        printk("%d blocks written.\n", blocks-blocksremaining);

        return 0;
    }
    else
    {
        printk("ERROR: sflashChipWrite: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

sflashChipRead(uint32 block_start, char *addr, uint32 blocks)
{
    uint32 blocksize;
    uint32 totalblocks;
    uint32 block;
    uint32 blocksremaining;

    if(sflash_exists)
    {
        blocksize = sflashGetBlockSize(0);
        totalblocks = blocksremaining = sflashGetTotalBlocks();

        for(block=block_start; blocks && (block<totalblocks); block++)
        {
            sflashBlockRead(block, addr);

            printk(".");

            addr += blocksize;
            blocksremaining -= 1;
        }

        printk("\n%d blocks read.\n", blocks-blocksremaining);

        return 0;
    }
    else
    {
        printk("ERROR: sflashChipRead: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

#define BURST_SIZE 128

int sflashBlockWrite(uint32 block, char *data) 
{
    uint32 offset;
    uint32 regVal;
    uint32 i;
    uint32 blocksize = sflashGetBlockSize(block);
    uint32 blockoffset = sflashGetBlockStartOffset(block);

    if(sflash_exists)
    {
        for(offset=0; offset<blocksize; offset+=BURST_SIZE)
        {
            if(!(offset & 0x3FF)) { printk("."); }

            //pre-write sequence
            regVal = GETREG(SFLASH_CNTL1);
            MODIFYFLD(regVal,SFLASH_CNTL1,WRITE_ENABLE, 0x1);
            MODIFYFLD(regVal,SFLASH_CNTL1,BCNT_OVER_WTE_EN, 0x1);
            MODIFYFLD(regVal,SFLASH_CNTL1,READ_STATUS, 0x0);
            MODIFYFLD(regVal,SFLASH_CNTL1,BYTE_CNT, 0x0);
#ifdef CONFIG_STW4X225
            MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
            MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x2);
#endif
            SETREG(SFLASH_CNTL1,regVal);

            SETFLD(SFLASH_CNTL2_STATUS,SEC_COMMAND, 0x6);

            SETMEM_SFLASHMM8(0x0,0x0);

            if(sflashCheckStatusRegister()) return -1;
            if(sflashCheckIdleAndSoftReset()) return -1;

            //set up the byte count
            regVal=GETREG(SFLASH_CNTL1);
            MODIFYFLD(regVal,SFLASH_CNTL1, WRITE_ENABLE, 0x0);
            MODIFYFLD(regVal,SFLASH_CNTL1, BCNT_OVER_WTE_EN, 0x1);
            MODIFYFLD(regVal,SFLASH_CNTL1, BYTE_CNT, BURST_SIZE-1); // 0x0 for 1 byte
            SETREG(SFLASH_CNTL1,regVal);

            SETFLD(SFLASH_CNTL2_STATUS,SEC_COMMAND, 0x0); //Write Enable Finished

            for(i=0; i<BURST_SIZE; i++)
                SETMEM_SFLASHMM8(blockoffset + offset + i, *data++);

            if(sflashCheckWriteInProgress(SFLASH_DEFAULT_WRITE_TIMEOUT)) return -1;
            if(sflashCheckStatusRegister()) return -1;
            if(sflashCheckIdleAndSoftReset()) return -1;
        }

        sflashReadEnableSequence();

        return 0;
    }
    else
    {
        printk("ERROR: sflashBlockWrite: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

int sflashBlockVerify(uint32 block, char *buffer)
{
    uint32 offset;
    uint32 blocksize;
    uint32 blockoffset;
    uint8 actual;
    uint8 expected;

    if(sflash_exists)
    {
        blocksize = sflashGetBlockSize(block);
        blockoffset = sflashGetBlockStartOffset(block);

        sflashReadEnableSequence();

        for(offset=0; offset<blocksize; offset++)
        {
            actual = GETMEM_SFLASHMM8(blockoffset + offset);
            expected = *buffer++;

            if(actual != expected)
            {
                printk("\nERROR: sflashBlockVerify: block=%d, offset=%d, actual=0x%02X, expected=%02X\n", block, offset, actual, expected);
                return -1;
            }
        }

        return 0;
    }
    else
    {
        printk("ERROR: sflashBlockWriteVerify: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

int sflashBlockRead(uint32 block, char *data) 
{
    uint32 offset;
    uint32 blocksize;
    uint32 blockoffset;
    uint32 *p = (uint32 *) data;

    if(sflash_exists)
    {
        blocksize = sflashGetBlockSize(block);
        blockoffset = sflashGetBlockStartOffset(block);

        sflashReadEnableSequence();

        for(offset=0; offset<blocksize; offset+=4)
        {
            *p++ = GETMEM_SFLASHMM32(blockoffset + offset);
        }

        return 0;
    }
    else
    {
        printk("ERROR: sflashBlockRead: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

int sflashCheckIdleAndSoftReset(void)
{
    return 0;
}

/************************************************************************/

int sflashCheckStatusRegister(void)
{
    UINT32 timeout;

    if(sflash_exists)
    {
        timeout = SFLASH_DEFAULT_STATUS_TIMEOUT * FLASH_MS_TIMER_COUNT;

        while (1)
        {
            if((GETFLD(SFLASH_CNTL2_STATUS,SFLASH_BUSY) |
                GETFLD(SFLASH_CNTL2_STATUS,ROMPARIF_BUSY) |
                GETFLD(SFLASH_CNTL2_STATUS,PARIF_BUSY) |
                GETFLD(SFLASH_CNTL2_STATUS,PARIFROM_BUSY) |
                GETFLD(SFLASH_CNTL2_STATUS,READY_BUSY) |
                GETFLD(SFLASH_CNTL2_STATUS,SEPROM_BUSY)) == 0)
            {
                return 0;
            }

            if (timeout-- == 0)
            {
                printk("ERROR: sflashCheckStatusRegister timeout\n");
                return -1;
            }

            udelay(1000);
        }
    }
    else
    {
        printk("ERROR: sflashCheckStatusRegister: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

int sflashCheckWriteInProgress(uint32 timeout)
{
    UINT32 reg;

    if(sflash_exists)
    {
        timeout *= FLASH_MS_TIMER_COUNT;

        while(1)
        {
            reg = GETREG(SFLASH_CNTL1);
            MODIFYFLD(reg,SFLASH_CNTL1,WRITE_ENABLE, 0x0);
            MODIFYFLD(reg,SFLASH_CNTL1,BCNT_OVER_WTE_EN, 0x1);
            MODIFYFLD(reg,SFLASH_CNTL1,READ_STATUS, 0x1);
            MODIFYFLD(reg,SFLASH_CNTL1,BYTE_CNT, 0x0);
#ifdef CONFIG_STW4X225
            MODIFYFLD(reg,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
            MODIFYFLD(reg,SFLASH_CNTL1,SCK_PRESCALE, 0x2);
#endif
            SETREG(SFLASH_CNTL1,reg);

            SETFLD(SFLASH_CNTL2_STATUS,SEC_COMMAND, 0x5);

            reg = GETMEM_SFLASHMM8(0x0);

            SETFLD(SFLASH_CNTL2_STATUS,SEC_COMMAND, 0x0);
            SETFLD(SFLASH_CNTL1,READ_STATUS,0x0);

            if(!(reg & 0x1))
            {
                return 0;
            }

            if(timeout-- == 0)
            {
                printk("ERROR: sflashCheckWriteInProgress timeout\n");
                return -1;
            }

            udelay(1000);
        }

        return 0;
    }
    else
    {
        printk("ERROR: sflashCheckWriteInProgress: sflash not initialized\n");
    }

    return -1;
}

/************************************************************************/

void sflashReadEnableSequence(void)
{
#ifdef CONFIG_STW4X225
    UINT32 regVal;

    regVal = GETREG(SFLASH_CNTL1);
    MODIFYFLD(regVal,SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
    MODIFYFLD(regVal,SFLASH_CNTL1,SCK_PRESCALE, 0x1);
    SETREG(SFLASH_CNTL1,regVal);

    SETFLD(SFLASH_CNTL1,WTRIG_PRE_FLD_SCK_PRESCALE, 0x0);
#endif
}

/************************************************************************/

int sflashGetBufferSize(uint32 imagesize)
{
    if(sflash_exists)
    {
        uint32 block_size = sflashGetBlockSize(0);

        return ((imagesize + block_size - 1) / block_size) * block_size;
    }
    else
    {
        printk("ERROR: sflashBlockEraseVerify: sflash not initialized\n");

        return 0;
    }
}



void stw_flashinfo (int *flashsize, int *default_num)
{
    uint32 bootrom_straps;

    *flashsize = 0;
    *default_num = 0;

    bootrom_straps = GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK;

    switch(bootrom_straps)
    {
    case STRAPS__BOOTROM_SEL__NAND_32Mb:
    case STRAPS__BOOTROM_SEL__NAND_64Mb:
    case STRAPS__BOOTROM_SEL__NAND_128Mb:
#ifdef STW_FLASH_DEBUG
      printk("STW5X226 DEBUG - bootstrap: NAND flash\n");
#endif
      flash_mask |= pflashInit();
      if(flash_mask & (1<<PFLASH1_NUM))
        {
	  *flashsize = pflashGetChipSize(CS1);
	  *default_num = PFLASH1_NUM;
        }
        else if(flash_mask & (1<<PFLASH0_NUM))
        {
	  *flashsize = pflashGetChipSize(CS0);
	  *default_num = PFLASH0_NUM;
        }
        break;

    case STRAPS__BOOTROM_SEL__ST_M25P80:
    case STRAPS__BOOTROM_SEL__ST_M25P10:
    case STRAPS__BOOTROM_SEL__ST_M25P05:
    case STRAPS__BOOTROM_SEL__ST_M25P20:
    case STRAPS__BOOTROM_SEL__NEXTFLASH_NX25F011:
    case STRAPS__BOOTROM_SEL__ATMEL_AT25F1024:
    case STRAPS__BOOTROM_SEL__ST_M25P40:
#ifdef STW_FLASH_DEBUG
      printk("STW5X226 DEBUG - bootstrap: Serial flash.\n");
#endif
        flash_mask |= pflashInit();
	if(flash_mask & (1<<PFLASH1_NUM))
        {
	  *flashsize = pflashGetChipSize(CS1);
	  *default_num = PFLASH1_NUM;
        }
        else if(flash_mask & (1<<PFLASH0_NUM))
        {
	  *flashsize = pflashGetChipSize(CS0);
	  *default_num = PFLASH0_NUM;
        }
        break;

    case STRAPS__BOOTROM_SEL__NOR_8:
    case STRAPS__BOOTROM_SEL__NOR_16:
#ifdef STW_FLASH_DEBUG
      printk("STW5X226 DEBUG - bootstrap: NOR flash.\n");
#endif
      nflashInit();
      flash_mask |= pflashInit();

      if(flash_mask & (1<<PFLASH1_NUM))
        {
	  *flashsize = pflashGetChipSize(CS1);
	  *default_num = PFLASH1_NUM;
        }
      else if(flash_mask & (1<<PFLASH0_NUM))
        {
	  *flashsize = pflashGetChipSize(CS0);
	  *default_num = PFLASH0_NUM;
        }
        break;
    }

    // switch to flexbus mode as default
    //pcuSetMode(PCU_MODE_FLEXBUS);
    //printk("STW5X226 DEBUG - flash size: 0x%x default number: 0x%x\n", *flashsize, *default_num);

}


/* PCU routines */

int pcuSetMode(int mode)
{
    static uint32 old_pcu_mode = PCU_MODE_NONE;

#if !defined(CONFIG_STWX220) && !defined(CONFIG_STW4X225)
    uint32 reg;
#endif

    //printk("X226 DEBUG - pcuSetMode - mode: 0x%x\n", mode);

    if(mode == old_pcu_mode) return 0;

    old_pcu_mode = mode;

    switch(mode)
    {
    case PCU_MODE_FLEXBUS:

#ifndef CONFIG_STWX220
        // turn off sflash aperture
        SETFLD_REGMM32(APER_PCU_SFLASH_CNTL, APERSIZE, 0);

        // turn off pflash aperture
        SETFLD_REGMM32(APER_PCU_PFLASH_CNTL, APERSIZE, 0);

        // disable sflash registers
        if(GETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN))  SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN,  0);
        if(GETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN))  SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN,  0);
        if(GETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN))  SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN,  0);
        if(GETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN)) SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 0);
#ifndef CONFIG_STW4X225
        if(GETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN)) SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 0);
#endif

        // disable pflash registers
        SETFLD_REGMM32(PFLASH_CNTL, PF32MBIT_EN, 0);
        SETFLD_REGMM32(PFLASH_CNTL, PF64MBIT_EN, 0);
        SETFLD_REGMM32(PFLASH_CNTL, PF128MBIT_EN, 0);

        // set flexbus muxing to flexbus (nflash)
        SETREG_REGMM32(FBUS_SELECT, 0x0);
#endif

        // board specific switch to flexbus
        sbdSelectFlexBus();

        // turn on flexbus apertures (if size is not zero)
        SETFLD_REGMM32(APER_PCU_FBC0_ADDR, BASE_ADDRESS, FBC0BASE>>16);
        SETFLD_REGMM32(APER_PCU_FBC0_ADDR, BASE_OUT, 0x00000000);
        SETFLD_REGMM32(APER_PCU_FBC0_CNTL, APERSIZE, FBC0SIZE);
        SETFLD_REGMM32(APER_PCU_FBC1_ADDR, BASE_ADDRESS, FBC1BASE>>16);
        SETFLD_REGMM32(APER_PCU_FBC1_ADDR, BASE_OUT, 0x00000000);
        SETFLD_REGMM32(APER_PCU_FBC1_CNTL, APERSIZE, FBC1SIZE);
        SETFLD_REGMM32(APER_PCU_FBC2_ADDR, BASE_ADDRESS, FBC2BASE>>16);
        SETFLD_REGMM32(APER_PCU_FBC2_ADDR, BASE_OUT, 0x00000000);
        SETFLD_REGMM32(APER_PCU_FBC2_CNTL, APERSIZE, FBC2SIZE);
        SETFLD_REGMM32(APER_PCU_FBC3_ADDR, BASE_ADDRESS, FBC3BASE>>16);
        SETFLD_REGMM32(APER_PCU_FBC3_ADDR, BASE_OUT, 0x00000000);
        SETFLD_REGMM32(APER_PCU_FBC3_CNTL, APERSIZE, FBC3SIZE);
        SETFLD_REGMM32(APER_PCU_FBC4_ADDR, BASE_ADDRESS, FBC4BASE>>16);
        SETFLD_REGMM32(APER_PCU_FBC4_ADDR, BASE_OUT, 0x00000000);
        SETFLD_REGMM32(APER_PCU_FBC4_CNTL, APERSIZE, FBC4SIZE);
        SETFLD_REGMM32(APER_PCU_FBC5_ADDR, BASE_ADDRESS, FBC5BASE>>16);
        SETFLD_REGMM32(APER_PCU_FBC5_ADDR, BASE_OUT, 0x00000000);
        SETFLD_REGMM32(APER_PCU_FBC5_CNTL, APERSIZE, FBC5SIZE);

        return 0;

#ifndef CONFIG_STWX220
    case PCU_MODE_SFLASH_SEPST05:
    case PCU_MODE_SFLASH_SEPST10:
    case PCU_MODE_SFLASH_SEPST20:
    case PCU_MODE_SFLASH_SEPST40:
    case PCU_MODE_SFLASH_SEPST80:
    case PCU_MODE_SFLASH_SEPISSI:
    case PCU_MODE_SFLASH_SEPATMEL:

        // turn off flexbus (nflash) apertures
        SETFLD_REGMM32(APER_PCU_FBC0_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC1_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC2_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC3_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC4_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC5_CNTL, APERSIZE, 0);

        // turn off pflash aperture
        SETFLD_REGMM32(APER_PCU_PFLASH_CNTL, APERSIZE, 0);

        // disable pflash registers
        SETFLD_REGMM32(PFLASH_CNTL, PF32MBIT_EN, 0);
        SETFLD_REGMM32(PFLASH_CNTL, PF64MBIT_EN, 0);
        SETFLD_REGMM32(PFLASH_CNTL, PF128MBIT_EN, 0);

        // set flexbus muxing to sflash
        SETREG_REGMM32(FBUS_SELECT, 0x4);

        // board specific switch to serial flash
        sbdSelectSerialFlash(mode);

        // enable sflash registers
        switch(mode)
        {
        case PCU_MODE_SFLASH_SEPST05:
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 0);
#ifndef CONFIG_STW4X225
            SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 0);
#endif
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN, 1);
            break;
        case PCU_MODE_SFLASH_SEPST10:
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 0);
#ifndef CONFIG_STW4X225
            SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 0);
#endif
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN, 1);
            break;
#ifndef CONFIG_STW4X225
        case PCU_MODE_SFLASH_SEPST20:
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 1);
            break;
        case PCU_MODE_SFLASH_SEPST40:
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 1);
            break;
        case PCU_MODE_SFLASH_SEPST80:
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 1);
            break;
#endif
        case PCU_MODE_SFLASH_SEPISSI:
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 0);
#ifndef CONFIG_STW4X225
            SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 0);
#endif
            SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN, 1);
            break;
        case PCU_MODE_SFLASH_SEPATMEL:
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN, 0);
            SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN, 0);
#ifndef CONFIG_STW4X225
            SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 0);
#endif
            SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 1);
            break;
        }

#ifndef CONFIG_STW4X225
        reg = GETREG(SFLASH_CNTL1);
        MODIFYFLD(reg, SFLASH_CNTL1, WTRIG_PRE_FLD_SCK_PRESCALE, 0x1);
        MODIFYFLD(reg, SFLASH_CNTL1, SCK_PRESCALE, sck_prescale);
        SETREG(SFLASH_CNTL1, reg);
#endif

        // turn on sflash aperture
        SETFLD_REGMM32(APER_PCU_SFLASH_CNTL, APERSIZE, 0xB);

        return 0;
#endif

#ifndef CONFIG_STWX220
    case PCU_MODE_PFLASH_32MBIT: 
    case PCU_MODE_PFLASH_64MBIT: 
    case PCU_MODE_PFLASH_128MBIT:

        // turn off flexbus (nflash) apertures
        SETFLD_REGMM32(APER_PCU_FBC0_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC1_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC2_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC3_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC4_CNTL, APERSIZE, 0);
        SETFLD_REGMM32(APER_PCU_FBC5_CNTL, APERSIZE, 0);

        // turn off sflash aperture
        SETFLD_REGMM32(APER_PCU_SFLASH_CNTL, APERSIZE, 0);

        // disable sflash registers
        if(GETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN))  SETFLD_REGMM32(SFLASH_CNTL1, SEPST05_EN,  0);
        if(GETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN))  SETFLD_REGMM32(SFLASH_CNTL1, SEPST10_EN,  0);
        if(GETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN))  SETFLD_REGMM32(SFLASH_CNTL1, SEPISSI_EN,  0);
        if(GETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN)) SETFLD_REGMM32(SFLASH_CNTL1, SEPATMEL_EN, 0);
#ifndef CONFIG_STW4X225
        if(GETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN)) SETFLD_REGMM32(SFLASH_CNTL2_STATUS, ST25P80_EN, 0);
#endif

        // set flexbus muxing to pflash
        SETREG_REGMM32(FBUS_SELECT, 0x8);

        // board specific switch to nand flash
        sbdSelectNandFlash(mode);

        // enable pflash registers for 32Mb chip
        switch(mode)
        {
        case PCU_MODE_PFLASH_32MBIT:
            SETFLD_REGMM32(PFLASH_CNTL, PF64MBIT_EN, 0);
            SETFLD_REGMM32(PFLASH_CNTL, PF128MBIT_EN, 0);
            SETFLD_REGMM32(PFLASH_CNTL, PF32MBIT_EN, 1);
            break;
        case PCU_MODE_PFLASH_64MBIT:
            SETFLD_REGMM32(PFLASH_CNTL, PF32MBIT_EN, 0);
            SETFLD_REGMM32(PFLASH_CNTL, PF128MBIT_EN, 0);
            SETFLD_REGMM32(PFLASH_CNTL, PF64MBIT_EN, 1);
            break;
        case PCU_MODE_PFLASH_128MBIT:
            SETFLD_REGMM32(PFLASH_CNTL, PF32MBIT_EN, 0);
            SETFLD_REGMM32(PFLASH_CNTL, PF64MBIT_EN, 0);
            SETFLD_REGMM32(PFLASH_CNTL, PF128MBIT_EN, 1);
            break;
        }

        // turn on pflash aperture
        SETFLD_REGMM32(APER_PCU_PFLASH_CNTL, APERSIZE, 0xB);

        return 0;
#endif
    }

    return -1;
}




/*****************************************************************************/

int pcuInit(void)
{
    uint32 r;

    printk("pcuInit start...\n");

#if !defined(CONFIG_STWX220) && !defined(CONFIG_STW4X225)
    sck_prescale = GETFLD_REGMM32(SFLASH_CNTL1, SCK_PRESCALE);
#endif

    SETREG_REGMM32(FBC_MSTORE0,  FBC_MSTORE0_VAL);
    SETREG_REGMM32(FBC_MSTORE1,  FBC_MSTORE1_VAL);
    SETREG_REGMM32(FBC_MSTORE2,  FBC_MSTORE2_VAL);
    SETREG_REGMM32(FBC_MSTORE3,  FBC_MSTORE3_VAL);
    SETREG_REGMM32(FBC_MSTORE4,  FBC_MSTORE4_VAL);
    SETREG_REGMM32(FBC_MSTORE5,  FBC_MSTORE5_VAL);
    SETREG_REGMM32(FBC_MSTORE6,  FBC_MSTORE6_VAL);
    SETREG_REGMM32(FBC_MSTORE7,  FBC_MSTORE7_VAL);
    SETREG_REGMM32(FBC_MSTORE8,  FBC_MSTORE8_VAL);
    SETREG_REGMM32(FBC_MSTORE9,  FBC_MSTORE9_VAL);
    SETREG_REGMM32(FBC_MSTORE10, FBC_MSTORE10_VAL);
    SETREG_REGMM32(FBC_MSTORE11, FBC_MSTORE11_VAL);
    SETREG_REGMM32(FBC_MSTORE12, FBC_MSTORE12_VAL);
    SETREG_REGMM32(FBC_MSTORE13, FBC_MSTORE13_VAL);
    SETREG_REGMM32(FBC_MSTORE14, FBC_MSTORE14_VAL);
    SETREG_REGMM32(FBC_MSTORE15, FBC_MSTORE15_VAL);
    SETREG_REGMM32(FBC_MSTORE16, FBC_MSTORE16_VAL);
    SETREG_REGMM32(FBC_MSTORE17, FBC_MSTORE17_VAL);
    SETREG_REGMM32(FBC_MSTORE18, FBC_MSTORE18_VAL);
    SETREG_REGMM32(FBC_MSTORE19, FBC_MSTORE19_VAL);
    SETREG_REGMM32(FBC_MSTORE20, FBC_MSTORE20_VAL);
    SETREG_REGMM32(FBC_MSTORE21, FBC_MSTORE21_VAL);
    SETREG_REGMM32(FBC_MSTORE22, FBC_MSTORE22_VAL);
    SETREG_REGMM32(FBC_MSTORE23, FBC_MSTORE23_VAL);
    SETREG_REGMM32(FBC_MSTORE24, FBC_MSTORE24_VAL);
    SETREG_REGMM32(FBC_MSTORE25, FBC_MSTORE25_VAL);
    SETREG_REGMM32(FBC_MSTORE26, FBC_MSTORE26_VAL);
    SETREG_REGMM32(FBC_MSTORE27, FBC_MSTORE27_VAL);
    SETREG_REGMM32(FBC_MSTORE28, FBC_MSTORE28_VAL);
    SETREG_REGMM32(FBC_MSTORE29, FBC_MSTORE29_VAL);
    SETREG_REGMM32(FBC_MSTORE30, FBC_MSTORE30_VAL);
    SETREG_REGMM32(FBC_MSTORE31, FBC_MSTORE31_VAL);
    SETREG_REGMM32(FBC_MSTORE32, FBC_MSTORE32_VAL);
    SETREG_REGMM32(FBC_MSTORE33, FBC_MSTORE33_VAL);
    SETREG_REGMM32(FBC_MSTORE34, FBC_MSTORE34_VAL);
    SETREG_REGMM32(FBC_MSTORE35, FBC_MSTORE35_VAL);
    SETREG_REGMM32(FBC_MSTORE36, FBC_MSTORE36_VAL);
    SETREG_REGMM32(FBC_MSTORE37, FBC_MSTORE37_VAL);
    SETREG_REGMM32(FBC_MSTORE38, FBC_MSTORE38_VAL);
    SETREG_REGMM32(FBC_MSTORE39, FBC_MSTORE39_VAL);
    SETREG_REGMM32(FBC_MSTORE40, FBC_MSTORE40_VAL);
    SETREG_REGMM32(FBC_MSTORE41, FBC_MSTORE41_VAL);
    SETREG_REGMM32(FBC_MSTORE42, FBC_MSTORE42_VAL);
    SETREG_REGMM32(FBC_MSTORE43, FBC_MSTORE43_VAL);
    SETREG_REGMM32(FBC_MSTORE44, FBC_MSTORE44_VAL);
    SETREG_REGMM32(FBC_MSTORE45, FBC_MSTORE45_VAL);
    SETREG_REGMM32(FBC_MSTORE46, FBC_MSTORE46_VAL);
    SETREG_REGMM32(FBC_MSTORE47, FBC_MSTORE47_VAL);
    SETREG_REGMM32(FBC_MSTORE48, FBC_MSTORE48_VAL);
    SETREG_REGMM32(FBC_MSTORE49, FBC_MSTORE49_VAL);
    SETREG_REGMM32(FBC_MSTORE50, FBC_MSTORE50_VAL);
    SETREG_REGMM32(FBC_MSTORE51, FBC_MSTORE51_VAL);
    SETREG_REGMM32(FBC_MSTORE52, FBC_MSTORE52_VAL);
    SETREG_REGMM32(FBC_MSTORE53, FBC_MSTORE53_VAL);
    SETREG_REGMM32(FBC_MSTORE54, FBC_MSTORE54_VAL);
    SETREG_REGMM32(FBC_MSTORE55, FBC_MSTORE55_VAL);
    SETREG_REGMM32(FBC_MSTORE56, FBC_MSTORE56_VAL);
    SETREG_REGMM32(FBC_MSTORE57, FBC_MSTORE57_VAL);
    SETREG_REGMM32(FBC_MSTORE58, FBC_MSTORE58_VAL);
    SETREG_REGMM32(FBC_MSTORE59, FBC_MSTORE59_VAL);
    SETREG_REGMM32(FBC_MSTORE60, FBC_MSTORE60_VAL);
    SETREG_REGMM32(FBC_MSTORE61, FBC_MSTORE61_VAL);
    SETREG_REGMM32(FBC_MSTORE62, FBC_MSTORE62_VAL);
    SETREG_REGMM32(FBC_MSTORE63, FBC_MSTORE63_VAL);

    r = WC_GETREG_REGMM32(FBC_APER0_CNTL);
    WC_MODIFYFLD(r, FBC_APER0_CNTL, WRITEVEC, FBC0_WRITEVEC);
    WC_MODIFYFLD(r, FBC_APER0_CNTL, READVEC, FBC0_READVEC);
    WC_SETREG_REGMM32(FBC_APER0_CNTL, r);

    r = WC_GETREG_REGMM32(FBC_APER1_CNTL);
    WC_MODIFYFLD(r, FBC_APER1_CNTL, WRITEVEC, FBC1_WRITEVEC);
    WC_MODIFYFLD(r, FBC_APER1_CNTL, READVEC, FBC1_READVEC);
    WC_MODIFYFLD(r, FBC_APER1_CNTL, DBUSWIDTH, FBC1_DBUSWIDTH);
    WC_MODIFYFLD(r, FBC_APER1_CNTL, ABUSLATCH, FBC1_ABUSLATCH);
    WC_SETREG_REGMM32(FBC_APER1_CNTL, r);

    r = WC_GETREG_REGMM32(FBC_APER2_CNTL);
    WC_MODIFYFLD(r, FBC_APER2_CNTL, WRITEVEC, FBC2_WRITEVEC);
    WC_MODIFYFLD(r, FBC_APER2_CNTL, READVEC, FBC2_READVEC);
    WC_MODIFYFLD(r, FBC_APER2_CNTL, DBUSWIDTH, FBC2_DBUSWIDTH);
    WC_MODIFYFLD(r, FBC_APER2_CNTL, ABUSLATCH, FBC2_ABUSLATCH);
    WC_SETREG_REGMM32(FBC_APER2_CNTL, r);

    r = WC_GETREG_REGMM32(FBC_APER3_CNTL);
    WC_MODIFYFLD(r, FBC_APER3_CNTL, WRITEVEC, FBC3_WRITEVEC);
    WC_MODIFYFLD(r, FBC_APER3_CNTL, READVEC, FBC3_READVEC);
    WC_MODIFYFLD(r, FBC_APER3_CNTL, DBUSWIDTH, FBC3_DBUSWIDTH);
    WC_MODIFYFLD(r, FBC_APER3_CNTL, ABUSLATCH, FBC3_ABUSLATCH);
    WC_SETREG_REGMM32(FBC_APER3_CNTL, r);

    r = WC_GETREG_REGMM32(FBC_APER4_CNTL);
    WC_MODIFYFLD(r, FBC_APER4_CNTL, WRITEVEC, FBC4_WRITEVEC);
    WC_MODIFYFLD(r, FBC_APER4_CNTL, READVEC, FBC4_READVEC);
    WC_MODIFYFLD(r, FBC_APER4_CNTL, DBUSWIDTH, FBC4_DBUSWIDTH);
    WC_MODIFYFLD(r, FBC_APER4_CNTL, ABUSLATCH, FBC4_ABUSLATCH);
    WC_SETREG_REGMM32(FBC_APER4_CNTL, r);

    r = WC_GETREG_REGMM32(FBC_APER5_CNTL);
    WC_MODIFYFLD(r, FBC_APER5_CNTL, WRITEVEC, FBC5_WRITEVEC);
    WC_MODIFYFLD(r, FBC_APER5_CNTL, READVEC, FBC5_READVEC);
    WC_MODIFYFLD(r, FBC_APER5_CNTL, DBUSWIDTH, FBC5_DBUSWIDTH);
    WC_MODIFYFLD(r, FBC_APER5_CNTL, ABUSLATCH, FBC5_ABUSLATCH);
    WC_SETREG_REGMM32(FBC_APER5_CNTL, r);

    pcuSetMode(PCU_MODE_FLEXBUS);

    return 0;
}

int pcuFlexBusGetWidth(int aper)
{
    switch(aper)
    {
    case 0: return (GETFLD_REGMM32(FBC_APER0_CNTL, DBUSWIDTH) ? 16 : 8);
    case 1: return (GETFLD_REGMM32(FBC_APER1_CNTL, DBUSWIDTH) ? 16 : 8);
    case 2: return (GETFLD_REGMM32(FBC_APER2_CNTL, DBUSWIDTH) ? 16 : 8);
    case 3: return (GETFLD_REGMM32(FBC_APER3_CNTL, DBUSWIDTH) ? 16 : 8);
    case 4: return (GETFLD_REGMM32(FBC_APER4_CNTL, DBUSWIDTH) ? 16 : 8);
    case 5: return (GETFLD_REGMM32(FBC_APER5_CNTL, DBUSWIDTH) ? 16 : 8);
    }

    return 0;
}

/*****************************************************************************/

void pcuFlexBusWrite8(uint32 aper, uint32 offset, uint32 value)
{
    switch(aper)
    {
    case 0: SETMEM_PCU0MM8(offset, value); break;
    case 1: SETMEM_PCU1MM8(offset, value); break;
    case 2: SETMEM_PCU2MM8(offset, value); break;
    case 3: SETMEM_PCU3MM8(offset, value); break;
    case 4: SETMEM_PCU4MM8(offset, value); break;
    case 5: SETMEM_PCU5MM8(offset, value); break;
    }
}

/*****************************************************************************/

void pcuFlexBusWrite16(uint32 aper, uint32 offset, uint32 value)
{
    switch(aper)
    {
    case 0: SETMEM_PCU0MM16(offset, value); break;
    case 1: SETMEM_PCU1MM16(offset, value); break;
    case 2: SETMEM_PCU2MM16(offset, value); break;
    case 3: SETMEM_PCU3MM16(offset, value); break;
    case 4: SETMEM_PCU4MM16(offset, value); break;
    case 5: SETMEM_PCU5MM16(offset, value); break;
    }
}

/*****************************************************************************/

void pcuFlexBusWrite32(uint32 aper, uint32 offset, uint32 value)
{
    switch(aper)
    {
    case 0: SETMEM_PCU0MM32(offset, value); break;
    case 1: SETMEM_PCU1MM32(offset, value); break;
    case 2: SETMEM_PCU2MM32(offset, value); break;
    case 3: SETMEM_PCU3MM32(offset, value); break;
    case 4: SETMEM_PCU4MM32(offset, value); break;
    case 5: SETMEM_PCU5MM32(offset, value); break;
    }
}

/*****************************************************************************/

uint32 pcuFlexBusRead8(uint32 aper, uint32 offset)
{
    switch(aper)
    {
    case 0: return GETMEM_PCU0MM8(offset);
    case 1: return GETMEM_PCU1MM8(offset);
    case 2: return GETMEM_PCU2MM8(offset);
    case 3: return GETMEM_PCU3MM8(offset);
    case 4: return GETMEM_PCU4MM8(offset);
    case 5: return GETMEM_PCU5MM8(offset);
    }

    return 0;
}

/*****************************************************************************/

uint32 pcuFlexBusRead16(uint32 aper, uint32 offset)
{
    switch(aper)
    {
    case 0: return GETMEM_PCU0MM16(offset);
    case 1: return GETMEM_PCU1MM16(offset);
    case 2: return GETMEM_PCU2MM16(offset);
    case 3: return GETMEM_PCU3MM16(offset);
    case 4: return GETMEM_PCU4MM16(offset);
    case 5: return GETMEM_PCU5MM16(offset);
    }

    return 0;
}

/*****************************************************************************/

uint32 pcuFlexBusRead32(uint32 aper, uint32 offset)
{
    switch(aper)
    {
    case 0: return GETMEM_PCU0MM32(offset);
    case 1: return GETMEM_PCU1MM32(offset);
    case 2: return GETMEM_PCU2MM32(offset);
    case 3: return GETMEM_PCU3MM32(offset);
    case 4: return GETMEM_PCU4MM32(offset);
    case 5: return GETMEM_PCU5MM32(offset);
    }

    return 0;
}


void sbdSelectFlexBus(void)
{
    sbdSelectNandFlash(pflashGetPcuMode());
}

void sbdSelectNandFlash(int mode)
{
    uint32 r;
    uint32 bootrom_straps = GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK;

    // select pflash by setting GPIO10 if booted from NAND flash
    r = GETREG_REGMM32(GPIO_SEL);
    r &= ~((1<<GPIOSEL__1394_TRANSPORT)|(1<<GPIOSEL__DEBUG_BUS_A));
    SETREG_REGMM32(GPIO_SEL, r);

    r = GETREG_REGMM32(GPIOA_DIR);
    r |= (1<<10);
    SETREG_REGMM32(GPIOA_DIR, r);

    r = GETREG_REGMM32(GPIOA_MASK);
    r |= (1<<10);
    SETREG_REGMM32(GPIOA_MASK, r);

    r = GETREG_REGMM32(GPIOA_DATA);

    switch(bootrom_straps)
    {
    case STRAPS__BOOTROM_SEL__NAND_32Mb:
    case STRAPS__BOOTROM_SEL__NAND_64Mb:
    case STRAPS__BOOTROM_SEL__NAND_128Mb:  r |= (1<<10);   break;
    default:                               r &= ~(1<<10);  break;
    }

    SETREG_REGMM32(GPIOA_DATA, r);
}

/*****************************************************************************/

void sbdSelectSerialFlash(int mode)
{
    uint32 r;
    uint32 bootrom_straps = GETREG_REGMM32(STRAPS_VALUE) & STRAPS__BOOTROM_SEL__MASK;

    // select sflash setting GPIO10 if booted from serial flash
    r = GETREG_REGMM32(GPIO_SEL);
    r &= ~((1<<GPIOSEL__1394_TRANSPORT)|(1<<GPIOSEL__DEBUG_BUS_A));
    SETREG_REGMM32(GPIO_SEL, r);

    r = GETREG_REGMM32(GPIOA_DIR);
    r |= (1<<10);
    SETREG_REGMM32(GPIOA_DIR, r);

    r = GETREG_REGMM32(GPIOA_MASK);
    r |= (1<<10);
    SETREG_REGMM32(GPIOA_MASK, r);

    r = GETREG_REGMM32(GPIOA_DATA);

    switch(bootrom_straps)
    {
    case STRAPS__BOOTROM_SEL__ST_M25P05:
    case STRAPS__BOOTROM_SEL__ST_M25P10:
    case STRAPS__BOOTROM_SEL__ST_M25P20:
    case STRAPS__BOOTROM_SEL__ST_M25P40:
    case STRAPS__BOOTROM_SEL__ST_M25P80:
    case STRAPS__BOOTROM_SEL__NEXTFLASH_NX25F011:
    case STRAPS__BOOTROM_SEL__ATMEL_AT25F1024:     r |= (1<<10);   break;
    default:                                       r &= ~(1<<10);  break;
    }

    SETREG_REGMM32(GPIOA_DATA, r);
}

