/*
 * Robert Lembree, lembree@metrolink.com
 * Copyright (C) 2001, Metro Link, Inc., All rights reserved
 *
 * based on code from Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 1999,2000 MIPS Technologies, Inc.  All rights reserved.
 *
 * ########################################################################
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ########################################################################
 *
 * Putting things on the screen/serial line using PMONs facilities.
 *
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/serialP.h>
#include <linux/serial_reg.h>
#include <asm/system.h>
#include <asm/io.h>
#include <asm/serial.h>



unsigned int xilleon_serial_in(struct async_struct *info, int offset)
{
	return (*(volatile unsigned char *)(info->port + offset) & 0xff);
}


void xilleon_serial_out(struct async_struct *info, int offset,
				int value)
{
#ifdef CONFIG_IKOS
	if (offset == UART_LCR)
		return;
#endif

        *(volatile unsigned char *)(info->port + offset) = (unsigned char)value;
        __asm__("sync\n");

}


static struct serial_state rs_table[] = {
	SERIAL_PORT_DFNS	/* Defined in serial.h */
};


/*
 * Hooks to fake "prom" console I/O before devices 
 * are fully initialized. 
 */
static struct async_struct prom_port_info = {0};

void __init setup_prom_printf(int tty_no) {
	int t;
	struct serial_state *ser = &rs_table[tty_no];

	prom_port_info.state = ser;
	prom_port_info.magic = SERIAL_MAGIC;
	prom_port_info.port = ser->port;
	prom_port_info.flags = ser->flags;

	if (tty_no != 0) {

		/* pmon only sets up uart 0 */

		/*
		 * Clear all interrupts
		 */
		xilleon_serial_in(&prom_port_info, UART_LSR);
		xilleon_serial_in(&prom_port_info, UART_RX);
		xilleon_serial_in(&prom_port_info, UART_IIR);
		xilleon_serial_in(&prom_port_info, UART_MSR);

		/*
		 * Now, initialize the UART 
		 */	
		/* reset DLAB */
		xilleon_serial_out(&prom_port_info, UART_LCR, UART_LCR_WLEN8);
		if (prom_port_info.flags & ASYNC_FOURPORT) {
			prom_port_info.MCR = UART_MCR_DTR | UART_MCR_RTS;
			t = UART_MCR_DTR | UART_MCR_OUT1;
		} else {
			prom_port_info.MCR 
				= UART_MCR_DTR | UART_MCR_RTS | UART_MCR_OUT2;
			t = UART_MCR_DTR | UART_MCR_RTS;
		}

		prom_port_info.MCR = t;		/* no interrupts, please */
		xilleon_serial_out(&prom_port_info, UART_MCR, prom_port_info.MCR);
		
		/* baud rate is fixed to 115200 */
		t = prom_port_info.state->baud_base / 115200;
		/* set DLAB */
		xilleon_serial_out(&prom_port_info, UART_LCR, UART_LCR_WLEN8 | 
				UART_LCR_DLAB);

		/* LS of divisor */
		xilleon_serial_out(&prom_port_info, UART_DLL, t & 0xff  
		/* MS of divisor */);
		xilleon_serial_out(&prom_port_info, UART_DLM, t >> 8);
		/* reset DLAB */
		xilleon_serial_out(&prom_port_info, UART_LCR, UART_LCR_WLEN8);
	}
}

int putPromChar(char c)
{
        if (!prom_port_info.state) { 	/* need to init device first */
		return 0;
	}

	while ((xilleon_serial_in(&prom_port_info, UART_LSR) & UART_LSR_THRE) == 0)
		;

	xilleon_serial_out(&prom_port_info, UART_TX, c);

	return 1;
}

char getPromChar(void)
{
	if (!prom_port_info.state) { 	/* need to init device first */
		return 0;
	}

	while (!(xilleon_serial_in(&prom_port_info, UART_LSR) & 1))
		;

	return(xilleon_serial_in(&prom_port_info, UART_RX));
}

static char buf[1024];

void __init prom_printf(char *fmt, ...)
{
	va_list args;
	int l;
	char *p, *buf_end;
	long flags;

	int putPromChar(char);

	/* Low level, brute force, not SMP safe... */
	save_and_cli(flags);
	va_start(args, fmt);
	l = vsprintf(buf, fmt, args); /* hopefully i < sizeof(buf) */
	va_end(args);

	buf_end = buf + l;

	for (p = buf; p < buf_end; p++) {
		/* Crude cr/nl handling is better than none */
		if(*p == '\n')putPromChar('\r');
		putPromChar(*p);
	}
	restore_flags(flags);
}
