/*
 * BRIEF MODULE DESCRIPTION
 *	Xilleon specific pci support.
 *
 * Author: Pete Popov <ppopov@mvista.com, or source@mvista.com>
 *
 *  Based on the Metrolink port:
 *  Robert Lembree, lembree@metrolink.com
 *  Copyright (C) 2001, Metro Link, Inc., All rights reserved
 *
 * 2002 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/config.h>

#ifdef CONFIG_PCI

#include <linux/types.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/init.h>

#include <asm/ati/pci.h>
#include <asm/ati/xilleon.h>
#include <asm/ati/xilleonint.h>
#include <asm/pci_channel.h>

#define PCI_ACCESS_READ  0
#define PCI_ACCESS_WRITE 1

#undef DEBUG
#ifdef 	DEBUG
#define	DBG(x...)	printk(x)
#else
#define	DBG(x...)	
#endif

static struct resource pci_io_resource = {
	"pci IO space", 
	XILLEON_PCI_IO_BASE,
	XILLEON_PCI_IO_BASE + XILLEON_PCI_IO_SIZE,
	IORESOURCE_IO
};

static struct resource pci_mem_resource = {
	"pci memory space", 
	XILLEON_PCI_MEMORY_BASE,
	XILLEON_PCI_MEMORY_BASE + XILLEON_PCI_MEMORY_SIZE,
	IORESOURCE_MEM
};

extern struct pci_ops xilleon_pci_ops;

struct pci_channel mips_pci_channels[] = {
	//{&xilleon_pci_ops, &pci_io_resource, &pci_mem_resource, 0, 0xa8},
	{&xilleon_pci_ops, &pci_io_resource, &pci_mem_resource, 0, 0xa8},
	{(struct pci_ops *) NULL, (struct resource *) NULL,
	 (struct resource *) NULL, (int) NULL, (int) NULL}
};


static int config_access(unsigned char access_type, struct pci_dev *dev, 
			 unsigned char reg, u32 * data)
{
	u32 cntl_val, cntl_val0, status;

	int offset = ((dev->bus->number << 16) | 
		      (dev->devfn << 8) |
		      (reg/4 << 2));

	if (dev->devfn > 0xa8) {
		*data = 0xffffffff;
		return 0;
	}

	// read and save the current value of the APER_CP_PCIC2_CNTL reg

	cntl_val = cntl_val0 = GETREG_REGMM32(APER_CP_PCIC2_CNTL);

        // change the PCI_ATYPE to configure

	MODIFYFLD(cntl_val, APER_CP_PCIC2_CNTL,
		  PCI_ATYPE, PCI_ATYPE_CFG);

	SETREG_REGMM32(APER_CP_PCIC2_CNTL, cntl_val);

        //read or write from the pcic2 aperture

	if (access_type == PCI_ACCESS_WRITE) {
		(*(volatile u32 *)(XILLEON_PCIC2BASE + offset)) = *data;
	} else {
		*data = (*(volatile u32 *)(XILLEON_PCIC2BASE + offset));
	}
	asm volatile ("sync");

        //restore the APER_CP_PCIC2_CNTL reg

        SETREG_REGMM32(APER_CP_PCIC2_CNTL, cntl_val0);

	status = GETREG_REGMM32(PCIC_COMMAND_STATUS);
	if(status & PCIC_COMMAND_STATUS__RECEIVED_MASTER_ABORT__MASK)
	{
		/* clear master abort */
		SETFLD_REGMM32(PCIC_COMMAND_STATUS, RECEIVED_MASTER_ABORT, 1);
		*data = 0xffffffff;
	}

	if(status & PCIC_COMMAND_STATUS__RECEIVED_TARGET_ABORT__MASK)
	{
		/* clear target abort */
		SETFLD_REGMM32(PCIC_COMMAND_STATUS, RECEIVED_TARGET_ABORT, 1);
		*data = 0xffffffff;
	}

        return 0;
}


int xilleon_read_config_byte(struct pci_dev *dev, int index, u8 * val)
{
	u32 data = 0;
		
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	*val = (data >> ((index & 3) << 3)) & 0xff;

	return PCIBIOS_SUCCESSFUL;	
}


int xilleon_read_config_word(struct pci_dev *dev, int index, u16 * val)
{
	u32 data = 0;
		
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	*val = (data >> ((index & 3) << 3)) & 0xffff;

	return PCIBIOS_SUCCESSFUL;
}

int xilleon_read_config_dword(struct pci_dev *dev, int index, u32 * val)
{
	u32 data = 0;
		
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	*val = data; 

	return PCIBIOS_SUCCESSFUL;	
}


int xilleon_write_config_byte(struct pci_dev *dev, int index, u8 val)
{
	u32 data = 0;
       
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	data = (data & ~(0xff << ((index & 3) << 3))) |
	       (val << ((index & 3) << 3));

	if (config_access(PCI_ACCESS_WRITE, dev, index, &data))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}

int xilleon_write_config_word(struct pci_dev *dev, int index, u16 val)
{
	u32 data = 0;
       
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	data = (data & ~(0xffff << ((index & 3) << 3))) |
	       (val << ((index & 3) << 3));

	if (config_access(PCI_ACCESS_WRITE, dev, index, &data))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}

int xilleon_write_config_dword(struct pci_dev *dev, int index, u32 val)
{
	if (config_access(PCI_ACCESS_WRITE, dev, index, &val))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}

struct pci_ops xilleon_pci_ops = {
	xilleon_read_config_byte,
        xilleon_read_config_word,
	xilleon_read_config_dword,
	xilleon_write_config_byte,
	xilleon_write_config_word,
	xilleon_write_config_dword
};
#endif /* CONFIG_PCI */
