
/*
 * Robert Lembree, lembree@metrolink.com
 * Copyright (C) 2001, Metro Link, Inc., All rights reserved
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 */

#include <asm/ati/xilleon.h>
#include <asm/ati/pci.h>

#include <asm/addrspace.h>

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pci.h>

#include <asm/ati/xilleonint.h>

#define PCI_ACCESS_READ  0
#define PCI_ACCESS_WRITE 1


void PCI_HwInit(void)
{
}


static int
config_access(unsigned char access_type, struct pci_dev *dev,
	      unsigned char reg, u32 *data)
{
	u32 cntl_val, cntl_val0;

	int offset = ((dev->bus->number << 16) | 
		      (dev->devfn << 8) |
		      (reg/4 << 2));

	// read and save the current value of the APER_CP_PCIC2_CNTL reg

	cntl_val = cntl_val0 = GETREG_REGMM32(APER_CP_PCIC2_CNTL);

        // change the PCI_ATYPE to configure

	MODIFYFLD(cntl_val, APER_CP_PCIC2_CNTL,
		  PCI_ATYPE, PCI_ATYPE_CFG);

	SETREG_REGMM32(APER_CP_PCIC2_CNTL, cntl_val);

        //read or write from the pcic2 aperture

	if (access_type == PCI_ACCESS_WRITE) {
		(*(volatile u32 *)(XILLEON_PCIC2BASE + offset)) = *data;
	} else {
		*data = (*(volatile u32 *)(XILLEON_PCIC2BASE + offset));
	}

        //restore the APER_CP_PCIC2_CNTL reg

        SETREG_REGMM32(APER_CP_PCIC2_CNTL, cntl_val0);

        return 0;
}



static int
PCI_readbyte(struct pci_dev *dev, int  index, u8 *byte_read)
{
	u32 data = 0;
		
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	*byte_read = (data >> ((index & 3) << 3)) & 0xff;

	return PCIBIOS_SUCCESSFUL;	
}



static int
PCI_readword(struct pci_dev *dev,
	     int  index,
	     u16 *word_read)
{
	u32 data = 0;
		
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	*word_read = (data >> ((index & 3) << 3)) & 0xffff;

	return PCIBIOS_SUCCESSFUL;
}



static int
PCI_readdword(struct pci_dev *dev,
	      int   index,
	      u32 *dword_read)
{
	u32 data = 0;
		
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	*dword_read = data; 

	return PCIBIOS_SUCCESSFUL;	
}



static int
PCI_writebyte(struct pci_dev *dev,
	      int index,
	      u8 byte_write)
{
	u32 data = 0;
       
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	data = (data & ~(0xff << ((index & 3) << 3))) |
	       (byte_write << ((index & 3) << 3));

	if (config_access(PCI_ACCESS_WRITE, dev, index, &data))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}



static int
PCI_writeword(struct pci_dev *dev,
	      int index,
	      u16 word_write)
{
	u32 data = 0;
       
	if (config_access(PCI_ACCESS_READ, dev, index, &data))
		return -1;

	data = (data & ~(0xffff << ((index & 3) << 3))) |
	       (word_write << ((index & 3) << 3));

	if (config_access(PCI_ACCESS_WRITE, dev, index, &data))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}



static int
PCI_writedword(struct pci_dev *dev,
	       int  index,
	       u32 dword_write)
{
	if (config_access(PCI_ACCESS_WRITE, dev, index, &dword_write))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}


void __init pcibios_fixup_irqs(void)
{
        struct pci_dev *dev;
        int slot_num = 0;
	int pci_func = 0; 
	u8 interrupt = 0;

	pci_for_each_dev(dev) {
		slot_num = PCI_SLOT(dev->devfn);
		pci_func = PCI_SLOT(dev->devfn);
		pci_read_config_byte(dev, PCI_INTERRUPT_LINE, &interrupt);

		if (dev->vendor == PCI_VENDOR_ID_ATI)
			switch (dev->device) {
			case PCI_DEVICE_ID_ATI_XILLEON_IDE:
				dev->irq = XILLEON_IDE_INT;
				pci_write_config_byte(dev, PCI_INTERRUPT_LINE, 
						      XILLEON_IDE_INT);
				break;
			case PCI_DEVICE_ID_ATI_XILLEON_USB:
				dev->irq = XILLEON_USB_INT;
				pci_write_config_byte(dev, PCI_INTERRUPT_LINE,
						      XILLEON_USB_INT);
				break;
			case PCI_DEVICE_ID_ATI_XILLEON_LPC:
				dev->irq = XILLEON_LPC1_INT;
				pci_write_config_byte(dev, PCI_INTERRUPT_LINE, 
						      XILLEON_LPC1_INT);
				break;
			case PCI_DEVICE_ID_ATI_XILLEON_DAIO:
				dev->irq = XILLEON_DAIO1_INT;
				pci_write_config_byte(dev, PCI_INTERRUPT_LINE,
						      XILLEON_DAIO1_INT);
				break;
			default:
				break;
			}

		else if(dev->irq == 0) {
/*
 * If not an ATI vendor code, assume this device is on the external PCI bus
 * and allocate an IRQ from the appropriate pool.  Note that there is no
 * guarantee that the IRQ we are handing out hasn't already been allocated
 * by the BIOS (or pmon), but it's *probably* either initializing all or
 * none.   Also note that non-ATI devices can only live on the external
 * bus, and ATI devices can only be builtin.
 */
            dev->irq = EXTPCI_IRQBASE + slot_num - 9;  /* first external pci slot is 9 */
            printk("pcibios_fixup_irqs: fixup slot %d vendor %x irq %d\n", slot_num, dev->vendor, dev->irq);
            pci_write_config_byte(dev, PCI_INTERRUPT_LINE, 
                                  dev->irq);

		}

	}
}


struct pci_ops xilleon_pci_ops = {
	PCI_readbyte,
	PCI_readword,
	PCI_readdword,
	PCI_writebyte,
	PCI_writeword,
	PCI_writedword
};


void __init pcibios_init(void)
{
	u32 val = 0;

	printk("PCI: Probing PCI hardware on host bus 0.\n");

	/* enable SB devices */
	/* added enable automatic incalidation of the PCI read prefetch cache, set the bit 5 */
 
	SETREG_REGMM32(PCIC_DEBUG_CNTL, 0x0000000F | 0x00000020);
	udelay(1000);
	
#if 1  /* should already be given to us by PMON */
	/* set pci1 aperture (memory) */
	SETREG_REGMM32(APER_CP_PCIC1_ADDR,
		       PCI_MEMORY_APER_BASEOUT | PCI_MEMORY_APER_BASE>>16);
	
	/* set pci2 aperture (shared i/o & config) */

	SETREG_REGMM32(APER_CP_PCIC2_ADDR, PCI_IO_APER_BASE>>16);

        /* change pci2 aperture to i/o */

	val = GETREG_REGMM32(APER_CP_PCIC2_CNTL);
	
	MODIFYFLD(val, APER_CP_PCIC2_CNTL,
		  PCI_ATYPE, PCI_ATYPE_IO);
	
	SETREG_REGMM32(APER_CP_PCIC2_CNTL, val);
#endif

#if 0
    /* enable burst reads and writes -- not verified yet */
    SETFLD_REGMM32(PCIC_BUSMASTER_CNTL, BUS_READ_COMBINE_EN, 1);
    SETFLD_REGMM32(PCIC_BUSMASTER_CNTL, BUS_WRT_COMBINE_EN, 1);
    SETFLD_REGMM32(PCIC_BUSMASTER_CNTL, BUS_MSTR_RD_MULT, 1);

    SETFLD_REGMM32(PCIC_BUSSLAVE_CNTL, BUS_READ_BURST, 1);
#endif
	pci_scan_bus(0, &xilleon_pci_ops, NULL);				
	pcibios_fixup_irqs();
	printk("PCI: Probing PCI hardware on host bus 0 done.\n");
}



int __init
pcibios_enable_device(struct pci_dev *dev)
{
	u16 cmd, old_cmd;

	pci_read_config_word(dev, PCI_COMMAND, &cmd);
	old_cmd = cmd;
	cmd |= PCI_COMMAND_IO;
	pci_write_config_word(dev, PCI_COMMAND, cmd);

	printk("PCI: enabling device %s (%04x -> %04x)\n",
	       dev->slot_name, old_cmd, cmd);

	return 0;
}


void __init
pcibios_align_resource(void *data, struct resource *res, unsigned long size)
{
}


char * __init
pcibios_setup(char *str)
{
	/* Nothing to do for now.  */

	return str;
}


struct pci_fixup pcibios_fixups[] = {
	{ 0 }
};


void __init
pcibios_update_resource(struct pci_dev *dev, struct resource *root,
                        struct resource *res, int resource)
{
	unsigned long where, size;
	u32 reg;

	where = PCI_BASE_ADDRESS_0 + (resource * 4);
	size = res->end - res->start;
	pci_read_config_dword(dev, where, &reg);
	reg = (reg & size) | (((u32)(res->start - root->start)) & ~size);
	pci_write_config_dword(dev, where, reg);
}




/*
 *  Called after each bus is probed, but before its children
 *  are examined.
 */
void __init pcibios_fixup_bus(struct pci_bus *b)
{
	pci_read_bridge_bases(b);
}

unsigned int pcibios_assign_all_busses(void)
{
	return 0;
}
