/*
 * Copyright (C) 2001 Metro Link, Inc., All rights reserved
 *              lembree@metrolink.com
 * based on
 * Copyright 2000 MontaVista Software Inc.
 * Author: MontaVista Software, Inc.
 *         	ppopov@mvista.com or source@mvista.com
 * based on
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 1999,2000 MIPS Technologies, Inc.  All rights reserved.
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * PROM library initialisation code.
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/string.h>
#include <linux/interrupt.h>

#include <asm/ati/xilleon.h>
#include <asm/ati/prom.h>

#include <asm/bootinfo.h>

long unsigned int chipId;

static int revid;
static int devid;

#define REVID_A11 0
#define REVID_A12 1


#define X220_HBIU_DEVICE_ID     0x4845
#define X225_HBIU_DEVICE_ID     0x4855
#define X226_HBIU_DEVICE_ID     0x4865
#define X216_HBIU_DEVICE_ID     0x4860


#define DEVID_X220 ((devid & HBIU_DEVICE_ID__DEVICE_ID__MASK)==X220_HBIU_DEVICE_ID)
#define DEVID_X225 ((devid & HBIU_DEVICE_ID__DEVICE_ID__MASK)==X225_HBIU_DEVICE_ID)
#define DEVID_X226 ((devid & HBIU_DEVICE_ID__DEVICE_ID__MASK)==X226_HBIU_DEVICE_ID)
#define DEVID_X216 ((devid & HBIU_DEVICE_ID__DEVICE_ID__MASK)==X216_HBIU_DEVICE_ID)

#define X225_A11 (DEVID_X225 && (revid==REVID_A11))

int prom_argc;
char **prom_argv, **prom_envp;
char arcs_cmdline[CL_SIZE];

#ifdef CONFIG_XILLEON_LOADADDR
#define LINUX_RAM_OFFS      (CONFIG_XILLEON_LOADADDR & 0x0fffffff)
#else
#define LINUX_RAM_OFFS      (0x00100000)
#endif

#define PROM_ENV_MAX_SIZE 512
#define PROM_ENV_MAX_VARS 64
#ifdef MAKE_PROM_ENV_COPY
static char prom_env_copy[PROM_ENV_MAX_SIZE];
static char *prom_env_list_copy[PROM_ENV_MAX_VARS];
#endif

static int prom_memsize = 0;
static int prom_heaptop = 0;

extern void (*board_time_init)(void);
extern void (*board_timer_setup)(struct irqaction *irq);
extern void xilleon_time_init(void);
extern void xilleon_timer_setup(struct irqaction *irq);
extern void xilleon_serial_console_setup(void);

const char *get_system_type(void)
{
	if (DEVID_X220)
		return "ATI Xilleon STW 220";
	else if (DEVID_X225)
		return "ATI Xilleon STW 225";
	else if (DEVID_X226)
		return "ATI Xilleon STW 226";
	else if (DEVID_X216)
		return "ATI Xilleon STW 216";
	else
		return "ATI Xilleon STW unknown rev";
}

static inline unsigned char str2hexnum(unsigned char c)
{
	if(c >= '0' && c <= '9')
	return c - '0';
	if(c >= 'a' && c <= 'f')
	return c - 'a' + 10;
	return 0; /* foo */
}

char * __init prom_getcmdline(void)
{
	return &(arcs_cmdline[0]);
}


void  __init prom_init_cmdline(void)
{
	char *cp;
	int actr;

	actr = 1; /* Always ignore argv[0] */

	cp = &(arcs_cmdline[0]);
	while(actr < prom_argc) {
	        strcpy(cp, prom_argv[actr]);
		cp += strlen(prom_argv[actr]);
		*cp++ = ' ';
		actr++;
	}

	*cp = '\0';
}


char *prom_getenv(char *envname)
{
	/*
	 * Return a pointer to the given environment variable.
	 * Environment variables are stored in the form of "memsize=64".
	 */

	char **env = prom_envp;
	int i;

	i = strlen(envname);

	while(*env) {
		if(strncmp(envname, *env, i) == 0) {
			return(*env + strlen(envname) + 1);
		}
		env++;
	}
	return(NULL);
}


int prom_getenv_int(char *envname)
{
	/*
	 * Return a pointer to the given environment variable.
	 * Environment variables are stored in the form of "memsize=64".
	 */

	char **env = prom_envp;
	int i;

	envname = "memsize";

	i = strlen(envname);

	while(*env) {
		if(strncmp(envname, *env, i) == 0) {
                    return (simple_strtol(*env + strlen(envname) + 1, NULL, 0));
		}
		env++;
	}
	return 0;
}



int __init page_is_ram(unsigned long pagenr)
{
    return 1;
}

void prom_free_prom_memory (void)
{
}


static unsigned long __init prom_init_memsize(void)
{
	char *memsize_str;
	unsigned int memsize;

	memsize_str = prom_getenv("memsize");
	if (!memsize_str) {
		memsize = 32;
		printk("memsize environment variable not set: assuming %dMB\n", memsize);
	} else {
		memsize = simple_strtol(memsize_str, NULL, 0);
        if (memsize < ((LINUX_RAM_OFFS>>20)+8)) {
            memsize = (LINUX_RAM_OFFS>>20) + 16;
            printk("invalid memsize environment variable. assuming %dMB\n", memsize);
        } else {
  		    printk("memsize: %dMB\n", memsize);
        }
	}
	return memsize;
}

void xilleon_detect_chip_id(void)
{

  /* assume it is X220 */
  chipId = 0x220;

  devid = GETFLD_REGMM16(HBIU_DEVICE_ID, DEVICE_ID);
  if (DEVID_X225) {
    revid = GETFLD_REGMM32(X225_SB_REVISION_ID, IDE_REV_ID);
    chipId = 0x225;
  }
  if (DEVID_X220)
    chipId = 0x220;
  
  if (DEVID_X226)
    chipId = 0x226;
  
  if (DEVID_X216)
    chipId = 0x216;

}

static void __init prom_init_heaptop(void)
{
    char *strg;

	prom_heaptop = 0x80000;
	strg = prom_getenv("heaptop");
	if (strg) 
	{
		char *endptr;
		prom_heaptop = simple_strtol(strg, &endptr, 16) & 0x0fffffff;
		if (endptr && *endptr != 0) {
			prom_heaptop = simple_strtol(strg, NULL, 0) & 0x0fffffff;
		}
	}
}

static inline void str2eaddr(unsigned char *ea, unsigned char *str)
{
	int i;

	for(i = 0; i < 6; i++) {
		unsigned char num;

		if((*str == '.') || (*str == ':'))
			str++;
		num = str2hexnum(*str++) << 4;
		num |= (str2hexnum(*str++));
		ea[i] = num;
	}
}

 
int get_ethernet_addr(char *ethernet_addr)
{
        char *ethaddr_str;

        ethaddr_str = prom_getenv("ethaddr");
	if (!ethaddr_str) {
	        printk("ethaddr not set in boot prom\n");
		return -1;
	}
	str2eaddr(ethernet_addr, ethaddr_str);

#ifdef DEBUG
	{
		int i=0;
		printk("get_ethernet_addr: ");
		for (i=0; i<5; i++)
			printk("%02x:", (unsigned char)*(ethernet_addr+i));
		printk("%02x\n", *(ethernet_addr+i));
	}
#endif

	return 0;
}


/* References to section boundaries */
extern char _end;

#define PFN_ALIGN(x)    (((unsigned long)(x) + (PAGE_SIZE - 1)) & PAGE_MASK)

void __init prom_meminit(void)
{
	u32 mem_size;

    prom_init_heaptop();
	mem_size = prom_init_memsize() << 20;
	prom_memsize = mem_size;

	add_memory_region(0, 0x80000, BOOT_MEM_ROM_DATA);
    add_memory_region(0x80000, LINUX_RAM_OFFS-0x80000, BOOT_MEM_RESERVED);
    add_memory_region(LINUX_RAM_OFFS, mem_size-LINUX_RAM_OFFS, BOOT_MEM_RAM);
}

#ifdef MAKE_PROM_ENV_COPY
static void prom_setup_env_copy(char **envp)
{
    char **env = envp;
    int i, j;

    i = 0;
    j = 0;

    while (*env) 
    {
	if ((i + strlen (*env) + 1) >= PROM_ENV_MAX_SIZE)
	    break;
        if ((j + 1) >= PROM_ENV_MAX_VARS)
	    break;

	strcpy (&prom_env_copy[i], *env);
	prom_env_list_copy[j] = &prom_env_copy[i];
	j++;
	i+= (strlen (*env) + 1);
	
	env++;
    }

    prom_env_list_copy[j] = NULL;
}
#endif

int prom_get_memsize(void)
{
    return (prom_memsize);
}

int prom_get_linuxstart(void)
{
    return (LINUX_RAM_OFFS);
}

int prom_get_heaptop(void)
{
    return (prom_heaptop);
}


int __init prom_init(int argc, char **argv, char **envp)
{
	int bus_speed;
	char *bus_mode;
	u32 straps; 

	prom_argc = argc;
	prom_argv = argv; /* Note : this memory goes away after this function returns !!! */

#ifdef MAKE_PROM_ENV_COPY
        prom_setup_env_copy (envp); 
	prom_envp = prom_env_list_copy; /* This is safe to use later but wastes memory */
#else
	prom_envp = envp; /* Note : this memory goes away after this function returns !!! */
#endif
	mips_machgroup = MACH_GROUP_ATI;
	mips_machtype = MACH_ATI_XILLEON;

	board_time_init = xilleon_time_init;
	board_timer_setup = xilleon_timer_setup;

	set_io_port_base(XILLEON_PCIC2BASE);
	ioport_resource.start = 0x00000000;
	ioport_resource.end = 0xffffffff;
	iomem_resource.start = 0x08000000;
	iomem_resource.end = 0xffffffff;

	prom_init_cmdline();
	xilleon_serial_console_setup();
	prom_printf("\nLINUX started...\n");

	xilleon_detect_chip_id();

	prom_printf("ATI Xilleon %x ", chipId);

	straps = GETREG_REGMM32(STRAPS_VALUE);
	if ((straps >> (chipId == 0x220 ? 0 : 1)) & 1)
		prom_printf("(big endian), ");
	else
		prom_printf("(little endian), ");

	switch ((straps >> (chipId == 0x220 ? 7 : 5)) & 3) {
	case 0:
		prom_printf("Solo");
		break;
	case 1:
		if (chipId == 0x220)
			prom_printf("Solo-");
		else
			prom_printf("<Reserved;1>");
		break;
	case 2:
		prom_printf("Peer");
		break;
	case 3:
		prom_printf("Peer+");
		break;
	default:
		prom_printf("unknown configuration");
	}

	if (chipId == 0x220) {
		bus_mode = "PCI";
		bus_speed = (straps & (1<<9) ? 66 : 33);
	} else {
		bus_mode = (straps & (1<<15) ? "AGP" : (straps & (1<<14) ? "MPX" : "PCI"));
		bus_speed = (straps & (3<<14) ? 66 : 33);
	}
	prom_printf(" mode\n");
	prom_printf("%s bus speed is %dMHz\n", bus_mode, bus_speed);
	prom_meminit();

	return 0;
}

/* Support to trace into the frame buffer */

typedef void (*trace_to_fb)(const char *msg);
typedef long long (*trace_get_timestamp64)(void);

static trace_to_fb trace_func_p=0; /* pointer to trace function */
static trace_get_timestamp64 trace_get_timestamp64_p=0; /* pointer to time-stamp function */ 
static char msg[1024];

void dbg_bsp_init_trace(unsigned int trace_func, unsigned int time_stamp64_func)
{
    trace_func_p=(trace_to_fb)trace_func;
    trace_get_timestamp64_p=(trace_get_timestamp64)time_stamp64_func;
}

void dbg_bsp_deinit_trace(void)
{
    trace_func_p=0;
    trace_get_timestamp64_p=0;
}

void dbg_trace_to_fb(const char *fmt, ...)
{
    va_list args;
   
    if (trace_func_p) {
        va_start(args, fmt);
        vsprintf(msg, fmt, args);
        va_end(args);

        trace_func_p(msg);
    }
}

long long dbg_trace_get_timestamp64(void)
{
    long long time_stamp = 0;

    if (trace_get_timestamp64_p)
        time_stamp = trace_get_timestamp64_p();

    return time_stamp;
}

EXPORT_SYMBOL(dbg_bsp_init_trace);
EXPORT_SYMBOL(dbg_bsp_deinit_trace);
EXPORT_SYMBOL(dbg_trace_get_timestamp64);
EXPORT_SYMBOL(dbg_trace_to_fb);
EXPORT_SYMBOL(prom_get_memsize);
EXPORT_SYMBOL(prom_get_linuxstart);
EXPORT_SYMBOL(prom_get_heaptop);
