/*
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 2000 MIPS Technologies, Inc.  All rights reserved.
 *
 * ########################################################################
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ########################################################################
 *
 * This is the interface to the remote debugger stub.
 *
 */

#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/serial.h>
#include <linux/serialP.h>
#include <linux/serial_reg.h>

#include <asm/serial.h>
#include <asm/io.h>

#include <asm/gdb-stub.h>
#include <asm/atomic.h>

#define GDB_BUF_SIZE 512 /* power of 2, please */

static char gdb_buf[GDB_BUF_SIZE];
static int gdb_buf_in_inx;
static atomic_t gdb_buf_in_cnt;
static int gdb_buf_out_inx;

int gdb_no_ints = 1; /* Default: no interrupts, please */
int gdb_enter = 1;	/* Default: do gdb hook on boot */
int gdb_ttyS = 1; /* Default: ttyS1 */

extern unsigned int xilleon_serial_in(struct async_struct *info, int offset);
extern void xilleon_serial_out(struct async_struct *info, int offset, int value);
extern void breakpoint(void);

static struct serial_state rs_table[RS_TABLE_SIZE] = {
	SERIAL_PORT_DFNS	/* Defined in serial.h */
};

static struct async_struct kdb_port_info = {0};

static __inline__ unsigned int serial_in(struct async_struct *info, int offset)
{
	return (*(volatile unsigned char *)(info->port + KSEG1 + offset) & 0xff);
}

static __inline__ void serial_out(struct async_struct *info, int offset,
				int value)
{
        *(volatile unsigned char *)(info->port + KSEG1 + offset) = 
		(unsigned char)value;
        __asm__("sync\n");
}

static void gdb_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
    int chr;
    int iir;

    do {
	    if (xilleon_serial_in(&kdb_port_info, UART_LSR) & UART_LSR_DR)
		    chr = xilleon_serial_in(&kdb_port_info, UART_RX);
        else 
            chr = -1;

	    iir = xilleon_serial_in(&kdb_port_info, UART_IIR);

	    if (chr < 0) continue;

        if (chr == 3) { /* Ctrl-C means remote interrupt */
            breakpoint();
            continue;
        }

		if (atomic_read(&gdb_buf_in_cnt) >= GDB_BUF_SIZE) {	/* buffer overflow, clear it */
		    gdb_buf_in_inx = 0;
		    atomic_set(&gdb_buf_in_cnt, 0);
		    gdb_buf_out_inx = 0;
		    break ;
		}
    
		gdb_buf[gdb_buf_in_inx++] = chr;
		gdb_buf_in_inx &= (GDB_BUF_SIZE - 1);
		atomic_inc(&gdb_buf_in_cnt);
    } while (iir & UART_IIR_RDI);
}

void rs_kgdb_hook(int tty_no) {
	int t;
	struct serial_state *ser = &rs_table[tty_no];

	kdb_port_info.state = ser;
	kdb_port_info.magic = SERIAL_MAGIC;
	kdb_port_info.port = ser->port;
	kdb_port_info.flags = ser->flags;

	/*
	 * Clear all interrupts
	 */
	xilleon_serial_in(&kdb_port_info, UART_LSR);
	xilleon_serial_in(&kdb_port_info, UART_RX);
	xilleon_serial_in(&kdb_port_info, UART_IIR);
	xilleon_serial_in(&kdb_port_info, UART_MSR);

	/*
	 * Now, initialize the UART 
	 */
	xilleon_serial_out(&kdb_port_info, UART_LCR, UART_LCR_WLEN8);	/* reset DLAB */
	if (kdb_port_info.flags & ASYNC_FOURPORT) {
		kdb_port_info.MCR = UART_MCR_DTR | UART_MCR_RTS;
		t = UART_MCR_DTR | UART_MCR_OUT1;
	} else {
		kdb_port_info.MCR 
			= UART_MCR_DTR | UART_MCR_RTS | UART_MCR_OUT2;
		t = UART_MCR_DTR | UART_MCR_RTS;
	}

	kdb_port_info.MCR = t;		/* no interrupts, please */
	xilleon_serial_out(&kdb_port_info, UART_MCR, kdb_port_info.MCR);
	
	/* baud rate is fixed to 115200 */
	t = kdb_port_info.state->baud_base / 115200;
	/* set DLAB */
	xilleon_serial_out(&kdb_port_info, UART_LCR, UART_LCR_WLEN8 | UART_LCR_DLAB);

	xilleon_serial_out(&kdb_port_info, UART_DLL, t & 0xff);/* LS of divisor */
	xilleon_serial_out(&kdb_port_info, UART_DLM, t >> 8);  /* MS of divisor */
	/* reset DLAB */
	xilleon_serial_out(&kdb_port_info, UART_LCR, UART_LCR_WLEN8);

    gdb_ttyS = tty_no;
}

int putDebugChar(char c)
{

	if (!kdb_port_info.state) { 	/* need to init device first */
		return 0;
	}

	while ((xilleon_serial_in(&kdb_port_info, UART_LSR) & UART_LSR_THRE) == 0)
		;

	xilleon_serial_out(&kdb_port_info, UART_TX, c);

	return 1;
}

char getDebugChar(void)
{
    int chr;

	if (!kdb_port_info.state) { 	/* need to init device first */
		return 0;
	}

    if (gdb_no_ints == 0 && atomic_read(&gdb_buf_in_cnt) != 0) { /* intr routine has q'd chars */
	    chr = gdb_buf[gdb_buf_out_inx++];
	    gdb_buf_out_inx &= (GDB_BUF_SIZE - 1);
	    atomic_dec(&gdb_buf_in_cnt);
	    return(chr);
    }

	while (!(xilleon_serial_in(&kdb_port_info, UART_LSR) & 1))
		;

	return(xilleon_serial_in(&kdb_port_info, UART_RX));
}

int gdb_hook_int(void)
{
	struct serial_state *ser = &rs_table[gdb_ttyS];
    int retval;

    if (gdb_enter == 0) { /* need to connect to gdb hook first */
		return 0;
	}

	if (!kdb_port_info.state) { /* need to init device first */
		return 0;
	}

    if (gdb_no_ints == 0) {
        gdb_no_ints = 1;

        xilleon_serial_out(&kdb_port_info, UART_IER, 0); /* turn off interrupts*/

        retval = request_irq(ser->irq,
                         gdb_interrupt,
                         SA_INTERRUPT,
                         "GDB-stub", NULL);
        if (retval) {
	        printk("rs_kgdb_hook: request_irq(irq=%d) failed: %d\n", ser->irq, retval);
            return -1;
        }

        xilleon_serial_out(&kdb_port_info, UART_IER, UART_IER_RDI); /* turn on interrupts*/
    }

    return 0;
}

static int __init kgdb_opt_no_ints(char *str)
{
    gdb_no_ints = simple_strtoul(str, NULL, 10);
	return 1;
}

static int __init kgdb_opt_enter(char *str)
{
    gdb_enter = simple_strtoul(str, NULL, 10);
	return 1;
}

__setup("gdb_no_ints=", kgdb_opt_no_ints);
__setup("gdb_enter=", kgdb_opt_enter);
