/*
 * BRIEF MODULE DESCRIPTION
 *	NEC Eagle Board specific PCI fixups.
 *
 * Copyright 2001 MontaVista Software Inc.
 * Author: Yoichi Yuasa
 *		yyuasa@mvista.com or source@mvista.com
 *
 * Author: Jun Sun, jsun@mvista.com or jsun@junsun.net
 * 		Modified for DDB4131 board
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/pci.h>

#include <asm/debug.h>
#include <asm/pci_channel.h>
#include <asm/vr4122/vr4122.h>
#include <asm/vr4122/ddb4131.h>

static struct resource vr4122_pci_io_resource = {
	"PCI I/O space",
	VR4122_PCI_IO_START,
	VR4122_PCI_IO_END,
	IORESOURCE_IO
};

static struct resource vr4122_pci_mem_resource = {
	"PCI memory space",
	VR4122_PCI_MEM_START,
	VR4122_PCI_MEM_END,
	IORESOURCE_MEM
};

extern struct pci_ops vr4122_pci_ops;

struct pci_channel mips_pci_channels[] = {
	{&vr4122_pci_ops, &vr4122_pci_io_resource, &vr4122_pci_mem_resource, 0, 256},
	{NULL, NULL, NULL, 0, 0}
};

void __init pcibios_fixup_resources(struct pci_dev *dev)
{
}

void __init pcibios_fixup(void)
{
	struct pci_dev *dev;

#define M1535_CONFIG_PORT 0x3f0
#define M1535_INDEX_PORT  0x3f0
#define M1535_DATA_PORT   0x3f1

	printk("Configuring ALI M1535 Super I/O mouse irq.\n");

	db_verify_warn(request_region(M1535_CONFIG_PORT, 2, "M1535 Super I/O config"), != NULL);

	/* Enter config mode. */
	outb(0x51, M1535_CONFIG_PORT);
	outb(0x23, M1535_CONFIG_PORT);

	/* Select device 0x07. */
	outb(0x07, M1535_INDEX_PORT);
	outb(0x07, M1535_DATA_PORT);

	/* Set mouse irq (register 0x72) to 12. */
	outb(0x72, M1535_INDEX_PORT);
	outb(0x0c, M1535_DATA_PORT);

	/* Exit config mode. */
	outb(0xbb, M1535_CONFIG_PORT);

	pci_for_each_dev(dev) {

		if((dev->vendor == PCI_VENDOR_ID_AL) &&
                   (dev->device == PCI_DEVICE_ID_AL_M1533) ) {
			u8 old;
			printk("Enabling ALI M1533 PS2 keyboard/mouse.\n");
			pci_read_config_byte(dev, 0x41, &old);
			pci_write_config_byte(dev, 0x41, old | 0xd0);
		}
	}
}

/*
 * We are fixing up IRQs based on the bus number and slot number.
 * We setup pci_ops such as slot 0 corresponds to AD:11 bit selected 
 * (per PCI standard)
 *
 * We are cheating here - all the devices that we care to assign IRQs
 * happen to have different slot number.  We thus just use a single
 * slot-indexed arrary to assigne IRQs.
 *
 * One exception on solution gear baseboard.  Slot 7 (AD18) is PCI-ISA
 * bridge on bus 0 and PCI slot 5 on bus 1.  Fortunately PCI-ISA bridge
 * does not care about IRQ number.  So we still use a single array
 * and fill in IRQ number for PCI5 at slot 7.
 */
#define		MAX_NUM_BUS		2
#define         MAX_SLOT_NUM            21
static const int pci_irq_map[MAX_SLOT_NUM] ={
	/* AD:11 */ 0xff,
	/* AD:12 */ IRQ_PCI_INTB,		/* AMD ether, on bus 1 */
	/* AD:13 */ 0xff,
	/* AD:14 */ 0xff,
	/* AD:15 */ 14,				/* hardcoded irq for IDE */
	/* AD:16 */ IRQ_PCI_INTD,		/* PCI INT-D, pci slot 3 */
	/* AD:17 */ IRQ_PCI_INTA,		/* PCI INT-A, pci slot 4 */
	/* AD:18 */ IRQ_PCI_INTD,		/* PCI-ISA bridge; PCI5 on bus1 */
	/* AD:19 */ IRQ_PCI_INTB,		/* PCI INT-B, pci slot 1 */
	/* AD:20 */ IRQ_PCI_INTA,		/* PCI INT-A, pci slot 2 */
	/* AD:21 */ 0xff,
	/* AD:22 */ 0xff,
	/* AD:23 */ 0xff,
	/* AD:24 */ 0xff,
	/* AD:25 */ 0xff,
	/* AD:26 */ 0xff,
	/* AD:27 */ 0xff,
	/* AD:28 */ 0x0,			/* southbridge PMU */
	/* AD:29 */ 0xff,
	/* AD:30 */ 0xff,
	/* AD:31 */ 9,				/* hardcoded, southbridge USB */
};
void __init pcibios_fixup_irqs(void)
{
	struct pci_dev *dev;
	unsigned int devnum;

	pci_for_each_dev(dev) {
		dev->irq = pci_irq_map[PCI_SLOT(dev->devfn)];
		db_warn(dev->irq != 0xff);
		pci_write_config_byte(dev, PCI_INTERRUPT_LINE, dev->irq);
	}
}

unsigned int pcibios_assign_all_busses(void)
{
	return 0;
}

